/*
 * Copyright (c) 2002 SAP AG - All Rights Reserved.
 *
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_progmodel/java/com/sap/tc/webdynpro/progmodel/api/IWDNodeInfo.java#3 $
 */
package com.sap.tc.webdynpro.progmodel.api;

import java.util.Iterator;

import com.sap.dictionary.runtime.IDataType;
import com.sap.dictionary.runtime.IStructure;
import com.sap.tc.cmi.metadata.ICMIModelClassInfo;

/**
 * Represents the static structural aspect of a Node and its NodeElements
 * at runtime keeping the meta data as defined at design time. Each Node has
 * a reference to a NodeInfo holding its metadata.
 * <p>
 * The relevant meta data accessible through IWDNodeInfo are:
 * <ul>
 *  <li>{@link #getName() name}
 *  <li>{@link #isSingleton() singleton}
 *  <li>{@link #isMultiple() multiple}
 *  <li>{@link #isMandatory() mandatory}
 *  <li>{@link #isMultipleSelection() multipleSelection}
 *  <li>{@link #isMandatorySelection() mandatorySelection}
 *  <li>{@link #getDataType() dataType}
 * </ul>
 * <p>
 * Node infos also provide a mean to dynamically add child nodes
 * (there are different methods for unmapped, mapped and recursive nodes)
 * or attributes (unmapped or mapped, several methods to add attributes
 * automatically from external meta data). You cannot remove such nodes or
 * attributes afterwards, you can only reset the whole context to its initial
 * state via {@link IWDContext#reset(boolean)}.
 *
 * @author SAP
 * @SAPWebDynproPart 2
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_progmodel/java/com/sap/tc/webdynpro/progmodel/api/IWDNodeInfo.java#3 $
 */
public interface IWDNodeInfo
{
  /**
   * Returns the name as described in the meta data.
   * @return the name as described in the meta data
   */
  String getName();

  /**
   * Returns the IWDContext this node belongs to.
   * @return the IWDContext this node belongs to.
   */
  IWDContext getContext();
  
  /**
   * Returns whether the described Node is singleton relatively to its parent.
   * A singleton Node exists exactly once per parent node and always keeps data
   * relatively to the parent's lead selection. In contrast, a non-singleton
   * node exists once for each element of the parent node and keeps data
   * relatively to that element.
   * <p>
   * The root Node and independent Nodes (direct children of the root node) are
   * always singletons.
   * <p>
   * <b>Note:</b> A singleton node always exists once <i>per parent</i> and keeps
   * data relative to the parent's lead selection element. This is inconsistent
   * if the parent node (or its lead selection) is empty. In this case, the node
   * is empty whatever the cardinality tells and throws an exception if you try
   * to add or bind elements.
   * 
   * @return <code>true</code> if the described Node is singleton relatively
   *  to its parent or <code>false</code> otherwise.
   */
  boolean isSingleton();

  /**
   * Returns whether the described Node is guaranteed to have at least one
   * element. According to this property the Node may return <code>null</code>
   * or throw a <code>ContextException</code> when asked for an element and
   * having none.
   * <p>
   * This property together with {@link #isMultiple() isMultiple} describes the
   * element cardinality of the Node.
   * <p>
   * <b>Note:</b> A node may yet be empty, even if this property declares it 
   * mandatory. This can happen if the node is a singleton and does currently
   * not have a valid parent element.
   * 
   * @return <code>true</code> if the described Node is guaranteed to have at
   *  least one element or <code>false</code> otherwise.
   */
  boolean isMandatory();

  /**
   * Returns whether the described Node may contain more than one element.
   * <p>
   * This property together with {@link #isMandatory() isMandatory} describes the
   * element cardinality of the Node.
   *
   * @return <code>true</code> if the described Node may contain more than one element
   * or <code>false</code> otherwise.
   */
  boolean isMultiple();

  /**
   * Returns whether the selection of the described Node is guaranteed to have
   * at least one element if the node is not empty.
   * <p>
   * This property together with {@link #isMultipleSelection() } describes the
   * selection cardinality of the Node.
   * <p>
   * <b>Note:</b> A node's selection may yet be empty, even if this property
   * declares it mandatory. This can happen if the node is empty.
   * 
   * @return <code>true</code> if the selection of the described Node is guaranteed to have
   * at least one element or <code>false</code> otherwise.
   */
  boolean isMandatorySelection();

  /**
   * Returns whether the selection of the described Node may contain more than
   * one element.
   * <p>
   * This property together with {@link #isMandatorySelection() } describes the
   * selection cardinality of the Node.
   *
   * @return <code>true</code> if the selection of the described Node may contain more than
   * one element or <code>false</code> otherwise.
   */
  boolean isMultipleSelection();

  /**
   * Returns the data type assigned to the <b>Node</b>. Since this type always
   * is a structure, it might be better to use {@link #getStructureType()}
   * directly.
   * @return the data type assigned to the <b>Node</b> or <code>null</code>
   */
  IDataType getDataType();

  /**
   * Returns the structured data type assigned to the <b>Node</b>. This may be
   * <code>null</code>. Then all attributes of the node must have a data type of
   * their own.
   * @return the structured data type assigned to the <b>Node</b> or <code>null</code>
   */
  IStructure getStructureType();

  /**
   * Returns the parent NodeInfo. Returns <code>null</code> for the root node.
   * @return the parent NodeInfo
   */
  IWDNodeInfo getParent();

  /**
   * Returns the NodeInfo for the named child node.
   * @param id the ID of the child node
   * @return the NodeInfo for the named child node or <code>null</code> if no
   *          child node of that name exists
   */
  IWDNodeInfo getChild(String id);

  /**
   * Returns the AttributeInfo for the named attribute.
   * @param name the name of the attribute
   * @return the AttributeInfo for the named attribute or <code>null</code> if
   *          no attribute of that name exists
   */
  IWDAttributeInfo getAttribute(String name);

  /** 
   * Returns an Iterator to iterate over the attributes' IWDAttributeInfo.
   * @return an Iterator for the IWDAttributeInfo
   */
  Iterator iterateAttributes();

  /**
   * Returns an Iterator to iterate over the child nodes' IWDNodeInfo.
   * @return an Iterator for the IWDNodeInfo
   */
  Iterator iterateChildren();

  /**
   *  Returns the path to the NodeInfo in the form ContextName[.NodeName].NodeName.
   * @return the path.
   */
  String getPathDescription();

  /**
   * Dynamically declares and adds an unmapped child NodeInfo with the given 
   * properties. The name must not be defined in this NodeInfo, neither for a node 
   * nor an attribute. Otherwise a <code>ContextException</code> will be thrown.
   * <p>
   * After a successful call to this method, an instance (or instances, in the
   * case of non-singleton nodes) of the newly declared node can be retrieved
   * with a call to {@link IWDNode#getChildNode(String,int)}.
   * <p>
   * If <code>dataType</code> is not <code>null</code>, a structure of that name
   * is searched in the dictionary and the returned NodeInfo automatically has
   * attributes for all simple fields of that structure. Otherwise it does
   * not contain any attribute yet and you must manually add attributes via 
   * {@link #addAttribute(String, String)}.
   * <p>
   * If you created a model node with element class <code>ICMIModelClass</code>
   * meaning a special class described by a <code>ICMIModelClassInfo</code>,
   * you can dynamically specify this class via
   * @link #setModelClassInfo(ICMIModelClassInfo)}. Afterwards you can add
   * the attributes automatically via {@link #addAttributesFromModelClassInfo()}.
   * 
   * @param name the name of the child NodeInfo. Must be unique all over the context.
   *    The name may only contain ascii letters, digits and underscore ('_'). It
   *    must start with a letter.
   * @param elementClass The element class for a model node, <code>null</code>
   *          for a value node.
   * @param singleton indicates, whether the described Node is singleton 
   *        relatively to its parent
   * @param mandatory indicates, whether the described Node is guaranteed to 
   *        have at least one element
   * @param multiple indicates, whether the described Node may contain more 
   *        than one element
   * @param mandatorySelection indicates, whether the selection of the 
   *        described Node is guaranteed to have at least one element
   * @param multipleSelection indicates, whether the selection of the described
   *        Node may contain more than one element.
   * @param initializeLeadSelection <code>true</code> if this node's selection
   *        shall be initialized when the collection is initialized
   * @param dataType the name of a data type, a data type (IStructure) or 
   *        <code>null</code>.
   * @param supplier A supply "function", see {@link IWDNodeCollectionSupplier}.
   *        May be <code>null</code>, in that case the elements remain empty and
   *        must be set manually during an event handler.
   * @param disposer A dispose "function", see {@link IWDNodeCollectionDisposer}.
   *        May be <code>null</code> if no application code is needed before
   *        deletion of the node.
   * @return the added NodeInfo
   * @throws ContextConfigurationException when some of these properties are
   *          inconsistent (e. g. independent node is not singleton)
   */
  IWDNodeInfo addChild(String name,
      Class elementClass,
      boolean singleton,
      boolean mandatory, boolean multiple,
      boolean mandatorySelection, boolean multipleSelection,
      boolean initializeLeadSelection,
      Object dataType,
      IWDNodeCollectionSupplier supplier,
      IWDNodeCollectionDisposer disposer);

  /**
   * Dynamically declares and adds an unmapped child NodeInfo with the given 
   * properties. Only existing for backward compatibility.
   * @see #addChild(String, Class, boolean, boolean, boolean, boolean, boolean, boolean, Object, IWDNodeCollectionSupplier, IWDNodeCollectionDisposer) 
   */
  IWDNodeInfo addChild(String name,
      Class elementClass,
      boolean singleton,
      boolean mandatory, boolean multiple,
      boolean mandatorySelection, boolean multipleSelection,
      boolean initializeLeadSelection,
      String dataType,
      IWDNodeCollectionSupplier supplier,
      IWDNodeCollectionDisposer disposer);
  
  /**
   * Dynamically declares and adds a mapped child NodeInfo with the given 
   * properties. The ID must not be defined in this NodeInfo, neither for a node 
   * nor an attribute. Otherwise an IllegalArgumentException will be thrown.
   * <p>
   * After a successful call to this method, an instance (or instances, in the
   * case of non-singleton nodes) of the newly declared node can be retrieved
   * with a call to {@link IWDNode#getChildNode(String,int)}.
   * <p>
   * The properties <code>mandatory</code>, <code>multiple</code> and
   * <code>dataType</code> cannot be defined, but are taken from the mapped 
   * node. The properties concerning the selection are only considered if the 
   * selection is not mapped.
   * <p>
   * <code>mappedContext</code> and <code>mappedPath</code> need not be defined
   * here (i. e. may be <code>null</code>), you can set them later using
   * {@link #setMapping(IWDContext, String, boolean)}.
   * <p>
   * The returned NodeInfo does not contain any attributes yet. You may add all
   * attributes from the origin node via {@link #addAttributesFromDataNode()}.
   * 
   * @param name the name of the child NodeInfo. Must be unique all over the context.
   *    The name may only contain ascii letters, digits and underscore ('_'). It
   *    must start with a letter.
   * @param elementClass The element class for a model node, <code>null</code>
   *        for a value node.
   * @param singleton indicates, whether the described Node is singleton
            relatively to its parent
   * @param mandatorySelection indicates, whether the selection of the 
   *        described Node is guaranteed to have at least one element
   * @param multipleSelection indicates, whether the selection of the described
   *        Node may contain more than one element.
   * @param mappedPath The path of the mapped node or
   *        <code>null</code> if you want to set the origin node later.
   * @param selectionMapped indicates whether the selection is mapped
   * @param initializeLeadSelection indicates whether the lead selection is to
   *        be set to 0 if the collection is filled
   * @return the added NodeInfo
   * @throws ContextConfigurationException when some of these properties are
   *        inconsistent (e. g. root node is not singleton)
   */
  IWDNodeInfo addMappedChild(String name,
      Class elementClass,
      boolean singleton,
      boolean mandatorySelection, boolean multipleSelection,
      String mappedPath, 
      boolean selectionMapped, boolean initializeLeadSelection);

  /**
   * Sets the mapping of a mapped node that has been declared before. This is
   * only allowed if no mapping had been given at declaration. (It has been
   * created using {@link #addMappedChild(String, Class, boolean, boolean, boolean, String, boolean, boolean)}
   * with <code>mappedPath</code> set to <code>null</code>). In constrast to the
   * immediate declaration, you can give the node instead of its name.
   * <p>
   * @param mappedNode The NodeInfo of the mapped node.
   * @param selectionMapped indicates whether the selection is mapped
   * @throws ContextConfigurationException when some of these properties are
   *          inconsistent or if the NodeInfo already has a valid mapping
   */
  void setMapping(IWDNodeInfo mappedNode, boolean selectionMapped);

  /**
   * Sets the mapping of a mapped node that has been declared before. This is
   * only allowed if no mapping had been given at declaration. (It has been
   * created using {@link #addMappedChild(String, Class, boolean, boolean, boolean, String, boolean, boolean)}
   * with <code>mappedPath</code> set to <code>null</code>). In constrast to the
   * immediate declaration, you can give the context instead of its name.
   * <p>
   * @param mappedContext The context containing the mapped node.
   * @param mappedPath The path of the mapped node.
   * @param selectionMapped indicates whether the selection is mapped
   * @throws ContextConfigurationException when some of these properties are
   *          inconsistent or if the NodeInfo already has a valid mapping
   */
  void setMapping(IWDContext mappedContext, String mappedPath, boolean selectionMapped);
  
  /**
   * Dynamically adds a recursive child NodeInfo having the same name as the
   * referenced node. This is not supported in the metamodel and only possible
   * dynamically.
   * @param parent The parent node info where the recursion shall start. May be
   *        <code>this</code>.
   * @return the added NodeInfo
   */
  IWDNodeInfo addRecursiveChild(IWDNodeInfo parent);

  /**
   * Dynamically adds a recursive child NodeInfo.
   * @param name The name of this child.
   *    The name may only contain ascii letters, digits and underscore ('_'). It
   *    must start with a letter.
   * @param parent The parent node info where the recursion shall start. May be
   *        <code>this</code>.
   * @return the added NodeInfo
   */
  IWDNodeInfo addRecursiveChild(String name, IWDNodeInfo parent);

  /**
   * Adds a new unmapped attribute to the structure. At a value node, it is a
   * value attribute. At a model node, it is a model attribute.
   * @param name The name of the attribute within the node, must be unique.
   *    The name may only contain ascii letters, digits and underscore ('_'). It
   *    must start with a letter.
   * @param dataType The name of the attribute's data type. It can be either a
   *    Java class (prefixed with "java:"), a type in the local Java DDIC
   *    (prefixed with "ddic:") or a type from a local dictionary (in that case
   *    the name is built
   *    "extern:<i>name.of.logical.dictionary</i>:<i>name.of.type</i>"). 
   *    If you choose a Java class, no UI element can be bound to this 
   *    attribute! A DDIC type may be either one that you defined within your 
   *    application or one of the predefined types. Note, that the predefined
   *    types are in the package "com.sap.dictionary" at run time.<br>
   *    If the type is Model based, consider retrieving the type from the Model
   *    metadata and using {@link #addAttribute(String, IDataType)}, especially
   *    if it is a CMI model.
   *    <b>Examples:</b>
   *    <ul>
   *      <li><code>java:java.util.Map</code>
   *      <li><code>ddic:com.sap.dictionary.string</code>
   *      <li><code>extern:com.sap.test.types.Flight:com.sap.test.types.BAPIRET2</code>
   *    </ul>
   * @return The added attribute.
   */
  IWDAttributeInfo addAttribute(String name, String dataType);
  
  /**
   * Adds a new unmapped value attribute to the structure.
   * @param name The name of the attribute within the node, must be unique.
   *    The name may only contain ascii letters, digits and underscore ('_'). It
   *    must start with a letter.
   * @param dataType The name of the attribute's data type. See
   *    {@link #addAttribute(String, String)} for details.
   * @return The added attribute.
   */
  IWDAttributeInfo addValueAttribute(String name, String dataType);
  
  /**
   * Adds a new unmapped value attribute to the structure.
   * @param name The name of the attribute within the node, must be unique.
   *    The name may only contain ascii letters, digits and underscore ('_'). It
   *    must start with a letter.
   * @param dataType The already known data type, e.g. from a Model
   * @return The added attribute.
   */
  IWDAttributeInfo addAttribute(String name, IDataType dataType);
  
  /**
   * Adds a new mapped attribute to the structure.
   * <p>
   * Note, that there are three different ways of attribute mapping. One is
   * working together with the node this attribute belongs to. In this case the
   * mapped node has several elements pointing to elements of the origin node
   * and the attribute is actually taken of this origin element.
   * <p>
   * The second one is mapped independently of the node. Then the mapping path
   * is evaluated on each attribute access. The origin attribute is always
   * accessed at the lead selection path. This is most probably only 
   * interesting for the context node, because even if a node contains several
   * elements, the attribute value of each element is identical because they all
   * map to the lead selection path and thus to the identical attribute. If the
   * mapping path cannot be followed to its end because one of the nodes does
   * not contain any element, a read access returns <code>null</code> (or
   * <code>0</code> or <code>false</code>, depending on the attribute type) and
   * a write access leads to a <code>ContextException</code>.
   * <p>
   * You distinguish between those two mappings via the <code>mappedName</code>.
   * Node based mapping is achieved by simply giving the name of the attribute
   * in the origin node without any path information. Pure attribute mapping
   * is achieved by giving a complete path starting with the component name.
   * <p>
   * IWDNodeInfo chooses the second case if <code>mappedPath</code> contains a
   * dot and the first case otherwise.
   * <p>
   * The third variant is a mixture of both. Technically it is the same as the
   * second one. But as soon as the node containing this attribute is also
   * mapped and the mapping path of the attribute is in the hierarchy below the
   * mapping path of the node, the mapping at runtime starts at the element and
   * only descends the rest of the path via the lead selection. See the example
   * below.
   * <p>
   * Examples:
   * <ul>
   *  <li><code>mappedNode.addMappedAttribute("mapped1", "value1")</code><br>
   *   This assumes that <code>mappedNode</code> is a mapped node with an origin
   *   node that has an attribute named "value1". It creates a attribute named
   *   "mapped1" taking its value of the attribute "value1" of the origin
   *   element.
   *  <li><code>node.addMappedAttribute("mapped2", ".CustomContext.Some.Node.value2"></code><br>
   *   This adds a mapped attribute named "mapped2" to <code>node</code>. The
   *   attribute maps to another attribute in the context named "CustomContext"
   *   of the current component. Its path within the component is "Some.Node.value2".
   *   Note that <code>mappedName</code> starts with a dot. This is because the
   *   first entry in the path is the name of an embedded component. Here the
   *   name is empty meaning the context is in the same component.
   *  <li><code>IWDNodeInfo node = parent.createMappedNode("MappedNode" ... ".Context.Another.Node",...)<br>
   *   node.addMappedAttribute("mapped3", ".Context.Another.Node.Child.value3")</code><br>
   *   In this case the node is mapped and the attribute is mapped, but not to 
   *   an attribute of the node itself. The mapping target of the attribute
   *   actually is in a child node of the node's mapping target.
   *   <p>
   *   Now, if you have an element of <code>MappedNode</code> and read its
   *   attribute <code>mapped3</code>, the contexts starts at the data node of
   *   <code>MappedNode</code> and descends to the child <code>Child</code>,
   *   takes its element at the lead selection and reads the attribute
   *   <code>value3</code>.
   * </ul>
   * @param name The name of the attribute within the node, must be unique.
   *    The name may only contain ascii letters, digits and underscore ('_'). It
   *    must start with a letter.
   * @param mappedName The name of the attribute in the origin node
   * @return The added attribute.
   */
  IWDAttributeInfo addMappedAttribute(String name, String mappedName);

  /**
   * Returns the meta data for an ICMIGenericModelClass or <code>null</code>
   * otherwise.
   * @return The model class info
   * @see #setModelClassInfo(ICMIModelClassInfo)
   */
  public ICMIModelClassInfo getModelClassInfo();

  /**
   * Sets the meta data for an ICMIGenericModelClass.
   * @param info The model class info
   * @see #getModelClassInfo()
   */
  public void setModelClassInfo(ICMIModelClassInfo info);

  /**
   * Automatically adds all attributes from the model metadata. This only works
   * if the elementClass is ICMIGenericModelClass and meta data have been
   * {@link #setModelClassInfo(ICMIModelClassInfo) set}.
   * @see com.sap.tc.cmi.model.ICMIGenericModelClass
   */
  public void addAttributesFromModelClassInfo();

  /**
   * Automatically adds all attributes from the data node. This method can only
   * be called on a mapped node.
   */
  public void addAttributesFromDataNode();

  /**
   * Sets the supplyingRelationRole for an ICMIGenericModelClass. The parent must
   * be an ICMIGenericModelClass, too, and the role is the one leading from the 
   * parent node's class to this one.
   * @param supplyingRelationRole The name of the supplyingRelationRole to set. 
   * @see com.sap.tc.cmi.model.ICMIGenericModelClass
   */
  public void setSupplyingRelationRole(String supplyingRelationRole);

  /**
   * Sets a new IWDNodeCollectionSupplier. You can reset it by setting 
   * <code>null</code>.
   * @param supplier The new collection supplier that will be used by every
   *    node based on this IWDNodeInfo to determine its collection.
   */
  void setCollectionSupplier(IWDNodeCollectionSupplier supplier);

  /**
   * Sets a new IWDNodeCollectionDisposer. You can reset it by setting 
   * <code>null</code>.
   * @param disposer The new collection disposer.
   */
  void setCollectionDisposer(IWDNodeCollectionDisposer disposer);

  /**
   * Adds a {@link IWDNodeNotificationListener}.
   * @param listener the notification listener
   * @see #removeNotificationListener(IWDNodeNotificationListener)
   */
  void addNotificationListener(IWDNodeNotificationListener listener);

  /**
   * Removes a {@link IWDNodeNotificationListener}.
   * @param listener the notification listener
   * @see #addNotificationListener(IWDNodeNotificationListener)
   */
  void removeNotificationListener(IWDNodeNotificationListener listener);
}
