/*
 * Copyright (c) 2002 SAP AG - All Rights Reserved.
 *
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_progmodel/java/com/sap/tc/webdynpro/progmodel/api/IWDNode.java#4 $
 */
package com.sap.tc.webdynpro.progmodel.api;

import com.sap.tc.cmi.util.ICMIObservableList; // just to make javadoc happy

import java.util.Collection;
import java.util.Comparator;

/**
 * Application Interface implemented by context nodes. While this class is used 
 * when programming dynamically, more convenient, typed subinterfaces of this
 * interface are generated for context nodes whose structure is declared within 
 * the Web Dynpro Development Workbench.
 * <p>
 * Each node holds a list of elements. If the list is not empty, one of the
 * elements is the "current" element, the "lead selection". The lead selection
 * can be changed by navigation (relative or absolute). Usually this list is
 * held exclusively by the node and can only be manipulated via the node. You
 * can share an <code>{@link com.sap.tc.cmi.util.ICMIObservableList}</code>
 * with the node by {@link #bind(Collection) binding} such a list to it. See
 * {@link #bind(Collection)} for details.
 * <p>
 * The node can be filled by {@link #bind(Collection) binding} elements to it.
 * You can {@link #addElement(IWDNodeElement) add}, {@link #removeElement(IWDNodeElement) remove},
 * {@link #moveElement(int, int) move} or {@link #swapElements(int, int) swap}
 * elements. Its content can be {@link #invalidate() invalidated}. Then, if it 
 * has a supply function (either generated or via 
 * {@link IWDNodeCollectionSupplier}), it will call this function upon further 
 * access to refill itself.
 * <p>
 * A node may have different cardinalities. At design time they are 0..1, 1..1,
 * 0..n and 1..n. At run time there are the flags <i>mandatory</i> (minimum 
 * cardinality is 1) and <i>multiple</i> (maximum cardinality greater than 1).
 * <p>
 * If mandatory is <code>false</code>, the node is initially empty. Then all UI 
 * elements bound to this node are empty and read only. If mandatory is <code>true</code>,
 * the node is guaranteed to be non-empty upon access. This is very convenient 
 * for value nodes, but take care with model nodes. Because model node elements 
 * require a model instance, the framework can <b>not</b> automatically create 
 * an element to achieve this guarantee (without a supply function) and throws 
 * an exception if the node is empty and you try to access an element.
 * <p>
 * If multiple is <code>false</code> you may only {@link #bind(Collection) bind} 
 * single element collections to the node. (You may find
 * <code>java.util.Collections.singletonList(Object)</code> convenient in this 
 * case.)
 * <p>
 * Besides the lead selection, the node also has a (multiple) selection. You can
 * add or remove elements from this selection via {@link #setSelected(int, boolean)}.
 * You can query this selection via {@link #isMultiSelected(int)}.
 * <p>
 * You can dynamically add value attributes to a node by changing its metadata. 
 * You can add {@link IWDNodeInfo#addAttribute(String, String) unmapped} and 
 * {@link IWDNodeInfo#addMappedAttribute(String, String) mapped} attributes.
 * <p>
 * A node may have child nodes. These child nodes are automatically maintained 
 * by the framework. Nevertheless you can dynamically add nodes by 
 * changing the metadata. See the several addChild methods in {@link IWDNodeInfo}.
 * <p>
 * Access to child nodes is possible through {@link #getChildNode(String, int)}.
 * <p>
 * For many of these methods, the Web Dynpro generator generates special type 
 * safe variants.
 * @author SAP
 * @SAPWebDynproPart 2
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_progmodel/java/com/sap/tc/webdynpro/progmodel/api/IWDNode.java#4 $
 */
public interface IWDNode
{
  /**
   * Constant value indicating that no element is selected.
   */
  int NO_SELECTION = -1;

  /**
   * Constant value indicating the use of the lead selection in some methods.
   */
  int LEAD_SELECTION = -1;

  /**
   * Returns the context, this node belongs to.
   * @return the context, this node belongs to
   */
  IWDContext getContext();

  /**
   * With this method you can bind a new Collection to the Node. It is allowed
   * to bind <code>null</code>; in that case the node is valid containing an
   * empty list.
   * <p>
   * This collection may contain elements for this node, (in case of mapping)
   * elements of any origin node of this or (for a model node) instances of the
   * model class.
   * <p>
   * The node internally creates a <code>List</code> of NodeElements, either
   * being the ones supplied or wrappers thereof. The collection itself is 
   * thrown away. The order of the elements in the List is the order of the
   * Collection's Iterator.
   * <p>
   * The Collection must obey the cardinalities of the Node. It is not allowed
   * to bind an empty collection to a Node of cardinality 1..1 or 1..n, and it
   * is not allowed to bind more than one item to a Node of cardinality 0..1
   * or 1..1.
   * <p> 
   * If the Collection is an <code>ICMIObservableList</code>, the Node behaves
   * differently. In that case, the list is passed to the origin node (in case
   * of mapped nodes). This node does not copy the elements and throw the list
   * away, but uses the list itself to store the elements.
   * <p>
   * An <code>ICMIObservableList</code> is not accessed at all during
   * <code>bind</code>, if cardinality is 0..n and the selection cardinality is
   * is 0..1 or 0..n. Otherwise, <code>isEmpty()</code> or even <code>size()</code>
   * (cardinality 0..1 or 1..1) is called.
   * <p>
   * Take care: If a node has an <code>ICMIObservableList</code> attached,
   * modifications to the node ({@link #addElement(IWDNodeElement)},
   * {@link #removeElement(IWDNodeElement)}, {@link #sortElements(Comparator)}...)
   * change the list and vice versa!
   * <p>
   * <b>Warning:</b> Never use <code>bind</code> if the node has a supplying 
   * relation role. You effectively get the node and the underlying model out of
   * sync. It is a bad practice to use bind outside the supply function, if 
   * there is one, too. Future releases will probably throw an exception in such
   * cases, but old application will still be allowed to for compatibility
   * reasons.
   * 
   * @param items the collection of items, which should be bound to this node
   * @see com.sap.tc.cmi.util.ICMIObservableList
   * @throws ContextException if
   *   <ul>
   *     <li>any of the items do not match the node
   *     <li>the node is singleton and the parent node is empty or does not have
   *       a valid lead selection.
   *   </ul>
   */
  void bind(Collection items);

  /**
   * With this method the node will be invalidated. It clears its element list
   * and calls the supply function again when asked for an element.
   * <p>
   * <b>Warning:</b>This method is not allowed within a supply function.
   * @throws ContextException when called from within a supply function.
   */
  void invalidate();

  /**
   * With this method the node will be validated. In detail, the method does one
   * of the following (in the given order):
   * <ul>
   *  <li>If the node is valid, nothing is done
   *  <li>If there is a supplying relation role or a supply function, it will be
   *    called.
   *  <li>If the cardinality is 1..1 or 1..n, one empty element will be added.
   *  <li>An empty collection is bound.
   * </ul>
   * @return <code>true</code>, if the node has been validated, i.e. has been
   *      invalid before
   */
  boolean validate();

  /**
   * Returns the meta data for this node.
   * @return the meta data for this node
   */
  IWDNodeInfo getNodeInfo();

  /**
   * Returns the element at the given index.
   * @param index the index of the element
   * @return the element at the given index
   * @throws IndexOutOfBoundsException if the node does not have such an element.
   */
  IWDNodeElement getElementAt(int index);

  /**
   * Returns the size of the element list.
   * @return the size of the element list
   */
  int size();

  /**
   * Returns whether the element list is empty.
   * @return <code>true</code> if the list is empty
   *                  <code>false</code> otherwise.
   */
  boolean isEmpty();

  /**
   * Returns the index of the lead selection or NO_SELECTION if the element
   * list is empty.
   * @return the index of the lead selection or NO_SELECTION if the element
   * list is empty.
   * @see #setLeadSelection(int)
   */
  int getLeadSelection();

  /**
   * Sets the lead selection to the given index. It may be called with
   * NO_SELECTION to reset (clear) the lead selection.<p>
   *
   * @param index the index of the element
   *
   * @throws ContextException if caller tries to reset the selection, but selection is mandatory
   * @throws IndexOutOfBoundsException if the given index is not within the element list
   * 
   * @see #getLeadSelection()
   */
  void setLeadSelection(int index);

  /**
   * Clears the Node's multiple selection; does not change the lead selection.
   */
  void clearSelection();

  /**
   * Returns whether the element at the given index is selected.
   * @param index the index of the element
   * @return <code>true</code> if the element is selected.
   * @deprecated This method always returns <code>true</code> if the lead 
   *    selection is positioned at <code>index</code>. As a consequence it is
   *    not possible to determine if the element at the lead selection also is
   *    selected via multiple selection. Instead, compare <code>index</code> with 
   *    {@link #getLeadSelection()} or use {@link #isMultiSelected(int)} depending 
   *    on your needs. This method may be withdrawn with the first new NetWeaver 
   *    release in 2006.
   */
  boolean isSelected(int index);

  /**
   * Returns whether the element at the given index is selected. Does not look
   * at the lead selection.
   * @param index the index of the element
   * @return true if the element is selected.
   */
  boolean isMultiSelected(int index);

  /**
   * Selects or unselects the element at the given index.
   * @param index the index of the element
   * @param selected flag, which indicates, whether the element
   * should be selected or unselected
   */
  void setSelected(int index, boolean selected);

  /**
   * Sets the lead selections of the whole node's subtree such that the path to
   * the given element is selected and all other nodes are unselected. You may
   * pass <code>null</code> to unselect the complete subtree.
   * @param element The element that is to be selected in the subtree
   * @throws ContextException if <code>element</code> is not in this node's
   *      subtree.
   */
  void setTreeSelection(IWDNodeElement element);

  /**
   * Returns the current element (at the lead selection) or <code>null</code>
   * if no element is selected.
   * @return the current element (at the lead selection) or <code>null</code>
   * if no element is selected.
   */
  IWDNodeElement getCurrentElement();

  /**
   * Returns the parent element. Returns <code>null</code>, if there is no such
   * element. The parent element is the one element of the parent node that
   * this node refers to. In case of the context root node, there is none. In
   * case of independent nodes, it is the one element of the root node. In case
   * of singleton nodes, it is the parent's lead selection element. In case of
   * non-singleton, the node exists once for each element of the parent and this
   * method returns exactly this element.
   * @return the parent element or <code>null</code> if there is none
   */
  IWDNodeElement getParentElement();

  /**
   * Adds an element to the node collection. If an ICMIObservableList is bound
   * to the node, this list is modified!
   * @param element An element that matches the node.
   * @throws ContextException if the element does not match the node or if the
   *    node is singleton and the parent node has no valid lead selection element.
   */
  void addElement(IWDNodeElement element);

  /**
   * Adds an element to the node collection at the given index. If an 
   * ICMIObservableList is bound to the node, this list is modified!
   * @param element An element that matches the node.
   * @param index the index where the element shall be inserted
   * @throws ContextException if the element does not match the node or if the
   *    node is singleton and the parent node has no valid lead selection element.
   */
  void addElement(int index, IWDNodeElement element);
  
  /**
   * Removes an element from the node collection. If an ICMIObservableList is
   * bound to the node, this list is modified!
   * @param element The element to be removed
   * @return <code>true</code> if the element was in.
   */
  boolean removeElement(IWDNodeElement element);
  
  /**
   * Moves an element to another position within the node collection. If an 
   * ICMIObservableList is bound to the node, this list is modified!
   * @param from the index of the element to be moved.
   * @param to the index it shall be moved to.
   * @throws IllegalArgumentException if either <code>from</code> or
   *          <code>to</code> are invalid indexes
   */
  void moveElement(int from, int to);
  
  /**
   * Swaps two elements within the node collection. If an 
   * ICMIObservableList is bound to the node, this list is modified!
   * @param index1 the index of the one element
   * @param index2 the index of the other element
   * @throws IllegalArgumentException if either <code>index1</code> or
   *          <code>index2</code> are invalid indexes
   */
  void swapElements(int index1, int index2);

  /**
   * Moves the lead selection to the first element in the node collection,
   * if the list is not empty. Otherwise the method returns silently.
   */
  void moveFirst();

  /**
   * Moves the lead selection to the last element in the node collection,
   * if the list is not empty. Otherwise the method returns silently.
   */
  void moveLast();

  /**
   * Moves the lead selection to the previous element in the node collection,
   * if such an element exists. Otherwise the method returns <code>null</code>
   * and does not change the lead selection.
   * @return the element at the previous position or <code>null</code> if there
   *          is none
   */
  IWDNodeElement movePrevious();

  /**
   * Moves the lead selection to the next element in the node collection,
   * if such an element exists. Otherwise the method returns <code>null</code>
   * and does not change the lead selection.
   * @return the element at the next position or <code>null</code> if there
   *          is none
   */
  IWDNodeElement moveNext();

  /**
   * Moves the lead selection to the element with the given index.
   * @param index the index of the element
   * @throws IndexOutOfBoundsException if index is within the range of the node collection
   */
  void moveTo(int index);


  /**
   * Returns the child Node with the given name at the given index. An index of
   * LEAD_SELECTION is allowed and means the lead selection. If the child Node
   * is singleton, only the special value LEAD_SELECTION and the current value of
   * <code>getLeadSelection()</code> are allowed.
   * Returns <code>null</code> if no node with that name exist.
   *
   * @param name the name of the child node
   * @param index the index of the element
   *
   * @return the child node with the given name at the given index
   *
   * @throws IllegalArgumentException if <code>index</code> is not within the range of the node
   *   collection (all nodes) or if index doesn't represent the lead selection index
   *   (singleton nodes only)
   */
  IWDNode getChildNode(String name, int index);

  /**
   * Creates a new node element of the type used for this <b>value</b> node. 
   * <p>
   * The element is not yet inserted in the node collection,
   * its {@link IWDNodeElement#node()} method will return <code>null</code>.
   * @return the created node element
   */
  IWDNodeElement createElement();

  /**
   * Creates a new node element of the type used for this <b>model</b> node.
   * <p>
   * The element is not yet inserted in the node collection,
   * its {@link IWDNodeElement#node()} method will return <code>null</code>.
   * @param reference a reference to a model instance holding the data for this
   *        node element
   * @return the created node element
   */
  IWDNodeElement createElement(Object reference);

  /**
   * Sorts the elements according to the Comparator. The Comparator gets
   * IWDNodeElements for this node. Since <code>Collections.sort(List)</code> is
   * used, this sort is guaranteed to be <i>stable</i>: equal elements will not
   * be reordered as a result of the sort.
   * <p>
   * Note that the element list is only sorted once. Further adds will again be
   * at the end of the list rendering it unsorted again.
   * <p>
   * If an ICMIObservableList is bound to the node, this list is sorted!
   * 
   * @param comparator a Comparator for <code>IWDNodeElement</code>s of this
   *    node.
   */
  void sortElements(Comparator comparator);
  
  /**
   * Notifies all (directly and indirectly) mapped Nodes. This notification is
   * sent to an IWDNodeNotificationListener that is registered at the NodeInfo.
   * <p>
   * <b>Warning:</b> For technical reasons, this notification will only be sent
   * to nodes that have been touched at least once. The reason is that the
   * mapping links are set up upon first access. Before that an origin node does
   * not even know that there is another node mapping to it.
   * 
   * @param payload The payload is passed to all nodes with the notification.
   * @see IWDNodeNotificationListener
   * @see IWDNodeInfo#addNotificationListener(IWDNodeNotificationListener)
   * @see #validate()
   */
  void notifyAllMappedNodes(Object payload);
}
