/*
 * Copyright (c) 2002 SAP AG - All Rights Reserved.
 *
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_progmodel/java/com/sap/tc/webdynpro/progmodel/api/IWDContext.java#1 $
 */
package com.sap.tc.webdynpro.progmodel.api;

import com.sap.dictionary.runtime.ISimpleType;
import com.sap.dictionary.runtime.ISimpleTypeModifiable;


/**
 * Access to the context data of a controller.
 * <p>
 * All data managed by a controller is contained in its context.
 * Contexts are structured as a tree of nodes which in turn contain attributes.
 * <p>
 * The context interface provides access to the {@link #getRootNode() root node},
 * to the {@link #getRootNodeInfo() meta data} of the root node and to the
 * {@link #getTypeOf(String) type info} for a context attribute.
 * <p>
 * For contexts created with Web Dynpro design time tools, sub interfaces will
 * be generated that provide typed access to the subnodes of the context rood node.
 *
 * @author SAP
 * @SAPWebDynproPart 2
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_progmodel/java/com/sap/tc/webdynpro/progmodel/api/IWDContext.java#1 $
 */
public interface IWDContext
{
  /**
   * Returns the name of this context. The name of a context is the name of the
   * controller to which it belongs, not the unqualified class name of the 
   * context.
   * @return the name of this context
   */
  String getName();

  /**
   * Returns the root node of this context. Since the context root node is
   * created at component creation time, and always has a cardinality of 1..1, 
   * it is guaranteed that the context root node will always exist, and that it
   * will always contain exactly one element.
   * @return the root node of this context
   */
  IWDNode getRootNode();

  /**
   * Returns the node info of the root node of this context.
   * @return the root node info of this context
   */
  IWDNodeInfo getRootNodeInfo();

  /**
   * Returns the data type of the context attribute identified by the given dot
   * delimited path.
   * @param path the path, which identifies the context attribute
   * @return the data type of the context attribute
   * @throws ContextException if the path does not identify an attribute or
   *      if the attribute's type is not an ISimpleType
   */
  ISimpleType getTypeOf(String path);

  /**
   * Returns a modifiable data type of the context attribute identified by the
   * given dot delimited path. Subsequent calls to this method for the same
   * value of path are guaranteed to return the same instance.
   * @param path the path, which identifies the context attribute
   * @return the modifiable data type of the context attribute
   * @throws ContextException if the path does not identify an attribute or
   *      if the attribute's type is not an ISimpleType
   */
  ISimpleTypeModifiable getModifiableTypeOf(String path);

  /**
   * Arbitrarily resets the context to its initial state. All nodes are
   * invalidated, and all dynamically added nodes and attributes are destroyed
   * from the metadata.<p>
   * 
   * <b>Warning:</b> It is the responsibility of the application developer to
   * ensure that all node and attribute mapping or binding dependencies are
   * deactivated before this method is called.<p>
   * 
   * Calling <code>reset()</code> when mapping or binding dependencies are still
   * active will have unpredictable results and possibly lead to a fatal error
   * in the Web Dynpro Framework for your application.
   * 
   * @see IWDNode#invalidate()
   */
  void reset();

  /**
   * Conditionally resets the context to its initial state. By setting the 
   * <code>clear</code> parameter to <code>false</code>, data in node
   * collections created at design time will be preserved.  All dynamically
   * added nodes and attributes are destroyed from the metadata.<p>
   * 
   * Calling reset(true) is functionally equivalent to calling reset().<p>
   * 
   * The same warning given for method {@link #reset()} about mapped or bound
   * nodes and attributes also applies to <code>reset(boolean clear)</code>.
   *
   * @param clear Whether the data shall be cleared before cleaning the 
   *      metadata.
   * @see #reset()
   */
  void reset(boolean clear);
  
  /**
   * Global reset of all changedByClient flags in all context node elements.
   */
  void resetChangedByClient();

  /**
   * Returns the API of the controller this context belongs to.
   */
  IWDController getController();
}
