package com.sap.tc.webdynpro.progmodel.api;

import com.sap.tc.webdynpro.services.api.WDHelperFactory;

/**
 * This class implements the well known ABAP feature "MOVE_CORRESPONDING".
 * <p>
 * @see #copyCorresponding(Object, Object)
 * @author SAP
 * @SAPWebDynproPart 2
 */
public abstract class WDCopyService {

  /**
   * The Web Dynpro equivalent for the well know abap feature
   * "MOVE_CORRESPONDING". It intentionally has been renamed because the term
   * "move", although always being used in Cobol and ABAP, is not quite correct
   * and unusual in Java.
   * <p>
   * The method inspects all attributes of <code>source</code>, searches a
   * compatible attribute in <code>target</code> and copies the value if it 
   * finds one. Values are considered compatible if their Java class matches:
   * <ul>
   *   <li> All numeric classes (Byte, Short, Integer, Long, Float, Double,
   *        BigInteger and BigDecimal) and primitives (byte, short, int, long,
   *        float, double) are compatible.
   *   <li> Boolean class and primitive are compatible.
   *   <li> Character (class) and char (primitive) are compatible.
   *   <li> Otherwise source and target attribute must be based on the same class.
   * </ul>
   * <p>
   * Currently, both <code>source</code> and <code>target</code> may be either
   * an {@link IWDNodeElement} or an {@link com.sap.tc.cmi.model.ICMIGenericModelClass}.
   * If any object is of another class, it is simply ignored. These both classes
   * have been chosen, because they can tell about their structure at run time
   * without the need of Java reflection.
   * <p>
   * @param source The source object
   * @param target The target object
   */
  public static void copyCorresponding(Object source, Object target) {
    instance.copyCorresponding(source, target);
  }

  /**
   * Copies all elements from <code>source</code> to <code>target</code>. 
   * Afterwards <code>source</code> contains as much elements as <code>target</code>,
   * each one either copied via {@link #copyCorresponding(Object, Object)} (if
   * <code>target</code> is a value node) or holding the same model instance (if
   * <code>source</code> and <code>target</code> both are model nodes holding
   * the same class). The lead selection will be copied, too.
   * <p>
   * Note: If you copy a model node to another model node, only the reference to
   * the model instance is copied, but not additional value attributes. Copying
   * a model node to a value node works fine. 
   * @param source The source node
   * @param target The target node
   * @throws WDRuntimeException if <code>target</code> is a model node and 
   *    <code>source</code> is <b>not</b> a model node holding the same class.
   */
  public static void copyElements(IWDNode source, IWDNode target) {
    instance.copyElements(source, target);
  }

  /**
   * Copies all elements from to complete subtree of <code>source</code> to
   * <code>target</code>. This is done by applying {@link #copyElements(IWDNode, IWDNode)}
   * to the nodes and then searching for equally named subnodes and recursively
   * applying <code>copySubtree</code> to them. 
   * @param source The source node
   * @param target The target node
   * @throws WDRuntimeException if {@link #copyElements(IWDNode, IWDNode)}
   *    fails for a node in the subtree.
   */
  public static void copySubtree(IWDNode source, IWDNode target) {
    instance.copySubtree(source, target);
  }

  /**
   * Describes the features required from the internal copy service implementation.
   * 
   * @SAPWebDynproPart 2
   */
  public interface IWDCopyService {
    void copyCorresponding(Object source, Object target);
    void copyElements(IWDNode source, IWDNode target);
    void copySubtree(IWDNode source, IWDNode target);
  }

  private static IWDCopyService instance = (IWDCopyService)WDHelperFactory.getInstance(IWDCopyService.class);
}
