/**
 * Copyright (c) 2002 SAP AG - All Rights Reserved.
 */
package com.sap.tc.webdynpro.progmodel.api;

/**
 * Functionality that is common to all elements in any context node and that can 
 * be used by Web Dynpro applications. While this class is used when programming
 * dynamically, more convenient, typed subinterfaces of this interface are 
 * generated for context nodes whose structure is declared within the Web 
 * Dynpro Development Workbench.
 *
 * @author SAP
 * @SAPWebDynproPart 2
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_progmodel/java/com/sap/tc/webdynpro/progmodel/api/IWDNodeElement.java#2 $
 */
public interface IWDNodeElement
{
  /**
   * Returns the node this element belongs to. If the element has been created
   * but not yet assigned to a node, <code>null</code> is returned.
   *
   * @return the node this element belongs to or <code>null</code>, if
   * the element has been created but not yet assigned to a node
   */
  IWDNode node();

  /**
   * Returns the index of this element in the collection of its node or -1
   * if the element has not been assigned to a node.
   *
   * @return the index of this element in the collection of its node or -1
   * if the element has not been assigned to a node
   */
  int index();

  /**
   * Sets the <code>changed</code> flag for this element. The Web Dynpro 
   * Runtime uses that flag to determine those elements that have been changed 
   * in a request/response cycle.
   * <p>
   * {@link #setAttributeValue(String, Object)} and generated mutator methods in 
   * generated sub interfaces of IWDNodeElement will call <code>changed()</code>
   * automatically. So applications that modify node elements don't have to call
   * <code>changed()</code> explicitly.
   * 
   * @param attributeName The name of the changed attribute or <code>null</code>
   *                       to mark <b>all</b> attributes changed.
   */
  void changed(String attributeName);

  /**
   * Returns <code>true</code> if the <code>changed</code> flag has been set for 
   * this element.
   * <p>
   * <strong>Note:</strong> This flag will be used and reset by the Web Dynpro
   * Runtime.
   * 
   * @return <code>true</code> if the <code>changed</code> has been set
   * for this element or <code>false</code> otherwise.
   */
  boolean isChanged();

  /**
   * Returns <code>true</code> if the <code>changedByClient</code> has been set 
   * for this element. This flag will be set by the framework when an update
   * from the client arrives. It remains <code>true</code> until you reset all
   * flags in the complete context via {@link IWDContext#resetChangedByClient()}.
   * <p>
   * Since this flag is kept per element, but each attribute may actually live
   * in a different element, Web Dynpro always marks both the mapped element and
   * the origin element marked if an attribute is changed. This has the
   * consequence that {@link IWDContext#resetChangedByClient()} must always be
   * called from the context in which you will look for client changes.
   * 
   * @return <code>true</code> if the <code>changedByClient</code> has been set
   * for this element or <code>false</code> otherwise.
   */
  boolean isChangedByClient();

  /**
   * Returns the value of the attribute with the given name as an
   * <code>Object</code>.
   * 
   * @see #setAttributeValue(String, Object)
   * @param attributeName the name of the attribute
   * @return the attribute with the given name as an <code>Object</code>
   * @throws ContextException if no attribute with that name exists
   */
  Object getAttributeValue(String attributeName);

  /**
   * Returns the value of the attribute with the given name formatted using
   * the session locale.
   * @param attributeName
   * @return the formatted value
   * @throws ContextException if no attribute with that name exists
   * @see #getAttributeValue(String)
   */
  String getAttributeAsText(String attributeName);

  /**
   * Sets the value of the attribute with the given name to the given value.
   * 
   * @see #getAttributeValue(String)
   * @param attributeName the name of the attribute
   * @param value the attribute value with the given name as an <code>Object</code>
   * @throws ContextException if no attribute with that name exists
   * @throws ClassCastException if the class of <code>value</code> does not
   *    match the declared class for the attribute
   */
  void setAttributeValue(String attributeName, Object value);

  /**
   * Returns <code>true</code> if this element has a model object
   * @return <code>true</code> if this element has a model object
   */
  boolean hasModel();

  /**
   * Returns the model object contained in this element.
   * @return the contained model object
   * @throws ContextException if this element is not a model node element
   */
  Object model();
}
