/* Title:        Web Dynpro Java Runtime: Messages
 * Description:  application API for reporting exceptions, warnings and success messages
 * Copyright:    Copyright (c) 2002
 * Company:      SAP AG
 * @author SAP
 * @version      $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_progmodel/java/com/sap/tc/webdynpro/progmodel/api/IWDMessageManager.java#1 $
 */

package com.sap.tc.webdynpro.progmodel.api;

import java.util.Iterator;

import com.sap.tc.webdynpro.services.exceptions.WDNonFatalException;
import com.sap.tc.webdynpro.services.exceptions.WDNonFatalRuntimeException;
import com.sap.tc.webdynpro.services.message.WDMessageType;
/**
 * IWDMessageManager is used for reporting runtime exceptions that will not lead to a termination of
 * the current Web Dynpro Component, but the error(s) is/are presented to the end user in a configurable
 * manner. Additionally IWDMessageManager is also used for reporting warnings and success messages 
 * 
 * IWDMessageMananger offers 2 different types of methods. First, all methods starting with 
 * <code>report...</code> only store an exception or exception message and return without raising any exception. 
 * These methods are used in conjunction with multiple error presentation. Second, all methods starting with 
 * <code>raise...</code> internally raise some exception in order to 
 * continue with framework error processing. If running in multpile error mode, using method 
 * <code>raisePendingExceptions</code> checks for any error still stored and in case
 * at least a single error is pending it also internally raise some exception in order to continue with 
 * framework error processing.
 * 
 * Some method have an addtional parameter <code>cancelNavigation</code>. When setting this parameter to true, 
 * the framework does not execute navigation to inbound plugs in case at least 1 error is pending. 
 * 
 * @SAPWebDynproPart 2
 */
public interface IWDMessageManager {
  
  
//  /**
//   * Interface describing a single message item.  
//   */
//  public interface Item {
//
//    /**
//     * Return is the message associated to the exception item. 
//     * 
//     * @return the message associated to the exception item
//     */
//    public String getMessage();
//
//    /**
//     * The type of message. 
//     * 
//     * @return the message type 
//     */
//    public WDMessageType getType();
//
//  }  
//  
//  

//   * @deprecated Use {@link #raiseException(WDNonFatalRuntimeException, boolean)} instead and 
//   * set parameter cancelNavigation to </code>true</code>.
//   */
//  public void raiseException(WDNonFatalRuntimeException ex);

	/**
	 * Report a <code>WDNonFatalRuntimeException</code> as error to the client. This method 
	 * does not return, but raises some framework internal exception instead in order to 
	 * return to the framework error handler.
	 *
	 * @param ex is the exception raised by some validation routine
	 * @param cancelNavigation: set this parameter to <code>true</code>, if all navigation steps 
	 * should be canceled due to this error. (see also {@link #cancelNavigation()}).  
	 */
	public void raiseException(WDNonFatalRuntimeException ex, boolean cancelNavigation);

//	/**
//	 * @deprecated Use {@link #raiseException(WDNonFatalException, boolean)} instead and 
//	 * set parameter cancelNavigation to </code>true</code>.
//	 */
//	public void raiseException(WDNonFatalException ex);

  /**
   * Reports a <code>WDNonFatalException</code> as error to the client. This method 
   * does not return, but raises some framework internal exception instead in order 
   * to return to the framework error handler.
   *
   * @param ex is the exception raised by some validation routine
	 * @param cancelNavigation: set this parameter to <code>true</code>, if all navigation steps 
	 * should be canceled due to this error. (see also {@link #cancelNavigation()}). 
   */
  public void raiseException(WDNonFatalException ex, boolean cancelNavigation);


//	/**
//	 * @deprecated Use {@link #raiseException(String, boolean)} instead and 
//	 * set parameter cancelNavigation to </code>true</code>.
//	 */
//	public void raiseException(String message);

  /**
   * Reports the given message to the client. This method does 
   * not return but raises some framework internal exception instead in order to return 
   * to the framework error handler.
   *
   * @param message the message raised by some validation routine
	 * @param cancelNavigation: set this parameter to <code>true</code>, if all navigation steps 
	 * should be canceled due to this error. (see also {@link #cancelNavigation()}). 
   */
  public void raiseException(String message, boolean cancelNavigation);


//	/**
//	 * @deprecated Use {@link #reportException(WDNonFatalException, boolean)} instead and 
//	 * set parameter cancelNavigation to </code>true</code>.
//	 */
//	public void reportException(WDNonFatalException ex);

  /**
   * Report the exception <code>ex</code> to the client. This method returns and may be 
   * called as often as necessary.
   * 
   * @param ex the exception raised by some validation routine
	 * @param cancelNavigation: set this parameter to <code>true</code>, if all navigation steps 
	 * should be canceled due to this error. (see also {@link #cancelNavigation()}).
   */
  public void reportException(WDNonFatalException ex, boolean cancelNavigation);
    

//	/**
//	 * @deprecated Use {@link #reportException(WDNonFatalRuntimeException, boolean)} instead and 
//	 * set parameter cancelNavigation to </code>true</code>.
//	 */
//	public void reportException(WDNonFatalRuntimeException ex); 

  /**
   * Report the exception <code>ex</code> to the client. This method returns and may be 
   * called as often as necessary.
   * 
   * @param ex is the exception to be reported.
	 * @param cancelNavigation: set this parameter to <code>true</code>, if all navigation steps 
	 * should be canceled due to this error. (see also {@link #cancelNavigation()}).
   */
  public void reportException(WDNonFatalRuntimeException ex,boolean cancelNavigation); 


//	/**
//	 * @deprecated Use {@link #reportException(String, boolean)} instead and 
//	 * set parameter cancelNavigation to </code>true</code>.
//	 */
//	public void reportException(String message);

	//***********************************************************************************************************
	// 
	//   S T R I N G       A P I s
	//
	//***********************************************************************************************************
  /**
   * Report error message <code>message</code>. This method returns and may be 
   * called as often as necessary.
   * 
   * @param message is the human readable message that will be displayed on the client
	 * @param cancelNavigation: set this parameter to <code>true</code>, if all navigation steps 
	 * should be canceled due to this error. (see also {@link #cancelNavigation()}).
   */
  public void reportException(String message,boolean cancelNavigation);

  /**
   * Report warning message <code>message</code>. 
   * 
   * @param message is the human readable message that will be displayed on the client
   */
  public void reportWarning(String message);
		
  /**
   * Report an message <code>message</code>. 
   * 
   * @param message is the human readable message that will be displayed on the client
   */
  public void reportSuccess(String message);


//	/**
//	 * @deprecated Use {@link #raiseInvalidContextAttributeException(IWDNodeElement, IWDAttributeInfo, WDNonFatalException, boolean)} instead and 
//	 * set parameter cancelNavigation to </code>true</code>.
//	 */
//	public void raiseInvalidContextAttributeException(IWDNodeElement element, String attributeName, WDNonFatalException ex );

  /**
   * Raises an exception caused by an invalid context attribute value.  
   * 
   * @param element is a reference to the context node element containing the invalid attribute
   * @param attribute is the attribute info of the attribute which is invalid
   * @param ex is the exception raised by some validation routine
	 * @param cancelNavigation: set this parameter to <code>true</code>, if all navigation steps 
	 * should be canceled due to this error. (see also {@link #cancelNavigation()}).
  */  
	public void raiseInvalidContextAttributeException(IWDNodeElement element, IWDAttributeInfo attribute, WDNonFatalException ex , boolean cancelNavigation);

//	/**
//	 * @deprecated Use {@link #reportInvalidContextAttributeException(IWDNodeElement , IWDAttributeInfo , WDNonFatalException, boolean)} instead and 
//	 * set parameter cancelNavigation to </code>true</code>.
//	 */
//	public void reportInvalidContextAttributeException(IWDNodeElement element, String attributeName, WDNonFatalException ex );

	/**
	 * Report an exception caused by an invalid context attribute value. This method returns and may be 
	 * called as often as necessary.
	 * 
	 * @param element is a reference to the context node element containing the invalid attribute
	 * @param attribute is the attribute info of the attribute which is invalid
	 * @param ex is the exception raised by some validation routine
	 * @param cancelNavigation: set this parameter to <code>true</code>, if all navigation steps 
	 * should be canceled due to this error. (see also {@link #cancelNavigation()}). 
	*/  
	public void reportInvalidContextAttributeException(IWDNodeElement element, IWDAttributeInfo attribute, WDNonFatalException ex, boolean cancelNavigation );
  





//	/**
//	 * @deprecated Use {@link #raiseInvalidContextAttributeException(IWDNodeElement, IWDAttributeInfo , String, boolean)} instead and 
//	 * set parameter cancelNavigation to </code>true</code>.
//	 */
//	public void raiseInvalidContextAttributeException(IWDNodeElement element, String attributeName, String message );
  
  /**
   * Report an exception caused by an invalid context attribute value. This method 
   * does not return, but raises some framework internal exception instead in order to 
   * return to the framework error handler. 
   * 
   * @param element is a reference to the context node element containing the invalid attribute
   * @param attribute is the attribute info of the attribute which is invalid
   * @param message is the error message to be displayed
	 * @param cancelNavigation: set this parameter to <code>true</code>, if all navigation steps 
	 * should be canceled due to this error. (see also {@link #cancelNavigation()}).
  */  
	public void raiseInvalidContextAttributeException(IWDNodeElement element, IWDAttributeInfo attribute, String message, boolean cancelNavigation );

//	/**
//	 * @deprecated Use {@link #reportInvalidContextAttributeException(IWDNodeElement, IWDAttributeInfo, String, boolean)} instead and 
//	 * set parameter cancelNavigation to </code>true</code>.
//	 */
//  public void reportInvalidContextAttributeException(IWDNodeElement element, String attributeName, String message );



  /**
   * Report an exception caused by an invalid context attribute value. This method returns and may be 
	 * called as often as necessary.
   * 
   * @param element is a reference to the context node element containing the invalid attribute
   * @param attribute is the attribute info of the attribute which is invalid
   * @param message is the error message to be displayed
	 * @param cancelNavigation: set this parameter to <code>true</code>, if all navigation steps 
	 * should be canceled due to this error. (see also {@link #cancelNavigation()}). 
  */  
  public void reportInvalidContextAttributeException(IWDNodeElement element, IWDAttributeInfo attribute, String message, boolean cancelNavigation );


  //***********************************************************************************************************
  // 
  //   M E S S A G E      I T E M      A P I s
  //
  //***********************************************************************************************************

//	/**
//	 * @deprecated Use {@link #raiseInvalidContextAttributeMessage(IWDNodeElement, IWDAttributeInfo, IWDMessage, Object[], boolean)} instead and 
//	 * set parameter cancelNavigation to </code>true</code>.
//	 */
//	public void raiseInvalidContextAttributeMessage(IWDNodeElement element, String attributeName, IWDMessage messageItem, Object[] args);


 /**
   * Raises a message caused by an invalid context attribute value.This method internally raises a Runtime exception 
   * and execution is continued in the Web Dynpro framework. It is not recommended to call this method using warnings
   * or success messages. 
   * 
   * @param element is a reference to the context node element containing the invalid attribute
   * @param attribute is the attribute info of the attribute which is invalid
   * @param messageItem is the message associated with the context attribute 
   * @param args are the arguments for the message parameters in the same way as used in @see java.text.MessageFormat
	 * @param cancelNavigation: set this parameter to <code>true</code>, if all navigation steps 
	 * should be canceled due to this message. This applies only for messages of type error. For all other 
	 * message types this parameter is ignored. (see also {@link #cancelNavigation()}). 
   */
	public void raiseInvalidContextAttributeMessage(IWDNodeElement element, IWDAttributeInfo attribute, IWDMessage messageItem, Object[] args, boolean cancelNavigation);
  

//	/**
//	 * @deprecated Use {@link #reportContextAttributeMessage(IWDNodeElement , IWDAttributeInfo , IWDMessage , Object[] , boolean , boolean)} instead and 
//	 * set parameter cancelNavigation to </code>true</code>.
//	 */
//	public void reportInvalidContextAttributeMessage(IWDNodeElement element, String attributeName, IWDMessage messageItem, Object[] args);

 /**
   * Report a message item caused for the given context attribute value. In case the message item 
   * is an error, this method returns and may be called as often as necessary. 
   * 
   * @param element is a reference to the context node element containing the invalid attribute
   * @param attribute is the attribute info of the attribute which is invalid
   * @param messageItem is the message associated with the context attribute 
   * @param args are the arguments for the message parameters in the same way as used in @see java.text.MessageFormat
	 * @param cancelNavigation: set this parameter to <code>true</code>, if all navigation steps 
	 * should be canceled due to this message. Tihs applies only for messages of type error. For all other message
	 * types this parameter is ignored. (see also {@link #cancelNavigation()}).
   */
  public void reportContextAttributeMessage(IWDNodeElement element, IWDAttributeInfo attribute, IWDMessage messageItem, Object[] args, boolean cancelNavigation);


//	/**
//	 * @deprecated Use {@link #raiseMessage(IWDMessage , Object[] , boolean )} instead and 
//	 * set parameter cancelNavigation to </code>true</code>.
//	 */
//	public void raiseMessage(IWDMessage messageItem, Object[] args);

 /**
   * Raises a message for the given context attribute value. This method internally raises a Runtime exception 
   * and execution is continued in the Web Dynpro framework. It is not recommended to call this method with warnings
   * and success messages. 
   * 
   * @param messageItem is the message associated with the context attribute 
   * @param args are the arguments for the message parameters in the same way as used in @see java.text.MessageFormat
	 * @param cancelNavigation: set this parameter to <code>true</code>, if all navigation steps 
	 * should be canceled due to this message. This applies only for messages of type error. For all other
	 * message types this parameter is ignored. (see also {@link #cancelNavigation()}).
   */
 public void raiseMessage(IWDMessage messageItem, Object[] args, boolean cancelNavigation);


// /**
//	* @deprecated Use {@link #reportMessage(IWDMessage , Object[] , boolean )} instead and 
//	* set parameter cancelNavigation to </code>true</code>.
//	*/
// public void reportMessage(IWDMessage messageItem, Object[] args);
//  
 /**
   * Report a message item caused by an invalid context attribute value. In case the message item 
   * is an error, this method returns and may be called as often as necessary.
   * 
   * 
   * @param messageItem is the message associated with the context attribute 
   * @param args are the arguments for the message parameters in the same way as used in @see java.text.MessageFormat
	 * @param cancelNavigation: set this parameter to <code>true</code>, if all navigation steps 
	 * should be canceled due to this message. This applies only for messages of type error. For all other
	 * message types this parameter is ignored. (see also {@link #cancelNavigation()}).
   */
 public void reportMessage(IWDMessage messageItem, Object[] args, boolean cancelNavigation);


  /**
   * This method checks if any exceptions were reported to the exception manager and are still stored in the 
   * exceptions manager. In case there is at least one exception still stored, this method does not return, but .
   * raises some framework internal exception instead in order to return to the framework error handler.
	*/
 public void raisePendingException();
  


  /**
   * This methods cancels all subsequent navigational steps for the current request/ response cycle. Since
   * Web Dynpro processes the navigational links in a queued manner the first inbound plug is called 
   * after all outbound plugs are called (as long as an outbound plug is not called from within an inbound plug). 
   * Calling this methods subpresses the processing of the outbound plugs. The queue of called plugs is resetted 
   * after the request / response cycle. Navigation has to be reinitiated for the next request (Eventhandler).
   * 
   * <b>Note:</b> Navigation is not only canceled for the component calling this method, but for the entire application!
   */
  public void cancelNavigation();

  /**
   * Returns true if no exception was reported yet.
   * 
   * @return true if no exception was reported yet
   */
  public boolean hasExceptions();

  /**
   * Returns true if no exception, warning, info or success message is stored.
   * 
   * @return true if no exception was reported yet
   */
  public boolean isEmpty();

//  /**
//   * @deprecated returns null. 
//   * 
//   * @return iterator containing <code>IWDExceptionManager.Item</code> instances
//   */
//   public Iterator iterator();
  
  /**
   * Ask for a specific message in the message managers queue. 
   * 
   * @return true if message  <code>messageItem</code> is currently 
   * pending in the message queue.
   */
  public boolean contains(IWDMessage messageItem);


  /**
   * Report an exception caused by an invalid context attribute value. This method 
   * does not return, but raises some framework internal exception instead in order to 
   * return to the framework error handler. 
   * 
   * @param element is a reference to the context node element containing the invalid attribute
   * @param attributeName is the name of the attribute which is invalid
   * @param ex is the exception raised by some validation routine
  */  
  
}
