/*
 * Copyright (c) 2002 SAP AG - All Rights Reserved.
 *
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_progmodel/java/com/sap/tc/webdynpro/progmodel/api/IWDComponentUsage.java#2 $
 */
package com.sap.tc.webdynpro.progmodel.api;

import com.sap.tc.webdynpro.progmodel.repository.IWDComponentUsageInfo;
import com.sap.tc.webdynpro.progmodel.repository.IWDEventHandlerInfo;
import com.sap.tc.webdynpro.progmodel.repository.IWDEventInfo;

/**
 * API of component usages.
 * A component usage provides access to the interfaces of an embedded component
 * and manages the lifecycle of an embedded component instance.
 * 
 * @author SAP
 * @SAPWebDynproPart 2
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_progmodel/java/com/sap/tc/webdynpro/progmodel/api/IWDComponentUsage.java#2 $
 */
public interface IWDComponentUsage {

  /**
   * Returns the metadata describing this component usage. 
   * @return the component usage metadata object
   */
  public IWDComponentUsageInfo getComponentUsageInfo();

  /**
   * Creates a component instance for this component usage.
   * The type defined by the component usage declared at design time is used
   * for creating the component instance.
   * The method fails if the Component Usage is associated to a Component
   * Interface Definition.
   * 
   * Precondition: hasActiveComponent() == false && isReferencing() == false
   */
  public void createComponent();

  /**
   * Creates a component instance for this component usage.
   * The Component implementation is in the same deployable object as this
   * Component Usage.
   * 
   * Precondition: hasActiveComponent() == false && isReferencing() == false
   * 
   * @param componentName The fully qualified name of the Component
   * Implementation to use
   */
  public void createComponent(String componentName);

  /**
   * Creates a component instance for this component usage.
   *
   * Precondition: hasActiveComponent() == false && isReferencing() == false
   * 
   * @param componentName The fully qualified name of the Component
   * Implementation to use
   * @param deployableObjectName Name of the deployable object which contains
   * the Component implementation to use
   */
  public void createComponent(String componentName, String deployableObjectName);

  /**
   * Deletes the component instance associated with this component usage.
   * If no instance exists, the method exits silently.
   * 
   * Precondition: isReferencing() == false
   */
  public void deleteComponent();

  /**
   * Returns true iff this component usage has an existing component instance.
   * @return true iff this component usage has an existing component instance
   */
  public boolean hasActiveComponent();

  /**
   * Get the interface controller of the component instance.
   * The returned generic controller interface object should be cast as the
   * specific component interface IExternal<&lt;>name<&gt;>.
   * 
   * Precondition: hasActiveComponent() == true
   */
  public IWDExternalControllerInterface getInterfaceController();

  /**
   * Get the configuration controller of the component instance.
   * The returned generic controller interface object should be cast as the
   * specific component interface IExternal<&lt;>name<&gt;>.
   * 
   * Precondition: hasActiveComponent() == true
   */
  public IWDExternalControllerInterface getConfigurationController();

  /**
   * Component usage does not manage its own component instance from now on.
   * Instead, the lifecycle is controlled by the component usage
   * <code>componentUsage</code>.
   * 
   * Note: Accessing a visual interface via a component usage in referecing mode
   * ist not supported. Reverse context mapping via a component usage in 
   * referencing mode is not supported. Nevertheless, the component's visual
   * interface and reverse context mapping can be used by the component usage
   * which refers to the component instnce without being in referencing mode. 
   * 
   * Precondition: isReferencing() == true || hasActiveComponent() == false
   * Precondition: The referenced component usage and this component usage must
   * refer to comatible types. This condition is met if <br>
   * * the component usages refer to the same component implementation<br>
   * * the component usages refer to the same component interface definition<br>
   * * the referencing component usage refers to a component interface
   * definition which is implemented by the component implementation the
   * referred component usage referres to<br>.
   * 
   * @param referencedUsage component usage this component usage will refer to
   */
  public void enterReferencingMode(IWDComponentUsage componentUsage);

  /**
   * @deprecated This API may be withdrawn with the first new NetWeaver release in 2006.
   *   Use {@link #enterReferencingMode(IWDComponentUsage)} instead.
   */
  public void enterReferencingModeUnsafe(IWDComponentUsage componentUsage);

  /**
   * Restores the lifecycle manage of the component usage back to itself.
   * If the component usage is not in referencing mode, then the method
   * exits silently.
   */
  public void leaveReferencingMode();

  /**
   * Returns true iff this component usage is in referencing mode.
   * @see IWDComponentUsage#enterReferencingMode(IWDComponentUsage)
   * @return true iff this component usage is in referencing mode
   */
  public boolean isReferencing();

  /**
   * Returns true if this component usage is referenced to by other component
   * usages.
   * @see IWDComponentUsage#enterReferencingMode(IWDComponentUsage)
   */
  public boolean isReferenced();

  /**
   * Creates a component usage with the given name which has the same component
   * interface type assigned as this component usage.
   * @param name the name of the component usage to create
   * @return the newly created component usage
   */
  public IWDComponentUsage createComponentUsageOfSameType(String name);

  /**
   * Subscribe the specified event handler to the specified event.
   * The event must be defined in the interface controller or in the
   * configuration controller of the component this component usage refers to.
   * The controller must be part of the parent component of this component
   * usage.
   * @param event Repository object of the event to subscribe to
   * @param eventHandler Repository object of the event handler
   * @param controller Controller which owns the event handler
   */
  public void addEventHandler(IWDEventInfo event, IWDEventHandlerInfo eventHandler, IWDController controller);

  /**
   * Unsubscribe the specified event handler from the specified event.
   * The event handler must have been subscribed with
   * {@link IWDComponentUsage#addEventHandler(IWDEventInfo, IWDEventHandlerInfo, IWDController)}
   * .
   * 
   * @param event Repository object of the event to unsubscribe from
   * @param eventHandler Repository object of the event handler
   * @param controller Controller which owns the event handler
   */
  public void removeEventHandler(IWDEventInfo event, IWDEventHandlerInfo eventHandler, IWDController controller);
}
