/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: SimpleSerializer.java,v 1.2 2004/03/04 18:53:22 jre Exp $
 */

package com.sapportals.wcm.util.xml;

import java.io.StringWriter;
import java.io.Writer;
import java.util.*;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.w3c.dom.*;

import com.sap.tc.logging.Location;

/**
 * Simple DOM serializer. Only supports UTF-8 encoding and simple indentation
 * (whitespace preservation not implemented - if whitespace preservation is
 * required simply do not indent). <p>
 *
 * Copyright (c) SAP AG 2001-2004
 * @author stefan.eissing@greenbytes.de
 * @author julian.reschke@greenbytes.de
 * @version $Id: SimpleSerializer.java,v 1.2 2004/03/04 18:53:22 jre Exp $
 */

public final class SimpleSerializer {

  private final static Location log = Location.getLocation(SimpleSerializer.class);

  /**
   * Namespace name for the namespace hardwired to the prefix "xml:"
   */
  public final static String NS_XML = "http://www.w3.org/XML/1998/namespace";

  /**
   * Namespace name for the namespace hardwired to the prefix "xmlns:"
   */
  public final static String NS_XMLNS = "http://www.w3.org/2000/xmlns/";


  /**
   * Serialize the document to a string.
   *
   * @param doc the document
   * @param indent whether to produce indented XML or not
   * @return the XML string or an empty string in case of error
   */
  public static String toString(Document doc, boolean indent) {
    return toString(doc, indent, true);
  }

  /**
   * Serialize the document to a string.
   *
   * @param doc the document
   * @param indent whether to produce indented XML or not
   * @param doNsProcessing enable/disable special namespace support in the
   *      serializer
   * @return the XML string or an empty string in case of error
   */
  public static String toString(Document doc, boolean indent, boolean doNsProcessing) {

    try {
      StringWriter sw = new StringWriter();
      write(sw, doc, indent, doNsProcessing);
      return sw.getBuffer().toString();
    }
    catch (java.io.IOException ex) {
            //$JL-EXC$      
      log.errorT("toString(80)", ex.toString());
      return "";
    }
  }

  /**
   * Serialize the element (and it's children) to a string.
   *
   * @param indent whether to produce indented XML or not
   * @param n node to serialize
   * @return the XML string or an empty string in case of error
   * @throws java.lang.IllegalArgumentException when node is a AttributeNode
   */
  public static String toString(Node n, boolean indent)
    throws java.lang.IllegalArgumentException {
    if (Node.ATTRIBUTE_NODE == n.getNodeType()) {
      throw new IllegalArgumentException("node is an attribute node");
    }
    try {
      StringWriter sw = new StringWriter();
      write(sw, n, indent, true);
      return sw.getBuffer().toString();
    }
    catch (java.io.IOException ex) {
            //$JL-EXC$      
      log.errorT("toString(104)", ex.toString());
      return "";
    }
  }

  private static void write(Writer writer, Node node, boolean indent, boolean doNsProcessing)
    throws java.io.IOException {
    SimpleSerializer ser = new SimpleSerializer(writer, indent ? "  " : null);
    ser.setNsAware(doNsProcessing);
    ser.serialize(node);
  }

  /**
   * Holds linked list of namespace maps.
   */
  private class NamespaceMap extends HashMap {

    private NamespaceMap parent;

    /**
     * Construct a new namespace map based on an existing one.
     *
     * @param parent Base map.
     */
    public NamespaceMap(NamespaceMap parent) {
      this.parent = parent;
    }

    /**
     * Resolve namespace prefix.
     *
     * @param prefix namespace prefix to be resolved
     * @return namespace name or <code>null</code> when not mapped.
     */
    public String get(String prefix) {

      String result = (String)super.get(prefix);

      if (result != null) {
        return result;
      }
      else if (this.parent == null) {
        return null;
      }
      else {
        return this.parent.get(prefix);
      }
    }

  }

  private String indent;
  private final int nesting;
  private final Writer writer;
  private final NamespaceMap map;
  private boolean nsProcessing = true;
  private boolean optimizeNSDeclarations;
  private Element openElement = null;

  /**
   * Set to <code>false</code> to switch off namespace processing (enabling
   * strict DOM level 1 serialization)
   *
   * @param nsAware flag
   */

  public void setNsAware(boolean nsAware) {
    this.nsProcessing = nsAware;
  }

  public void setOptimizeNSDeclarations(boolean doOptimize) {
    this.optimizeNSDeclarations = doOptimize;
  }
  
  /**
   * Construct a new Serializer
   *
   * @param writer output
   * @param indent indentation string
   */

  public SimpleSerializer(Writer writer, String indent) {
    this.indent = indent;
    this.writer = writer;
    this.nesting = 0;

    // create default map
    this.map = new NamespaceMap(null);
    this.map.put("", "");
    this.map.put("xml", NS_XML);
    this.map.put("xmlns", NS_XMLNS);
  }

  /**
   * Construct a new serializer with a specific namespace mapping as base map
   * (used when creating serializers for child nodes when streaming)
   *
   * @param writer output
   * @param map initial namespace map
   * @param indent indentation string
   * @param nesting depth of nesting
   */
  private SimpleSerializer(Writer writer, String indent, int nesting, NamespaceMap map) {
    this.indent = indent;
    this.writer = writer;
    this.map = map;
    this.nesting = nesting;
  }

  /**
   * Serializes the open tag of the specified element and returns a new
   * SimpleSerializer object for use in serializing any child elements (and
   * writing the closing tag).
   *
   * @param element Element for which the start tag shall be written
   * @return new Serializer object
   * @exception java.io.IOException Exception raised in failure situation
   */

  public SimpleSerializer openElement(Element element)
    throws java.io.IOException {
    this.openElement = element;

    NamespaceMap map = startElement(element, this.nesting, false, this.map);
    SimpleSerializer s = new SimpleSerializer(this.writer, this.indent, this.nesting + 1, map);
    s.setNsAware(this.nsProcessing);
    s.setOptimizeNSDeclarations(this.optimizeNSDeclarations);
    return s;
  }

  /**
   * Closes the currently open element. Do not use this object after calling
   * this method!
   *
   * @exception java.io.IOException Exception raised in failure situation
   */
  public void closeElement()
    throws java.io.IOException {
    endElement(this.openElement, this.nesting);
    this.openElement = null;
  }


  private void indent(int level)
    throws java.io.IOException {
    if (this.indent != null) {
      for (int i = 0; i < level; i++) {
        this.writer.write(this.indent);
      }
    }
  }

  private void newline()
    throws java.io.IOException {
    if (this.indent != null) {
      this.writer.write("\n");
    }
  }

  private void ser(Node n, int level, NamespaceMap map)
    throws java.io.IOException {

    switch (n.getNodeType()) {

      case Node.COMMENT_NODE:
        serComment((Comment)n, level + 1);
        break;
      case Node.DOCUMENT_NODE:
        for (Node c = n.getFirstChild(); c != null; c = c.getNextSibling()) {
          ser(c, -1, map);
        }
        break;
      case Node.DOCUMENT_TYPE_NODE:
        serDocumentType((DocumentType)n);
        break;
      case Node.ELEMENT_NODE:
        serElement((Element)n, level + 1, map);
        break;
      case Node.PROCESSING_INSTRUCTION_NODE:
        serPI((ProcessingInstruction)n, level + 1);
        break;
      case Node.CDATA_SECTION_NODE:
      // is handled like a text node
      case Node.TEXT_NODE:
        serText((Text)n, level + 1, false);
        break;
      default:
        log.debugT("ser(284)", "unsupported node type " + n.getNodeType());
        break;
    }
  }

  private String getPrefixFromNsDecl(Attr a) {
    String ns = a.getNamespaceURI();
    if (ns == null) {
      ns = "";
    }

    if (!ns.equals(NS_XMLNS) && !ns.equals("")) {
      return null;
    }
    else {
      String name = a.getName();

      if (name.startsWith("xmlns:")) {
        return name.substring("xmlns:".length());
      }
      else if (name.equals("xmlns")) {
        return "";
      }
      else {
        return null;
      }
    }
  }

  private boolean strequals(String a, String b) {
    if (a == null) {
      return a == b;
    }
    else {
      return a.equals(b);
    }
  }


  private NamespaceMap startElement(Element n, int level, boolean recurse,
    NamespaceMap parentMap)
    throws java.io.IOException {

    NamespaceMap map = parentMap;
    Map localDecls = null;
    Set knownNSAttributes = null;
    NamedNodeMap attrs = n.getAttributes();

    if (this.nsProcessing) {
      // start by adding all explicit namespace declarations to the map

      for (int i = 0; i < attrs.getLength(); i++) {
        Attr a = (Attr)attrs.item(i);

        String prefix = getPrefixFromNsDecl(a);

        if (prefix != null) {
          String nsUri = a.getNodeValue();

          if (!nsUri.equals(map.get(prefix))) {
            if (map == parentMap) {
              map = new NamespaceMap(parentMap);
            }
//m_logger.errorT(level + " adding NS declaration " + prefix + "->" + nsUri);
            map.put(prefix, nsUri);
          }
          else if (this.optimizeNSDeclarations) {
            // we know the namespace declarations already from previous elements
            if (knownNSAttributes == null) {
              knownNSAttributes = new HashSet();
            }
            knownNSAttributes.add(a);
          }
        }
      }

      // check namespace of this element
      String ns = n.getNamespaceURI();
      if (ns == null) {
        ns = "";
      }
      String pr = n.getPrefix();
      if (pr == null) {
        pr = "";
      }

      if (!strequals(ns, map.get(pr))) {
//m_logger.errorT(level + " need NS decl for this element: " + pr + "->" + ns);

        if (localDecls == null) {
          localDecls = new HashMap();
        }
        localDecls.put(pr, ns);

        if (map == parentMap) {
          map = new NamespaceMap(parentMap);
        }
        map.put(pr, ns);
      }

      // check namespaces of attributes
      for (int i = 0; i < attrs.getLength(); i++) {
        Attr a = (Attr)attrs.item(i);

        //m_logger.errorT(level + " attr name: " + a.getName() + ", prefix: " + a.getPrefix() +
        //  ", namespace: " + a.getNamespaceURI() +
        //  ", localname: " + a.getLocalName() +
        //  ", value: " + a.getNodeValue()
        //  );

        String ans = a.getNamespaceURI();
        if (ans == null) {
          ans = "";
        }
        String apr = a.getPrefix();
        String inherited = map.get(apr);

        if (!ans.equals("") && // attribute is in no namespace
        !strequals(ans, inherited) && // decl already in scope
        !strequals(ans, NS_XMLNS)) {// and not a NS decl...

          //m_logger.errorT(level + " need NS decl for this attribute: " + apr + "->" + ans);

          //        if (inherited != null)
          //        {
          // the prefix is already mapped -- we would need to do namespace
          // rewriting, which we don't want to
          //          throw new java.io.IOException ("SimpleSerializer doesn't support NS prefix rewriting.");
          //        }

          if (localDecls == null) {
            localDecls = new HashMap();
          }
          localDecls.put(apr, ans);

          if (map == parentMap) {
            map = new NamespaceMap(parentMap);
          }
          map.put(apr, ans);
        }
      }
    }

    Node firstChild = recurse ? n.getFirstChild() : null;
    String displayname = n.getNodeName();

    indent(level);

    if (recurse && firstChild == null) {
      // empty elememts
      this.writer.write("<" + displayname);
      serAttributes(attrs, localDecls, knownNSAttributes);
      this.writer.write("/>");
    }
    else if (recurse && firstChild.getNodeType() == Node.TEXT_NODE && firstChild.getNextSibling() == null) {
      // elements that contain just one text node
      this.writer.write("<" + displayname);
      serAttributes(attrs, localDecls, knownNSAttributes);
      this.writer.write(">");
      serText((Text)firstChild, level, true);
      this.writer.write("</" + displayname + ">");
    }
    else if (recurse && firstChild.getNodeType() == Node.ELEMENT_NODE && firstChild.getNextSibling() == null &&
      firstChild.hasChildNodes() == false) {
      // elements that contain just one empty element node
      this.writer.write("<" + displayname);
      serAttributes(attrs, localDecls, knownNSAttributes);
      this.writer.write(">");
      String oldindent = this.indent;
      this.indent = null;
      serElement((Element)firstChild, level, map);
      this.indent = oldindent;
      this.writer.write("</" + displayname + ">");
    }
    else if (recurse) {
      // all other cases
      this.writer.write("<" + displayname);
      serAttributes(attrs, localDecls, knownNSAttributes);
      this.writer.write(">");
      newline();
      for (Node c = n.getFirstChild(); c != null; c = c.getNextSibling()) {
        ser(c, level, map);
      }
      indent(level);
      this.writer.write("</" + displayname + ">");
    }
    else {// just open the tag
      this.writer.write("<" + displayname);
      serAttributes(attrs, localDecls, knownNSAttributes);
      this.writer.write(">");
    }

    newline();

    return map;
  }

  private void endElement(Element n, int level)
    throws java.io.IOException {
    String displayname = n.getNodeName();
    indent(level);
    this.writer.write("</" + displayname + ">");
    newline();
  }


  private void serElement(Element n, int level, NamespaceMap parentMap)
    throws java.io.IOException {
    startElement(n, level, true, parentMap);
  }

  private void serDocumentType(DocumentType dt)
    throws java.io.IOException {
    Element docElement = dt.getOwnerDocument().getDocumentElement();
    if (dt.getPublicId() != null && dt.getSystemId() != null) {
      this.writer.write("<!DOCTYPE " + docElement.getLocalName() + " PUBLIC \"" + dt.getPublicId() + "\" \"" + dt.getSystemId() + "\">");
    }
    else if (dt.getSystemId() != null) {
      this.writer.write("<!DOCTYPE " + docElement.getLocalName() + " SYSTEM \"" + dt.getSystemId() + "\">");
    }
    else {
      newline();
    }
  }

  private void serPI(ProcessingInstruction p, int level)
    throws java.io.IOException {
    indent(level);
    this.writer.write("<?" + p.getTarget() + " " + p.getNodeValue() + "?>");
    newline();
  }

  private void serComment(Comment c, int level)
    throws java.io.IOException {
    indent(level);
    this.writer.write("<!-- " + c.getData() + "-->");
    newline();
  }

  private void serText(Text t, int level, boolean noIndent)
    throws java.io.IOException {
    if (!noIndent) {
      indent(level);
    }
    this.writer.write(escape(t.getNodeValue(), false));
    if (!noIndent) {
      newline();
    }
  }

  /**
   * Escape characters in a string for inclusion into XML text content. The
   * generated string is guaranteed to only contain characters from the US-ASCII
   * character set.
   *
   * @param s un-escaped string
   * @param escapeQuoteChar whether to escape the quote character or not (for
   *      attribute values). Note that attributes are supposed to be quoted
   *      using double quotes, so apostroph characters are <em>not</em>
   *      replaced.
   * @return escaped string
   * @throws java.io.IOException when quoting not possible
   */
  public static String escape(String s, boolean escapeQuoteChar)
    throws java.io.IOException {

    // make it safe
    if (s == null) {
      return "";
    }

    int len = s.length();
    StringBuffer result = null;

    for (int i = 0; i < len; i++) {
      boolean skipOne = false;

      char c = s.charAt(i);
      String replaceBy = null;

      if (c == '&') {
        replaceBy = "&amp;";
      }
      else if (c == '<') {
        replaceBy = "&lt;";
      }
      // this is required because it might appear in ]]> (where escaping
      // is required as per XML spec, section 2.4
      else if (c == '>') {
        replaceBy = "&gt;";
      }
      else if (c == '"' && escapeQuoteChar) {
        replaceBy = "&quot;";
      }
//      else if (c == ("'").charAt(0))
//        result.append ("&apos;");
      else if (c < 32 && (c != 9 && c != 10 && c != 13)) {
        throw new java.io.IOException("invalid XML character");
      }
      else if (c >= 0xd800 && c <= 0xdbff) {
        // this may be a surrogate pair
        if (i + 1 < len) {
          char c2 = s.charAt(i + 1);
          skipOne = true;

          if (c2 >= 0xdc00 && c2 <= 0xdfff) {
            int uc = ((c - 0xd800) * 0x400) + (c2 - 0xdc00) + 0x10000;
            replaceBy = "&#" + uc + ";";
          }
        }

        if (replaceBy == null) {
          throw new java.io.IOException("invalid Unicode character sequence");
        }
      }
      else if (c > 127) {
        replaceBy = "&#" + ((int)c) + ";";
      }

      // init the string buffer first time we actually need to replace
      if (replaceBy != null && result == null) {
        result = new StringBuffer(len * 2);
        if (i != 0) {
          result.append(s.substring(0, i));
        }
      }

      // if result buffer exists, append to it
      if (result != null) {
        if (replaceBy != null) {
          result.append(replaceBy);
        }
        else {
          result.append(c);
        }
      }

      if (skipOne) {
        i += 1;
      }
    }

    return result == null ? s : result.toString();
  }

  private void serAttributes(NamedNodeMap attrs, Map localDecls, Set knownNSAttributes)
    throws java.io.IOException {

    for (int i = 0; i < attrs.getLength(); i++) {
      Attr a = (Attr)attrs.item(i);
      if (knownNSAttributes != null && knownNSAttributes.contains(a)) {
        continue;
      }
//      String ns = a.getNamespaceURI();
//      String nm = a.getLocalName();
//      String pr = a.getPrefix();
      String displayname = a.getNodeName();
//m_logger.errorT("attr: " + displayname + "=" + a.getNodeValue() + "   ns: " + ns);
      this.writer.write(" " + displayname + "=\042" + escape(a.getNodeValue(), true) + "\042");
    }

    // add local declarations if needed

    if (localDecls != null) {
      //log.errorT("attr: got locals, nsaware is " + this.m_nsProcessing);
      for (Iterator it = localDecls.keySet().iterator(); it.hasNext(); ) {
        String prefix = (String)it.next();
        String value = (String)localDecls.get(prefix);
        if (value == null) {
          value = "";
        }

        if (prefix == null || prefix.equals("")) {
          this.writer.write(" xmlns=\042" + escape(value, true) + "\042");
        }
        else {
          this.writer.write(" xmlns:" + prefix + "=\042" + escape(value, true) + "\042");
        }
      }
    }
  }


  /**
   * Serialize a node and it's children.
   *
   * @param n start node
   * @exception java.io.IOException Exception raised in failure situation
   */
  public void serialize(Node n)
    throws java.io.IOException {
    ser(n, this.nesting - 1, this.map);
    this.writer.flush();
  }
}
