/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.xml;

import com.sap.tc.logging.Location;

import java.io.*;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;

/**
 * Simple DOM serializer. Only supports UTF-8 encoding and simple indentation
 * (whitespace preservation not implemented - if whitespace preservation is
 * required simply do not indent). <p>
 *
 * Copyright (c) SAP AG 2003
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Id: SAXSerializer.java,v 1.3 2003/03/31 09:15:35 sei Exp $
 */
public final class SAXSerializer implements ContentHandler {

  private final static Location log = Location.getLocation(com.sapportals.wcm.util.xml.SAXSerializer.class);

  /**
   * Namespace name for the namespace hardwired to the prefix "xml:"
   */
  public final static String NS_XML = "http://www.w3.org/XML/1998/namespace";

  /**
   * Namespace name for the namespace hardwired to the prefix "xmlns:"
   */
  public final static String NS_XMLNS = "http://www.w3.org/2000/xmlns/";


  public static SAXSerializer getInstance(Writer writer, boolean indent) {
    return new SAXSerializer(writer, indent ? "  " : null);
  }

  public static SAXSerializer getInstance(OutputStream out, String encoding, boolean indent)
    throws UnsupportedEncodingException {
    Writer writer = new BufferedWriter(new OutputStreamWriter(out, encoding));
    return getInstance(writer, indent);
  }

  /**
   * Holds linked list of namespace maps.
   */
  private static class NamespaceMap extends HashMap {

    public final static NamespaceMap DEFAULT;
    static {
      DEFAULT = new NamespaceMap(null);
      DEFAULT.put("", "");
      DEFAULT.put("xml", NS_XML);
      DEFAULT.put("xmlns", NS_XMLNS);
    }

    private final NamespaceMap parent;
    private Map map;

    /**
     * Construct a new namespace map based on an existing one.
     *
     * @param parent Base map.
     */
    public NamespaceMap(NamespaceMap parent) {
      this.parent = parent;
      this.map = Collections.EMPTY_MAP;
    }

    public NamespaceMap getParent() {
      return this.parent;
    }

    /**
     * Resolve namespace prefix.
     *
     * @param prefix namespace prefix to be resolved
     * @return namespace name or <code>null</code> when not mapped.
     */
    public String get(String prefix) {

      String result = (String)this.map.get(prefix);

      if (result != null) {
        return result;
      }
      else if (this.parent == null) {
        return null;
      }
      else {
        return this.parent.get(prefix);
      }
    }

    public boolean containsMapping(String prefix, String namespace) {
      String existing = (String)this.map.get(prefix);
      if (existing != null) {
        return existing.equals(namespace);
      }
      else if (this.parent != null) {
        return this.parent.containsMapping(prefix, namespace);
      }
      else {
        return false;
      }
    }

    public String put(String prefix, String namespace) {
      if (this.map.isEmpty()) {
        this.map = new HashMap();
      }
      return (String)this.map.put(prefix, namespace);
    }

  }

  /**
   * TBD: Description of the class.
   */
  private static class SAXWriter {

    public static SAXWriter getInstance(Writer writer) {
      return new SAXWriter(writer);
    }

    private final Writer writer;

    private SAXWriter(Writer writer) {
      this.writer = writer;
    }

    public void write(String s)
      throws SAXException {
      try {
        this.writer.write(s);
      }
      catch (IOException ex) {
        throw new SAXException(ex);
      }
    }

    public void write(char c)
      throws SAXException {
      try {
        this.writer.write(c);
      }
      catch (IOException ex) {
        throw new SAXException(ex);
      }
    }

    public void write(int i)
      throws SAXException {
      try {
        this.writer.write(i);
      }
      catch (IOException ex) {
        throw new SAXException(ex);
      }
    }

    public void flush()
      throws SAXException {
      try {
        this.writer.flush();
      }
      catch (IOException ex) {
        throw new SAXException(ex);
      }
    }
  }

  private final SAXWriter writer;
  private final String indent;
  private final boolean nsProcessing = true;
  private NamespaceMap map;
  private int level;
  private boolean elementOpen;
  private boolean wasText;

  // private String openElement;

  private SAXSerializer() {
    // not to be called
    this.writer = null;
    this.indent = null;
  }

  private SAXSerializer(Writer writer, String indent) {
    this(writer, indent, NamespaceMap.DEFAULT);
  }

  /**
   * Construct a new serializer with a specific namespace mapping as base map
   * (used when creating serializers for child nodes when streaming)
   *
   * @param writer output
   * @param map initial namespace map
   * @param indent TBD: Description of the incoming method parameter
   */
  private SAXSerializer(Writer writer, String indent, NamespaceMap map) {
    this.writer = SAXWriter.getInstance(writer);
    this.indent = indent;
    this.map = map;
    this.level = -1;
    this.elementOpen = false;
  }

  //------------------------ ContentHandler ----------------------------------------

  public void startPrefixMapping(String prefix, String namespace)
    throws SAXException {
    if (log.beDebug()) {
      log.debugT("startPrefixMapping(235)", "start prefix mapping " + prefix + " -> " + namespace);
    }
  }

  public void endPrefixMapping(String prefix)
    throws SAXException {
    if (log.beDebug()) {
      log.debugT("endPrefixMapping(242)", "end prefix mapping " + prefix);
    }
  }

  public void startDocument()
    throws SAXException { }

  public void endDocument()
    throws SAXException {
    this.writer.flush();
  }

  public void startElement(String namespace, String localname, String qname, Attributes attributes)
    throws SAXException {
    if (this.elementOpen) {
      closeElement(false);
      newline();
    }

    this.map = new NamespaceMap(this.map);
    HashMap localDecls = null;

    if (this.nsProcessing) {
      // start by adding all explicit namespace declarations to the map

      for (int i = 0, n = attributes.getLength(); i < n; ++i) {
        String prefix = getPrefixFromNsDecl(attributes, i);
        if (prefix != null) {
          String nsUri = attributes.getValue(i);
          if (!this.map.containsMapping(prefix, nsUri)) {
//m_logger.errorT(level + " adding NS declaration " + prefix + "->" + nsUri);
            this.map.put(prefix, nsUri);
          }
        }
      }

      // check namespace of this element
      if (namespace == null) {
        namespace = "";
      }
      String pr = getPrefix(qname);

      if (!strequals(namespace, map.get(pr))) {
//m_logger.errorT(level + " need NS decl for this element: " + pr + "->" + ns);

        if (localDecls == null) {
          localDecls = new HashMap();
        }
        localDecls.put(pr, namespace);
        this.map.put(pr, namespace);
      }

      // check namespaces of attributes
      for (int i = 0, n = attributes.getLength(); i < n; ++i) {
        //m_logger.errorT(level + " attr name: " + a.getName() + ", prefix: " + a.getPrefix() +
        //  ", namespace: " + a.getNamespaceURI() +
        //  ", localname: " + a.getLocalName() +
        //  ", value: " + a.getNodeValue()
        //  );

        String ans = attributes.getURI(i);
        if (ans == null) {
          ans = "";
        }
        String apr = getPrefix(attributes.getQName(i));
        String inherited = this.map.get(apr);

        if (!ans.equals("") && // attribute is in no namespace
        !strequals(ans, inherited) && // decl already in scope
        !strequals(ans, NS_XMLNS)) {// and not a NS decl...

          //m_logger.errorT(level + " need NS decl for this attribute: " + apr + "->" + ans);

          //        if (inherited != null)
          //        {
          // the prefix is already mapped -- we would need to do namespace
          // rewriting, which we don't want to
          //          throw new java.io.IOException ("SimpleSerializer doesn't support NS prefix rewriting.");
          //        }

          if (localDecls == null) {
            localDecls = new HashMap();
          }
          localDecls.put(apr, ans);
          map.put(apr, ans);
        }
      }
    }

    String displayname = getQName(namespace, localname, qname);

    ++this.level;
    indent();
    this.writer.write("<");
    this.writer.write(displayname);
    serAttributes(attributes, localDecls);
    this.elementOpen = true;
  }

  public void endElement(String namespace, String localname, String qname)
    throws SAXException {
    if (this.elementOpen) {
      closeElement(true);
    }
    else {
      if (!this.wasText) {
        newline();
        indent();
      }
      String displayname = getQName(namespace, localname, qname);
      this.writer.write("</");
      this.writer.write(displayname);
      this.writer.write(">");
    }
    this.wasText = false;
    --this.level;
    this.map = this.map.getParent();
  }

  public void characters(char[] chars, int offset, int length)
    throws SAXException {
    if (this.elementOpen) {
      closeElement(false);
    }
    this.wasText = true;
    writeChars(chars, offset, length, false);
  }

  public void ignorableWhitespace(char[] chars, int i, int i1)
    throws SAXException { }

  public void processingInstruction(String target, String data)
    throws SAXException {
    if (this.elementOpen) {
      closeElement(false);
    }
    indent();
    this.writer.write("<?");
    this.writer.write(target);
    this.writer.write(" ");
    this.writer.write(data);
    this.writer.write("?>");
    this.wasText = false;
    newline();
  }

  public void setDocumentLocator(Locator locator) { }

  public void skippedEntity(String s)
    throws SAXException { }

  //------------------------ private -----------------------------------------------

  private void indent()
    throws SAXException {
    if (this.indent != null) {
      for (int i = 0; i < this.level; i++) {
        this.writer.write(this.indent);
      }
    }
  }

  private void newline()
    throws SAXException {
    if (this.indent != null) {
      this.writer.write("\n");
    }
  }

  private String getPrefixFromNsDecl(Attributes attributes, int index) {
    String ns = attributes.getURI(index);
    if (ns == null) {
      return null;
    }

    if (NS_XMLNS.equals(ns) || "".equals(ns)) {
      String qname = attributes.getQName(index);
      if (qname.startsWith("xmlns:")) {
        return qname.substring("xmlns:".length());
      }
      else if (qname.equals("xmlns")) {
        return "";
      }
      else {
        return null;
      }
    }

    return null;
  }

  private boolean strequals(String a, String b) {
    if (a == null) {
      return a == b;
    }
    else {
      return a.equals(b);
    }
  }

  private void serAttributes(Attributes attrs, HashMap localDecls)
    throws SAXException {

    for (int i = 0, n = attrs.getLength(); i < n; i++) {

//      String ns = a.getNamespaceURI();
//      String nm = a.getLocalName();
//      String pr = a.getPrefix();
      String qname = attrs.getQName(i);
//m_logger.errorT("attr: " + displayname + "=" + a.getNodeValue() + "   ns: " + ns);
      this.writer.write(" ");
      this.writer.write(qname);
      this.writer.write("=\042");
      this.writer.write(escape(attrs.getValue(i), true));
      this.writer.write("\042");
    }

    // add local declarations if needed

    if (localDecls != null) {
      //log.errorT("attr: got locals, nsaware is " + this.m_nsProcessing);
      java.util.Iterator it = localDecls.keySet().iterator();

      while (it.hasNext()) {
        String prefix = (String)it.next();
        String value = (String)localDecls.get(prefix);
        if (value == null) {
          value = "";
        }

        if (prefix == null || prefix.equals("")) {
          this.writer.write(" xmlns=\042");
          this.writer.write(escape(value, true));
          this.writer.write("\042");
        }
        else {
          this.writer.write(" xmlns:");
          this.writer.write(prefix);
          this.writer.write("=\042");
          this.writer.write(escape(value, true));
          this.writer.write("\042");
        }
      }
    }
  }

  private void closeElement(boolean empty)
    throws SAXException {
    if (empty) {
      this.writer.write("/>");
    }
    else {
      this.writer.write(">");
    }
    this.elementOpen = false;
  }

  private String getQName(String namespace, String localname, String qname) {
    return (qname != null) ? qname : localname;
  }

  private String getPrefix(String qname) {
    if (qname != null) {
      int index = qname.indexOf(':');
      if (index >= 0) {
        return qname.substring(0, index);
      }
    }
    return "";
  }

  private void writeChars(char[] buffer, int offset, int length, boolean escapeQuoteChar)
    throws SAXException {
    // make it safe
    if (buffer == null || length <= 0) {
      return;
    }

    int max = offset + length;
    for (int i = offset; i < max; i++) {
      boolean skipOne = false;
      char c = buffer[i];

      if (c == '&') {
        this.writer.write("&amp;");
      }
      else if (c == '<') {
        this.writer.write("&lt;");
      }
      // this is required because it might appear in ]]> (where escaping
      // is required as per XML spec, section 2.4
      else if (c == '>') {
        this.writer.write("&gt;");
      }
      else if (c == '"' && escapeQuoteChar) {
        this.writer.write("&quot;");
      }
//      else if (c == ("'").charAt(0))
//        result.append ("&apos;");
      else if (c < 32 && (c != 9 && c != 10 && c != 13)) {
        throw new SAXException("invalid XML character");
      }
      else if (c >= 0xd800 && c <= 0xdbff) {
        // this may be a surrogate pair
        if (i + 1 < max) {
          char c2 = buffer[i + 1];
          skipOne = true;

          if (c2 >= 0xdc00 && c2 <= 0xdfff) {
            int uc = (((int)c - 0xd800) * 0x400) + ((int)c2 - 0xdc00) + 0x10000;
            this.writer.write("&#");
            this.writer.write(uc);
            this.writer.write(';');
          }
        }

        throw new SAXException("invalid Unicode character sequence");
      }
      else if (c > 127) {
        this.writer.write("&#");
        this.writer.write((int)c);
        this.writer.write(';');
      }
      else {
        this.writer.write(c);
      }

      if (skipOne) {
        i += 1;
      }
    }
  }

  /**
   * Escape characters in a string for inclusion into XML text content. The
   * generated string is guaranteed to only contain characters from the US-ASCII
   * character set.
   *
   * @param s un-escaped string
   * @param escapeQuoteChar whether to escape the quote character or not (for
   *      attribute values). Note that attributes are supposed to be quoted
   *      using double quotes, so apostroph characters are <em>not</em>
   *      replaced.
   * @return escaped string
   * @exception SAXException Exception raised in failure situation
   */
  public static String escape(String s, boolean escapeQuoteChar)
    throws SAXException {

    // make it safe
    if (s == null) {
      return "";
    }

    int len = s.length();
    StringBuffer result = null;

    for (int i = 0; i < len; i++) {
      boolean skipOne = false;

      char c = s.charAt(i);
      String replaceBy = null;

      if (c == '&') {
        replaceBy = "&amp;";
      }
      else if (c == '<') {
        replaceBy = "&lt;";
      }
      // this is required because it might appear in ]]> (where escaping
      // is required as per XML spec, section 2.4
      else if (c == '>') {
        replaceBy = "&gt;";
      }
      else if (c == '"' && escapeQuoteChar) {
        replaceBy = "&quot;";
      }
//      else if (c == ("'").charAt(0))
//        result.append ("&apos;");
      else if (c < 32 && (c != 9 && c != 10 && c != 13)) {
        throw new SAXException("invalid XML character");
      }
      else if (c >= 0xd800 && c <= 0xdbff) {
        // this may be a surrogate pair
        if (i + 1 < len) {
          char c2 = s.charAt(i + 1);
          skipOne = true;

          if (c2 >= 0xdc00 && c2 <= 0xdfff) {
            int uc = (((int)c - 0xd800) * 0x400) + ((int)c2 - 0xdc00) + 0x10000;
            replaceBy = "&#" + uc + ";";
          }
        }

        if (replaceBy == null) {
          throw new SAXException("invalid Unicode character sequence");
        }
      }
      else if (c > 127) {
        replaceBy = "&#" + ((int)c) + ";";
      }

      // init the string buffer first time we actually need to replace
      if (replaceBy != null && result == null) {
        result = new StringBuffer(len * 2);
        if (i != 0) {
          result.append(s.substring(0, i));
        }
      }

      // if result buffer exists, append to it
      if (result != null) {
        if (replaceBy != null) {
          result.append(replaceBy);
        }
        else {
          result.append(c);
        }
      }

      if (skipOne) {
        i += 1;
      }
    }

    return result == null ? s : result.toString();
  }

}
