/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.xml;

import java.io.*;
import javax.xml.parsers.*;

import org.w3c.dom.*;
import org.xml.sax.*;

/**
 * <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author julian.reschke@greenbytes.de
 * @version $Id: DomBuilder.java,v 1.1 2002/07/16 16:58:38 jre Exp $
 */

public class DomBuilder {

  private final static com.sap.tc.logging.Location log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.xml.DomBuilder.class);

  final static DocumentBuilderFactory Factory = javax.xml.parsers.DocumentBuilderFactory.newInstance();

  final static EntityResolver noEntityResolver = new NoEntityResolver();
  final static ErrorHandler strictErrorHandler = new StrictErrorHandler();


  public static Document newDocument() {
    return getBuilder(true).newDocument();
  }

  public static Document parse(String input, boolean namespaceAware)
    throws org.xml.sax.SAXException, java.io.IOException {
    return parse(new InputSource(new StringReader(input)), namespaceAware);
  }

  public static Document parse(InputSource is, boolean namespaceAware)
    throws org.xml.sax.SAXException {
    try {
      DocumentBuilder builder = getBuilder(namespaceAware);
      return builder.parse(is);
    }
    catch (IOException ex) {
      log.debugT("parse(59)", "while parsing document" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(ex));
      throw new org.xml.sax.SAXException(ex);
    }
  }

  /**
   * Parse a string into a Document
   *
   * @param input the input stream
   * @return the Document
   * @exception org.xml.sax.SAXException Exception raised in failure situation
   * @exception java.io.IOException Exception raised in failure situation
   */

  public static Document parse(String input)
    throws org.xml.sax.SAXException, java.io.IOException {
    return parse(new InputSource(new StringReader(input)), true);
  }


  /**
   * Parse an input stream into a Document
   *
   * @param input the input stream
   * @param tolerateNSerrors when set to <code>true</code> , elements that
   *      aren't XML-NS-wellformed but which <i>are</i> XML-wellformed will be
   *      tolerated (and dropped)
   * @return the Document
   * @exception org.xml.sax.SAXException Exception raised in failure situation
   * @exception IOException Exception raised in failure situation
   */
  public static Document parse(InputStream input, boolean tolerateNSerrors)
    throws org.xml.sax.SAXException, IOException {
    Document doc = parse(new InputSource(input), !tolerateNSerrors);

    if (tolerateNSerrors) {
      // fixup NS errors, serialize and run it through a NS-aware parser
      fixupNsErrors(doc.getDocumentElement());
      if (log.beInfo()) {
        log.infoT("parse(98)", SimpleSerializer.toString(doc, true, false));
      }
      return parse(SimpleSerializer.toString(doc, false, false));
    }

    // HACK
    try {
      doc.getDocumentElement().getNamespaceURI();
    }
    catch (java.lang.NoSuchMethodError ex) {
      log.fatalT("parse(108)", "-------------------------------------------------------");
      log.fatalT("parse(109)", "Couldn't invoke: getDocumentElement().getNamespaceURI()");
      log.fatalT("parse(110)", "This is likely caused by outdated JAR files in the");
      log.fatalT("parse(111)", "CLASSPATH preceding the DOM-Level-2 that is needed.");
      log.fatalT("parse(112)", "Current DOM impl is: " + doc.getClass().getName());
      log.fatalT("parse(113)", "-------------------------------------------------------");
      throw ex;
    }

    return doc;
  }

  private static DocumentBuilder getBuilder(boolean namespaceAware) {

    try {
      DocumentBuilder builder = null;
      synchronized (Factory) {
        Factory.setNamespaceAware(namespaceAware);
        builder = Factory.newDocumentBuilder();
      }
      builder.setErrorHandler(strictErrorHandler);
      builder.setEntityResolver(noEntityResolver);
      return builder;
    }
    catch (javax.xml.parsers.ParserConfigurationException ex) {
            //$JL-EXC$      
      log.fatalT("getBuilder(133)", "creating DocumentBuilder" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(ex));
      return null;
    }

  }

  private static boolean fixupNsErrors(Node n) {
    // log.errorT("DB: " + n.getNodeName());

    if (n.getNodeType() != Node.ELEMENT_NODE) {
      return false;
    }

    if (!isNsWellformedElementName(n.getNodeName())) {
      Element ne = nsFixupElement((Element)n);
      if (ne != null) {
        n.getParentNode().replaceChild(ne, n);
        return false;
      }
      else {
        n.getParentNode().removeChild(n);
        return true;
      }
//      return false;
//      n.getParentNode().replaceChild (
//        n.getOwnerDocument().createComment ("Removed malformed XML: "
//          + DomSerializer.toString ((Element) n, false)),
//        n);
//      return false;
//      n.getParentNode().removeChild (n);
//      return true;
    }
    else {
      NodeList nl = n.getChildNodes();

      int i = 0;
      while (i < nl.getLength()) {
        // log.errorT("childs of " + n.getNodeName() + ": " + c.getNodeName());
        if (!fixupNsErrors(nl.item(i))) {
          ++i;
        }
      }
      return false;
    }
  }

  private static boolean isNsWellformedElementName(String name) {
    int colon = name.indexOf(":");
    char c = colon < 0 ? name.charAt(0) : name.charAt(colon + 1);
    return !Character.isDigit(c);
  }


  // try to fixup Exchange property names by removing a leading digit

  private static Element nsFixupElement(Element e) {

    // assumes name has format <prefix>:<digit><remainder>

    String oldname = e.getNodeName();
    int pos = oldname.indexOf(":");
    if (pos < 0) {
      return null;
    }

    String newName = e.getNodeName().substring(0, pos + 1) +
      e.getNodeName().substring(pos + 2);

    if (!isNsWellformedElementName(newName)) {
      return null;
    }

    // log.errorT("new element name: " + newName);

    Element ne = e.getOwnerDocument().createElement(newName);

    NamedNodeMap attrs = e.getAttributes();

    for (int i = 0; i < attrs.getLength(); i++) {
      ne.setAttributeNode((Attr)(attrs.item(i)).cloneNode(false));
    }

    for (Node c = e.getFirstChild(); c != null; c = c.getNextSibling()) {
      ne.appendChild(c);
    }

    return ne;
  }
}
