/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.usermanagement;
import com.sap.tc.logging.Location;

import com.sapportals.portal.security.usermanagement.*;
import com.sapportals.portal.security.usermanagement.*;
import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import java.util.*;

/**
 * SelectionUserManager <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author andreas.heix@sapportals.com
 * @created 18. Dezember 2001
 * @version $Id:$
 */

public final class SelectionUserManager {

  //flag if emails are allowed as adress
  boolean allowEmailAdress = false;

  //flag if uncheckedNames are allowed
  boolean allowUncheckedNames = false;

  //map to keep users, groups, emailadresses
  HashMap mapRecipients;

  //flag to see, if any recipients were deleted or not
  private boolean deletedRecipients = false;

//-----------------Added Functionality for Testing ---------------------/

  private String newRecipients = null;
  private boolean newRecipientsGiven = false;
  private String invalidIDs = null;
  private boolean invalidIDsGiven = false;
  private HashMap nonUniqueIDs = null;
  private boolean nonUniqueIDsGiven = false;

  //type constants, to distinguish between users, groups, e-mail...

  /**
   * Constant to identify a selected Object's Type (User, Group, Role: from
   * IUMPrincipal) or (email, unchecked)
   */
  public final static int TYPE_USER = IUMPrincipal.IUSER;

  /**
   * Constant to identify a selected Object's Type (User, Group, Role: from
   * IUMPrincipal) or (email, unchecked)
   */
  public final static int TYPE_ROLE = IUMPrincipal.IROLE;

  /**
   * Constant to identify a selected Object's Type (User, Group, Role: from
   * IUMPrincipal) or (email, unchecked)
   */
  public final static int TYPE_GROUP = IUMPrincipal.IGROUP;

  /**
   * Constant to identify a selected Object's Type (User, Group, Role: from
   * IUMPrincipal) or (email, unchecked)
   */
  public final static int TYPE_EMAIL = 100;

  /**
   * Constant to identify a selected Object's Type (User, Group, Role: from
   * IUMPrincipal) or (email, unchecked)
   */
  public final static int TYPE_UNCHECKED = 200;

  /**
   * this is a tag to separate entries in the string export of the users, groups
   * etc.
   */
  public final static String TAG_SEPARATOR = "";
  /**
   * tag to separate entries in the string export of the users, groups etc.
   */
  public final static String TAG_FINALIZER = "";
  /**
   * opening symbol for a selected group.
   */
  public final static String DISPLAY_GROUP_OPEN = "(";
  /**
   * closing symbol for a selected group.
   */
  public final static String DISPLAY_GROUP_CLOSE = ")";
  /**
   * opening symbol for a selected role.
   */
  public final static String DISPLAY_ROLE_OPEN = "[";
  /**
   * closing symbol for a selected role.
   */
  public final static String DISPLAY_ROLE_CLOSE = "]";
  /**
   * closing symbol for an unchecked user.
   */
  public final static String DISPLAY_UNCHECKED_CLOSE = "\"";
  /**
   * opening symbol for an unchecked user.
   */
  public final static String DISPLAY_UNCHECKED_OPEN = "\"";

  /**
   * tag to separate entries in the string export of the users, groups etc.
   */
  public final static String SEPARATOR = ";";


  /**
   * Constructor for the SelectionUserManager object
   *
   * @param allowEmails email-adresses (recognized by the "@"-symbol) are
   *      allowed as valid selection
   */
  public SelectionUserManager(boolean allowEmails) {
    doInitialize(allowEmails, false);
  }


  /**
   * Constructor for the SelectionUserManager object
   *
   * @param allowEmails email-adresses (recognized by the "@"-symbol) are
   *      allowed as valid selection
   * @param allowUncheckedNames names, that are not known by the usermanagement
   *      can be used as valid selection
   */
  public SelectionUserManager(boolean allowEmails, boolean allowUncheckedNames) {
    doInitialize(allowEmails, allowUncheckedNames);
  }


  /**
   * By default, unchecked Names and email-adresses are not allowed. With this
   * setting, only users, groups and roles of the usermanagement can be added to
   * the list of selected users
   */
  public SelectionUserManager() {
    doInitialize(false, false);
  }


  /**
   * fill a collection with ids, create display names
   *
   * @param ids StringArray of User-IDs
   * @param typeOfList value to distinguish between groups, roles, users...
   * @exception WcmException is thrown when problems in UserManagement occur
   */
  public void setList(String[] ids, int typeOfList)
    throws WcmException {
    for (int i = 0; i < ids.length; i++) {
      this.setList(ids[i], typeOfList);
    }
  }


  /**
   * fill a collection with id, create display name
   *
   * @param id User-ID
   * @param typeOfList value to distinguish between groups, roles, users...
   * @exception WcmException is thrown when problems in UserManagement occur
   */
  public void setList(String id, int typeOfList)
    throws WcmException {
    if ((!this.allowEmailAdress) && (typeOfList == this.TYPE_EMAIL)) {
      throw new WcmException("this SelectionUserManager does not support email-adresses");
    }
    if ((!this.allowUncheckedNames) && (typeOfList == this.TYPE_UNCHECKED)) {
      throw new WcmException("this SelectionUserManager does not support unchecked adresses");
    }
    Collection temp = (Collection)this.mapRecipients.get(new Integer(typeOfList));
    this.addWithoutDoubles(temp, new SelectionUser(id, typeOfList));
    //temp.add(new SelectionUser(id, typeOfList));

    //assign temp-Collection to fitting collection
    this.mapRecipients.put(new Integer(typeOfList), temp);
  }


  /**
   * fill a collection with id and displayName
   *
   * @param id User-ID
   * @param displayName DisplayName of the given User
   * @param typeOfList value to distinguish between groups, roles, users...
   * @exception WcmException is thrown when problems in UserManagement occur
   */
  public void setList(String id, String displayName, int typeOfList)
    throws WcmException {
    if ((!this.allowEmailAdress) && (typeOfList == this.TYPE_EMAIL)) {
      throw new WcmException("this SelectionUserManager does not support email-adresses");
    }
    if ((!this.allowUncheckedNames) && (typeOfList == this.TYPE_UNCHECKED)) {
      throw new WcmException("this SelectionUserManager does not support unchecked adresses");
    }

    Collection temp = (Collection)this.mapRecipients.get(new Integer(typeOfList));
    this.addWithoutDoubles(temp, new SelectionUser(id, displayName, typeOfList));

    //assign temp-Collection to fitting collection
    this.mapRecipients.put(new Integer(typeOfList), temp);
  }


  /**
   * @param typeOfList value to distinguish between groups, roles, users...
   * @return a Collection of SelectionUsers or NULL
   */
  public Collection getSelectionUsers(int typeOfList) {
    return (Collection)this.mapRecipients.get(new Integer(typeOfList));
  }


  /**
   * @param typeOfList value to distinguish between groups, roles, users...
   * @return an Array with the IDs of the list or NULL
   */
  public String[] getIDs(int typeOfList) {
    Collection temp = (Collection)this.mapRecipients.get(new Integer(typeOfList));
    if (temp == null) {
      return null;
    }

    String[] idList = new String[temp.size()];
    Iterator iter = temp.iterator();
    int i = 0;
    while (iter.hasNext()) {
      idList[i] = ((SelectionUser)iter.next()).getId();
      i++;
    }
    return idList;
  }


  /**
   * @return String to persist in Control, separated as described in
   *      initWithString
   */
  public String getSelectionList() {
    return this.collectionToString(this.getSelectionUsers(this.TYPE_USER))
       + this.collectionToString(this.getSelectionUsers(this.TYPE_GROUP))
       + this.collectionToString(this.getSelectionUsers(this.TYPE_ROLE))
       + this.collectionToString(this.getSelectionUsers(this.TYPE_EMAIL))
       + this.collectionToString(this.getSelectionUsers(this.TYPE_UNCHECKED));
  }


  /**
   * @param userList a List of user-ids, display-names etc. (Usually submitted
   *      from a user-Input)
   * @return a HashMap with three String[] Lists of nonUnique Recipients or
   *      NULL, if all is ok
   */
  public HashMap getNonUniqueIDs(String userList) {
    StringTokenizer token = new StringTokenizer(userList, this.SEPARATOR);
    boolean nonUniqueIDsExist = false;
    Collection nonUniqueUsers = new ArrayList();
    Collection nonUniqueGroups = new ArrayList();
    Collection nonUniqueRoles = new ArrayList();
    String user = new String();
    String[] groupResult;
    String[] userResult;
    String[] roleResult;

    while (token.hasMoreElements()) {
      user = token.nextToken().trim();

      String searchUser = prepareUserForSearch(this.getIntUserPresentation(user));

      //user typed in a group
      if (isGroup(user)) {
        groupResult = WPUMFactory.simpleGroupSearch(searchUser);
        userResult = new String[0];
        roleResult = new String[0];
      }
      //user typed in a role
      else if (isRole(user)) {
        roleResult = WPUMFactory.simpleRoleSearch(searchUser);
        userResult = new String[0];
        groupResult = new String[0];
      }
      else {
        userResult = WPUMFactory.simpleUserSearch(searchUser);
        groupResult = WPUMFactory.simpleGroupSearch(searchUser);
        roleResult = WPUMFactory.simpleRoleSearch(searchUser);
      }

      if (userResult == null) {
        userResult = new String[0];
      }
      if (groupResult == null) {
        groupResult = new String[0];
      }
      if (roleResult == null) {
        roleResult = new String[0];
      }
      //non unique --if there is a result list in more then one type(user,group,role)
      //or if there is a result list > 1
      if ((userResult.length > 1) || (groupResult.length > 1)
         || (roleResult.length > 1)
         || ((userResult.length > 0) && (groupResult.length > 0))
         || ((userResult.length > 0) && (roleResult.length > 0))
         || ((groupResult.length > 0) && (roleResult.length > 0))) {
        nonUniqueIDsExist = true;
        if (userResult.length > 0) {
          nonUniqueUsers.addAll(this.StringArray2Collection(userResult));
        }
        if (groupResult.length > 0) {
          nonUniqueGroups.addAll(this.StringArray2Collection(groupResult));
        }
        if (roleResult.length > 0) {
          nonUniqueRoles.addAll(this.StringArray2Collection(roleResult));
        }
      }

    }
    if (!nonUniqueIDsExist) {
      return null;
      // no non-unique IDs in list
    }
    // return the non-unique IDs via a HashMap
    HashMap returnMap = new HashMap(3);
    returnMap.put(new Integer(this.TYPE_USER).toString(), nonUniqueUsers);
    returnMap.put(new Integer(this.TYPE_GROUP).toString(), nonUniqueGroups);
    returnMap.put(new Integer(this.TYPE_ROLE).toString(), nonUniqueRoles);
    return returnMap;
  }


  /**
   * @param userList a List of user-ids, display-names etc. (Usually submitted
   *      from a user-Input)
   * @return a HashMap with three String[] Lists of nonUnique Recipients or
   *      NULL, if all is ok
   */
  public HashMap getUniqueAndValidIDs(String userList) {
    StringTokenizer token = new StringTokenizer(userList, this.SEPARATOR);
    boolean uniqueIDsExist = false;
    Collection uniqueUsers = new ArrayList();
    Collection uniqueGroups = new ArrayList();
    Collection uniqueRoles = new ArrayList();
    String user = new String();
    String[] groupResult;
    String[] userResult;
    String[] roleResult;

    while (token.hasMoreElements()) {
      user = token.nextToken().trim();

      String searchUser = prepareUserForSearch(this.getIntUserPresentation(user));

      //user typed in a group
      if (isGroup(user)) {
        groupResult = WPUMFactory.simpleGroupSearch(searchUser);
        userResult = new String[0];
        roleResult = new String[0];
      }
      //user typed in a role
      else if (isRole(user)) {
        roleResult = WPUMFactory.simpleRoleSearch(searchUser);
        userResult = new String[0];
        groupResult = new String[0];
      }
      else {
        userResult = WPUMFactory.simpleUserSearch(searchUser);
        groupResult = WPUMFactory.simpleGroupSearch(searchUser);
        roleResult = WPUMFactory.simpleRoleSearch(searchUser);
      }

      if (userResult == null) {
        userResult = new String[0];
      }
      if (groupResult == null) {
        groupResult = new String[0];
      }
      if (roleResult == null) {
        roleResult = new String[0];
      }
      //non unique --if there is a result list in more then one type(user,group,role)
      //or if there is a result list > 1
      if ((userResult.length > 1) || (groupResult.length > 1)
         || (roleResult.length > 1)
         || ((userResult.length > 0) && (groupResult.length > 0))
         || ((userResult.length > 0) && (roleResult.length > 0))
         || ((groupResult.length > 0) && (roleResult.length > 0))) {
        continue;
      }
      else {
        uniqueIDsExist = true;
        if (userResult.length > 0) {
          uniqueUsers.addAll(this.StringArray2Collection(userResult));
        }
        if (groupResult.length > 0) {
          uniqueGroups.addAll(this.StringArray2Collection(groupResult));
        }
        if (roleResult.length > 0) {
          uniqueRoles.addAll(this.StringArray2Collection(roleResult));
        }
      }

    }
    if (uniqueIDsExist == false) {
      return null;
      // no unique IDs in list
    }
    // return the unique IDs via a HashMap
    HashMap returnMap = new HashMap(3);
    returnMap.put(new Integer(this.TYPE_USER).toString(), uniqueUsers);
    returnMap.put(new Integer(this.TYPE_GROUP).toString(), uniqueGroups);
    returnMap.put(new Integer(this.TYPE_ROLE).toString(), uniqueRoles);
    return returnMap;
  }


  /**
   * Gets the displayString attribute of the SelectionUserManager object
   *
   * @return The displayString value
   */
  public String getDisplayString() {
    String temp = this.collectionToDisplayNames(this.getSelectionUsers(this.TYPE_USER))
       + this.collectionToDisplayNames(this.getSelectionUsers(this.TYPE_GROUP))
       + this.collectionToDisplayNames(this.getSelectionUsers(this.TYPE_ROLE))
       + this.collectionToDisplayNames(this.getSelectionUsers(this.TYPE_EMAIL))
       + this.collectionToDisplayNames(this.getSelectionUsers(this.TYPE_UNCHECKED));
    if (temp.startsWith(this.SEPARATOR)) {
      return temp.substring(2);
    }
    else {
      return temp;
    }
  }


  /**
   * Gets the invalidIDGiven attribute of the SelectionUserManager object
   *
   * @return The invalidIDGiven value
   */
  public boolean isInvalidIDGiven() {
    return this.invalidIDsGiven;
  }


  /**
   * Are non UniqueIDsGiven?
   *
   * @return The nonUniqueIDGiven value
   */
  public boolean isNonUniqueIDGiven() {
    return this.nonUniqueIDsGiven;
  }


  /**
   * Gets the nonUniqueIDs attribute of the SelectionUserManager object
   *
   * @return The nonUniqueIDs value
   */
  public HashMap getNonUniqueIDs() {
    return this.nonUniqueIDs;
  }


  /**
   * Are invalid IDs given?
   *
   * @return The invalidIDs value
   */
  public String getInvalidIDs() {
    return this.invalidIDs;
  }


  /**
   * @param persistentString HtmlB/Wdf-Controls persist the value of User-Input
   *      fields with strings. The SelectionUserManager, which takes care of the
   *      user-selection in controls, offers two methods to initialize itself
   *      with the persistent data from the control (that is this method!) and
   *      to create a String with all information needed, after the user-input
   *      has been submitted to the SelectionUserManager. That is the method
   *      <code>getSelectionList<code>. The String combines User-ID,
   *      descriptionName and Type of User (user, role, group...) in triples and
   *      separates them from the next entry.
   * @return flag, if operation successful or not
   * @exception WcmException Description of Exception
   */
  public boolean initWithString(String persistentString)
    throws WcmException {
    if (persistentString == null) {
      return false;
    }

    StringTokenizer userList = new StringTokenizer(persistentString, this.TAG_FINALIZER);
    StringTokenizer singleUser;
    Collection temp = new ArrayList();
    String userType = new String();
    String userId = new String();
    String userDisplayName = new String();
    boolean worked = false;

    while (userList.hasMoreTokens()) {
      singleUser = new StringTokenizer(userList.nextToken(), this.TAG_SEPARATOR);
      try {
        userType = singleUser.nextToken();
        userId = singleUser.nextToken();
        userDisplayName = singleUser.nextToken();
      }
      catch (Exception ex) {
            //$JL-EXC$        
        continue;
      }
      worked = true;
      this.setList(userId, userDisplayName, new Integer(userType).intValue());
    }

    return worked;
  }


  /**
   * @param userList a list of UserIDs, displaynames etc., ususally coming from
   *      a user-input-field
   * @return a SEPARATOR seperated List of invalid Recipients or NULL, if all is
   *      ok
   */
  public String[] checkForInvalidIDs(String userList) {
    StringTokenizer token = new StringTokenizer(userList, this.SEPARATOR);
    Collection inputIDs = new ArrayList();

    while (token.hasMoreElements()) {
      String user = token.nextToken().trim();

      //if no new input, continue
      if (this.checkForNewInput(user) == null) {
        continue;
      }
      //if it is emailadress and emailadresses are allowed, continue
      if ((this.allowEmailAdress) && (user.indexOf("@") > 0)) {
        continue;
      }
      //if it is an unchecked adress and unchecked adresses are allowed, continue
      if ((this.allowUncheckedNames) && (user.indexOf(this.DISPLAY_UNCHECKED_OPEN) > -1)) {
        continue;
      }

      //now check users, groups and roles, with and withouht the display signs "<>" and "[]"
      String searchUser = prepareUserForSearch(this.getIntUserPresentation(user));
      if (isGroup(user)) {
        String[] userResult = WPUMFactory.simpleGroupSearch(searchUser);
        if ((userResult != null) && (userResult.length > 0)) {
          continue;
        }
        inputIDs.add(user);
        continue;
      }
      if (isRole(user)) {
        String[] userResult = WPUMFactory.simpleRoleSearch(searchUser);
        if ((userResult != null) && (userResult.length > 0)) {
          continue;
        }
        inputIDs.add(user);
        continue;
      }

      String[] userResult = WPUMFactory.simpleUserSearch(searchUser);
      if ((userResult != null) && (userResult.length > 0)) {
        continue;
      }
      userResult = WPUMFactory.simpleGroupSearch(searchUser);
      if ((userResult != null) && (userResult.length > 0)) {
        continue;
      }
      userResult = WPUMFactory.simpleRoleSearch(searchUser);
      if ((userResult != null) && (userResult.length > 0)) {
        continue;
      }
      inputIDs.add(user);
    }

    //check if collection is empty
    if (inputIDs.size() == 0) {
      return null;
    }
    return (String[])inputIDs.toArray(new String[inputIDs.size()]);
  }


  /**
   * @param userList a list of UserIDs, displaynames etc., ususally coming from
   *      a user-input-field
   * @return a SEPARATOR seperated List of nonUnique Recipients or NULL, if all
   *      is ok
   */
  public String[] checkForNonUniqueIDs(String userList) {
    StringTokenizer token = new StringTokenizer(userList, this.SEPARATOR);
    Collection inputIDs = new ArrayList();
    String user = new String();
    String searchUser = new String();

    while (token.hasMoreElements()) {
      user = token.nextToken().trim();
      searchUser = prepareUserForSearch(this.getIntUserPresentation(user));

      //user typed in a group
      if (isGroup(user)) {
        String[] userResult = WPUMFactory.simpleGroupSearch(searchUser);
        if ((userResult == null) && (userResult.length != 1)) {
          inputIDs.add(user);
          continue;
        }
        continue;
      }

      //user typed in a role
      if (isRole(user)) {
        String[] userResult = WPUMFactory.simpleRoleSearch(searchUser);
        if ((userResult == null) && (userResult.length != 1)) {
          inputIDs.add(user);
          continue;
        }
        continue;
      }

      String[] userResult = WPUMFactory.simpleUserSearch(searchUser);
      String[] groupResult = WPUMFactory.simpleGroupSearch(searchUser);
      String[] roleResult = WPUMFactory.simpleRoleSearch(searchUser);

      if (userResult == null) {
        userResult = new String[0];
      }
      if (groupResult == null) {
        groupResult = new String[0];
      }
      if (roleResult == null) {
        roleResult = new String[0];
      }
      //non unique --if there is a result list in more then one type(user,group,role)
      //or if there is a result list > 1
      if ((userResult.length > 1) || (groupResult.length > 1)
         || (roleResult.length > 1)
         || ((userResult.length > 0) && (groupResult.length > 0))
         || ((userResult.length > 0) && (roleResult.length > 0))
         || ((groupResult.length > 0) && (roleResult.length > 0))) {
        inputIDs.add(user);
      }
    }
    if (inputIDs.size() == 0) {
      return null;
      // no non-unique IDs in list
    }
    return (String[])inputIDs.toArray(new String[inputIDs.size()]);
  }


  /**
   * @param userList a list of UserIDs, displaynames etc., ususally coming from
   *      a user-input-field
   * @return a List of all inputs, that are not known to the Manager at this
   *      time. <br>
   * <code>null</code>, if all inputs are known
   */
  public String[] checkForNewInput(String userList) {
    StringTokenizer token = new StringTokenizer(userList, this.SEPARATOR);
    Collection inputIDs = new ArrayList();
    while (token.hasMoreTokens()) {
      inputIDs.add(token.nextToken().trim());
    }

    //check if inputIDs are known in one of the 3 or 4 or 5 Collections
    Collection newInput = new ArrayList(this.searchUserInCollection(inputIDs, this.TYPE_USER));
    newInput = this.searchUserInCollection(newInput, this.TYPE_GROUP);
    newInput = this.searchUserInCollection(newInput, this.TYPE_ROLE);
    if (this.allowEmailAdress) {
      newInput = this.searchUserInCollection(newInput, this.TYPE_EMAIL);
    }
    if (this.allowUncheckedNames) {
      newInput = this.searchUserInCollection(newInput, this.TYPE_UNCHECKED);
    }

    //no new input
    if (newInput.size() == 0) {
      return null;
    }
    return (String[])newInput.toArray(new String[newInput.size()]);
  }


  /**
   * checks, if users who were selected have been deleted in the display
   *
   * @param input Description of Parameter
   */
  public void deleteFromInput(String input) {
    //1. copy collection
    // System.out.println("Time entered deleteFromInput: " + System.currentTimeMillis());
    Collection tempUsers = new ArrayList((Collection)this.mapRecipients.get(new Integer(this.TYPE_USER)));
    Collection tempGroups = new ArrayList((Collection)this.mapRecipients.get(new Integer(this.TYPE_GROUP)));
    Collection tempRoles = new ArrayList((Collection)this.mapRecipients.get(new Integer(this.TYPE_ROLE)));
    Collection tempEmail = new ArrayList();
    Collection tempUnchecked = new ArrayList();
    if (this.allowEmailAdress) {
      tempEmail = new ArrayList((Collection)this.mapRecipients.get(new Integer(this.TYPE_EMAIL)));
    }
    if (this.allowUncheckedNames) {
      tempUnchecked = new ArrayList((Collection)this.mapRecipients.get(new Integer(this.TYPE_UNCHECKED)));
    }

    //2. check if user is in the copied collection and if so, delete from that collection
    StringTokenizer token = new StringTokenizer(input, this.SEPARATOR);
    while (token.hasMoreTokens()) {
      String tokenValue = token.nextToken().trim();
      /*
       * boolean userFound = this.deleteUserInCollection(tempUsers, tokenValue);
       * if (!userFound) {
       * userFound = this.deleteUserInCollection(tempGroups, tokenValue);
       * }
       * if (!userFound) {
       * userFound = this.deleteUserInCollection(tempRoles, tokenValue);
       * }
       * if ((!userFound) && (this.allowEmailAdress)){
       * userFound = this.deleteUserInCollection(tempEmail, tokenValue);
       * }
       */
      boolean userFound = this.deleteUserIfDisplayNameChanged(tempUsers, tokenValue);
      if (!userFound) {
        userFound = this.deleteUserIfDisplayNameChanged(tempGroups, tokenValue);
      }
      if (!userFound) {
        userFound = this.deleteUserIfDisplayNameChanged(tempRoles, tokenValue);
      }
      if ((!userFound) && (this.allowEmailAdress)) {
        userFound = this.deleteUserIfDisplayNameChanged(tempEmail, tokenValue);
      }
      if ((!userFound) && (this.allowUncheckedNames)) {
        userFound = this.deleteUserIfDisplayNameChanged(tempUnchecked, tokenValue);
      }

    }

    //3. remove all objects still in the copied collection from the real collection
    if (tempUsers.size() > 0) {
      this.updateCollectionAfterDelete(tempUsers, this.TYPE_USER);
    }
    if (tempGroups.size() > 0) {
      this.updateCollectionAfterDelete(tempGroups, this.TYPE_GROUP);
    }
    if (tempRoles.size() > 0) {
      this.updateCollectionAfterDelete(tempRoles, this.TYPE_ROLE);
    }
    if (tempEmail.size() > 0) {
      this.updateCollectionAfterDelete(tempEmail, this.TYPE_EMAIL);
    }
    if (tempUnchecked.size() > 0) {
      this.updateCollectionAfterDelete(tempUnchecked, this.TYPE_UNCHECKED);
    }

  }


  /**
   * @param userList a list of UserIDs, displaynames etc., ususally coming from
   *      a user-input-field
   * @param checkInput flag, if it is needed to check for invalid or nonUnique
   *      IDs
   * @return flag, if operation successful or not
   * @exception WcmException is thrown when problems in the usermanagement occur
   */
  public boolean addFromInput(String userList, boolean checkInput)
    throws WcmException {
    String[] newInput = this.checkForNewInput(userList);
    String searchUser = null;
    Collection newUser = new ArrayList();
    Collection newGroup = new ArrayList();
    Collection newRole = new ArrayList();
    Collection newEmail = new ArrayList();
    Collection newUnchecked = new ArrayList();

    if (checkInput == true) {
      String user = new String();
      for (int i = 0; i < newInput.length; i++) {
        user = user + newInput[i] + this.SEPARATOR;
      }
      if ((user.endsWith(this.SEPARATOR))) {
        user = user.substring(0, user.length() - 1);
      }
      if ((this.checkForInvalidIDs(user) != null) || (this.checkForNonUniqueIDs(user) != null)) {
        return false;
      }
    }

    if (newInput == null) {
      return true;
      //no new users entered
    }

    for (int i = 0; i < newInput.length; i++) {
      searchUser = this.getIntUserPresentation(newInput[i]);

      //user typed in a group
      if (isGroup(newInput[i])) {
        String[] groupResult = WPUMFactory.simpleGroupSearch(this.prepareUserForSearch(searchUser));
        if ((groupResult != null) && (groupResult.length == 1)) {
          newGroup.add(new SelectionUser(groupResult[0], this.TYPE_GROUP));
        }
        continue;
      }

      //user typed in a role
      if (isRole(newInput[i])) {
        String[] roleResult = WPUMFactory.simpleRoleSearch(this.prepareUserForSearch(searchUser));
        if ((roleResult != null) && (roleResult.length == 1)) {
          newRole.add(new SelectionUser(roleResult[0], this.TYPE_ROLE));
        }
        continue;
      }

      if (newInput[i].indexOf("@") > 0) {
        newEmail.add(new SelectionUser(newInput[i], this.TYPE_EMAIL));
        continue;
      }
      if (newInput[i].indexOf(this.DISPLAY_UNCHECKED_OPEN) > -1) {
        newUnchecked.add(new SelectionUser(newInput[i], this.TYPE_UNCHECKED));
        continue;
      }

      String[] userResult = WPUMFactory.simpleUserSearch(this.prepareUserForSearch(searchUser));
      if (userResult == null) {
        userResult = new String[0];
      }
      String[] groupResult = WPUMFactory.simpleGroupSearch(this.prepareUserForSearch(searchUser));
      if (groupResult == null) {
        groupResult = new String[0];
      }
      String[] roleResult = WPUMFactory.simpleRoleSearch(this.prepareUserForSearch(searchUser));
      if (roleResult == null) {
        roleResult = new String[0];
      }

      if ((userResult.length > 1) || (groupResult.length > 1) || (roleResult.length > 1)
         || ((userResult.length == 1) && (groupResult.length == 1))
         || ((userResult.length == 1) && (roleResult.length == 1))
         || ((roleResult.length == 1) && (groupResult.length == 1))) {
        //ID is not unique
        continue;
      }
      if (userResult.length == 1) {
        newUser.add(new SelectionUser(userResult[0], this.TYPE_USER));
        continue;
      }
      else if (groupResult.length == 1) {
        newGroup.add(new SelectionUser(groupResult[0], this.TYPE_GROUP));
        continue;
      }
      else if (roleResult.length == 1) {
        newRole.add(new SelectionUser(roleResult[0], this.TYPE_ROLE));
        continue;
      }
      return false;
    }
    //assign temp-Collection to fitting collection
    if (this.allowEmailAdress) {
      ((Collection)this.mapRecipients.get(new Integer(this.TYPE_EMAIL))).addAll(newEmail);
    }
    if (this.allowUncheckedNames) {
      ((Collection)this.mapRecipients.get(new Integer(this.TYPE_UNCHECKED))).addAll(newUnchecked);
    }
    this.addWithoutDoubles((Collection)this.mapRecipients.get(new Integer(this.TYPE_USER)), newUser);
    this.addWithoutDoubles((Collection)this.mapRecipients.get(new Integer(this.TYPE_GROUP)), newGroup);
    this.addWithoutDoubles((Collection)this.mapRecipients.get(new Integer(this.TYPE_ROLE)), newRole);
    return true;
  }


  /**
   * Adds a feature to the EmailFromInput attribute of the SelectionUserManager
   * object
   *
   * @param userList The feature to be added to the EmailFromInput attribute
   * @return Description of the Returned Value
   * @exception WcmException Description of Exception
   */
  public boolean addEmailFromInput(String userList)
    throws WcmException {
    String[] newInput = this.checkForNewInput(userList);
    Collection newEmail = new ArrayList();

    if (newInput == null) {
      return true;
      //no new users entered
    }

    for (int i = 0; i < newInput.length; i++) {
      if (newInput[i].indexOf("@") > 0) {
        newEmail.add(new SelectionUser(newInput[i], this.TYPE_EMAIL));
        continue;
      }
    }

    if (this.allowEmailAdress) {
      ((Collection)this.mapRecipients.get(new Integer(this.TYPE_EMAIL))).addAll(newEmail);
    }
    return true;
  }


  /**
   * before calling this method, the SelectionUserManager needs to be
   * initialized with the persistent string, via the method <code>initWithString
   * </code>. All information for control logic afterwards (invalid or non
   * unique users), all actions (delete, add, format displaynames) are performed
   * in this method. The persistence for the control once again can be exported
   * via <code>setList</> .
   *
   * @param userInput A list of UserIDs, Displaynames etc. usually coming from a
   *      user-input-field.
   * @exception WcmException problems occured within the UserManagement
   */
  public void handleAll(String userInput)
    throws WcmException {
    //save size of all lists to see, if sth. was deleted.
    int oldNumberOfRecipients = getNumberOfRecipients();
    this.deleteFromInput(userInput);
    if (oldNumberOfRecipients < getNumberOfRecipients()) {
      this.deletedRecipients = true;
    }
    //now check for new Input
    String[] temp = this.checkForNewInput(userInput);
    if (temp == null) {
      this.newRecipientsGiven = false;
      return;
    }
    else {
      this.newRecipientsGiven = true;
      this.newRecipients = this.convertStringArray2String(temp);
    }

    //check for invalid ID
    this.invalidIDsGiven = false;
    temp = this.checkForInvalidIDs(this.newRecipients);
    if (temp != null) {
      //invalid IDs found; User will be informed
      this.invalidIDs = this.convertStringArray2String(temp);
      this.invalidIDsGiven = true;
    }

    //check for non-unique IDs
    this.nonUniqueIDsGiven = false;
    nonUniqueIDs = this.getNonUniqueIDs(this.newRecipients);
    if (nonUniqueIDs != null) {
      //user has to verify names in UserSelectionControl
      this.nonUniqueIDsGiven = true;
    }
    //all is ok. Add from Input
    this.addFromInput(newRecipients, false);
  }


  /**
   * Gets the intUserPresentation attribute of the SelectionUserManager object
   *
   * @param user Description of Parameter
   * @return The intUserPresentation value
   */
  private String getIntUserPresentation(String user) {
    //get User without []<>
    if ((isGroup(user)) || (isRole(user))) {
      return user.substring(1, user.length() - 1);
    }
    return user;
  }


  /**
   * Gets the group attribute of the SelectionUserManager object
   *
   * @param user Description of Parameter
   * @return The group value
   */
  private boolean isGroup(String user) {
    if ((user.startsWith(this.DISPLAY_GROUP_OPEN)) && (user.endsWith(this.DISPLAY_GROUP_CLOSE))) {
      return true;
    }
    else {
      return false;
    }
  }


  /**
   * Gets the role attribute of the SelectionUserManager object
   *
   * @param user Description of Parameter
   * @return The role value
   */
  private boolean isRole(String user) {
    if ((user.startsWith(this.DISPLAY_ROLE_OPEN)) && (user.endsWith(this.DISPLAY_ROLE_CLOSE))) {
      return true;
    }
    else {
      return false;
    }
  }


  /**
   * Gets the numberOfRecipients attribute of the SelectionUserManager object
   *
   * @return The numberOfRecipients value
   */
  private int getNumberOfRecipients() {
    int result = 0;
    Iterator iter = this.mapRecipients.keySet().iterator();
    while (iter.hasNext()) {
      result += ((ArrayList)this.mapRecipients.get((Integer)iter.next())).size();
    }
    return result;
  }


  /**
   * Description of the Method
   *
   * @param allowEmails Description of Parameter
   * @param allowUncheckedNames Description of Parameter
   */
  private void doInitialize(boolean allowEmails, boolean allowUncheckedNames) {

    this.allowEmailAdress = allowEmails;
    this.allowUncheckedNames = allowUncheckedNames;

    mapRecipients = new HashMap();
    mapRecipients.put(new Integer(this.TYPE_USER), new ArrayList());
    mapRecipients.put(new Integer(this.TYPE_GROUP), new ArrayList());
    mapRecipients.put(new Integer(this.TYPE_ROLE), new ArrayList());
    if (this.allowEmailAdress) {
      mapRecipients.put(new Integer(this.TYPE_EMAIL), new ArrayList());
    }
    if (this.allowUncheckedNames) {
      mapRecipients.put(new Integer(this.TYPE_UNCHECKED), new ArrayList());
    }
  }


  //return a string with type-tag separator-id-tag separator-display name-tag finalizer
  /**
   * Description of the Method
   *
   * @param user Description of Parameter
   * @return Description of the Returned Value
   */
  private String collectionToString(Collection user) {
    if (user == null) {
      return "";
    }
    SelectionUser selectUser;
    Iterator iter = user.iterator();
    String userList = new String();

    while (iter.hasNext()) {
      selectUser = (SelectionUser)iter.next();
      userList = userList + selectUser.getType() + this.TAG_SEPARATOR + selectUser.getId() + this.TAG_SEPARATOR + selectUser.getDisplayName() + this.TAG_FINALIZER;
    }
    return userList;
  }


  //return a display string for a Collection of SelectionUser or an empty string, if collection is null
  /**
   * Description of the Method
   *
   * @param user Description of Parameter
   * @return Description of the Returned Value
   */
  private String collectionToDisplayNames(Collection user) {
    if (user == null) {
      return new String();
    }

    SelectionUser selectUser;
    Iterator iter = user.iterator();
    StringBuffer displayNames = new StringBuffer();

    while (iter.hasNext()) {
      selectUser = (SelectionUser)iter.next();
      displayNames = displayNames.append(this.SEPARATOR + " " + selectUser.getDisplayName());
    }
    return displayNames.toString();
  }


  // return entries not found
  /**
   * Description of the Method
   *
   * @param userList Description of Parameter
   * @param userType Description of Parameter
   * @return Description of the Returned Value
   */
  private Collection searchUserInCollection(Collection userList, int userType) {
    Collection inputIDs = new ArrayList();
    Collection usersList = new ArrayList(userList);
    SelectionUser userSelection;
    String user = new String();
    boolean found = false;

    Iterator listIterator = usersList.iterator();
    while (listIterator.hasNext()) {
      user = (String)listIterator.next();
      Collection userCollection = this.getSelectionUsers(userType);
      Iterator iter = userCollection.iterator();
      found = false;
      while (iter.hasNext()) {
        userSelection = (SelectionUser)iter.next();
        if ((userSelection.getDisplayName().equals(user)) || (userSelection.getId().equalsIgnoreCase(user))) {
//          inputIDs.add(user);
          found = true;
        }
      }
      if (found == false) {
        inputIDs.add(user);
      }
    }
    return inputIDs;
  }


  /**
   * Description of the Method
   *
   * @param list Description of Parameter
   * @param input Description of Parameter
   * @return Description of the Returned Value
   */
  private boolean deleteUserInCollection(Collection list, String input) {
    Iterator iter = list.iterator();
    while (iter.hasNext()) {
      SelectionUser user = (SelectionUser)iter.next();
      if ((user.getDisplayName().equalsIgnoreCase(input))
         || (user.getId().equalsIgnoreCase(input))) {
        iter.remove();
        return true;
        //user was in list
      }
    }
    return false;
    //user was not in list
  }


  /**
   * Description of the Method
   *
   * @param list Description of Parameter
   * @param input Description of Parameter
   * @return Description of the Returned Value
   */
  private boolean deleteUserIfDisplayNameChanged(Collection list, String input) {
    Iterator iter = list.iterator();
    while (iter.hasNext()) {
      SelectionUser user = (SelectionUser)iter.next();
      if (user.getDisplayName().equalsIgnoreCase(input)) {
        iter.remove();
        return true;
        //user was in list
      }
    }
    return false;
    //user was not in list
  }


  /**
   * Description of the Method
   *
   * @param temp Description of Parameter
   * @param type Description of Parameter
   * @return Description of the Returned Value
   */
  private boolean updateCollectionAfterDelete(Collection temp, int type) {
    Collection persistentCollection = (Collection)this.mapRecipients.get(new Integer(type));
    Iterator iter = temp.iterator();
    while (iter.hasNext()) {
      deleteUserInCollection(persistentCollection, ((SelectionUser)iter.next()).getId());
    }
    return true;
  }


  /**
   * Description of the Method
   *
   * @param list Description of Parameter
   * @return Description of the Returned Value
   */
  private Collection StringArray2Collection(String[] list) {
    Collection result = new ArrayList();
    for (int i = 0; i < list.length; i++) {
      result.add(list[i]);
    }
    return result;
  }


  /**
   * Description of the Method
   *
   * @param user Description of Parameter
   * @return Description of the Returned Value
   */
  private String prepareUserForSearch(String user) {
    //add an "*" for search-operations
    if (!user.endsWith("*")) {
      return user + "*";
    }
    else {
      return user;
    }
  }


  /**
   * Description of the Method
   *
   * @param arrayList Description of Parameter
   * @return Description of the Returned Value
   */
  private String convertStringArray2String(String[] arrayList) {
    StringBuffer result = new StringBuffer();
    for (int i = 0; i < arrayList.length; i++) {
      result.append(arrayList[i] + this.SEPARATOR);
    }
    return result.substring(0, result.length() - 1);
  }

  private void addWithoutDoubles(Collection coll, SelectionUser su) {
    if (!coll.contains(su)) {
      coll.add(su);
    }
  }

  private void addWithoutDoubles(Collection coll, Collection list) {
    Iterator iter = list.iterator();
    SelectionUser su = null;
    while (iter.hasNext()) {
      su = (SelectionUser)iter.next();
      this.addWithoutDoubles(coll, su);
    }
  }
}
