/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.uri;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * Manages query parameters for RFC 2396 URIs. <p>
 *
 * Copyright (c) SAP AG 2001-2003
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Id: UriQuery.java,v 1.3 2003/02/17 14:24:04 jre Exp $
 */
public final class UriQuery {

  /**
   * Keeps one query parameter.
   */
  public final static class Parameter {
    public final String name;
    public final String value;

    public Parameter(String name, String value) {
      if (name == null) {
        throw new NullPointerException("parameter key cannot be null");
      }
      this.name = name;
      this.value = value;
    }
  }

  /**
   * Get an object for manipulating the given query string. The query string is
   * in RFC 2396 encoded form.
   *
   * @param query the query from an uri
   * @return object to manipulate query
   */
  public static UriQuery valueOf(String query) {
    return new UriQuery(query);
  }

  private final List params;

  private UriQuery(String query) {
    this.params = URICodec.DecodeQueryParameter(query);
  }

  /**
   * Construct an empty query.
   */
  public UriQuery() {
    this.params = new ArrayList(10);
  }

  public String toString() {
    return URICodec.EncodeQueryParameter(this.params);
  }

  /**
   * Convert query back to its RFC 2396 encoded form.
   *
   * @return encoded query string
   */
  public String toExternalForm() {
    return toString();
  }

  /**
   * Adds a new parameter to the end of the query
   *
   * @param name of parameter
   * @param value of parameter or <code>null</code> if not present
   */
  public void add(String name, String value) {
    this.params.add(new Parameter(name, value));
  }

  /**
   * Removes all occurances of parameter with given name.
   *
   * @param name to remove
   */
  public void remove(String name) {
    for (Iterator i = this.params.iterator(); i.hasNext(); ) {
      Parameter p = (Parameter)i.next();
      if (p.name.equals(name)) {
        i.remove();
      }
    }
  }

  /**
   * Remove all parameters from the query.
   */
  public void clear() {
    this.params.clear();
  }

  /**
   * Get iterator to iterate over all parameters
   *
   * @return iterator over all {@link Parameter}s
   */
  public Iterator iterator() {
    return this.params.iterator();
  }

  /**
   * Sets first occurance of parameter name to given value. If parameter was not
   * part of query already, the parameter is appended to the query.
   *
   * @param name of parameter
   * @param value of parameter or <code>null</code> if not present
   */
  public void set(String name, String value) {
    for (int i = 0, n = this.params.size(); i < n; ++i) {
      Parameter p = (Parameter)this.params.get(i);
      if (p.name.equals(name)) {
        this.params.set(i, new Parameter(name, value));
        return;
      }
    }
    add(name, value);
  }

  /**
   * Sets query to given list of parameters. All existing parameters will be
   * removed.
   *
   * @param parameter TBD: Description of the incoming method parameter
   */
  public void set(List parameter) {
    this.params.clear();
    this.params.addAll(parameter);
  }

  /**
   * Return number of parameters.
   *
   * @return number of parameters
   */
  public int size() {
    return this.params.size();
  }

  /**
   * Get the Parameter at given index
   *
   * @param index of parameter to get
   * @return Parameter at index
   * @throws IndexOutOfBoundsException when index not available
   */
  public Parameter get(int index)
    throws IndexOutOfBoundsException {
    return (Parameter)this.params.get(index);
  }
}
