/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.uri;

/**
 * A factory producing {@link IUri}s from various inputs. <pre>
 *     Uri ::= scheme ':' remainder
 * </pre> Copyright (c) SAP AG 2001-2004
 *
 * @author julian.reschke@greenbytes.de
 * @author stefan.eissing@greenbytes.de
 * @version $Id: UriFactory.java,v 1.12 2002/01/18 17:53:46 sei Exp $
 */

public class UriFactory {

  /**
   * Parse an {@link IHierarchicalUri} from its string representation
   *
   * @param externalForm string representation to parse
   * @return parsed URI
   * @throws java.net.MalformedURLException when URI is malformed or not
   *      hierarchical
   */
  public static IHierarchicalUri parseHierarchicalUri(String externalForm)
    throws java.net.MalformedURLException {
    IUri uri = parseUri(externalForm);
    if (uri instanceof IHierarchicalUri) {
      return (IHierarchicalUri)uri;
    }
    throw new java.net.MalformedURLException("is not hierarchical: " + externalForm);
  }

  /**
   * Parse an {@link IUri} from its string representation
   *
   * @param externalForm string representation to parse
   * @return parsed URI
   * @throws java.net.MalformedURLException when URI is malformed
   */
  public static IUri parseUri(String externalForm)
    throws java.net.MalformedURLException {
    return new PreUri(externalForm).getUri();
  }

  /**
   * Parse an URI reference from its string representation
   *
   * @param externalForm string representation to parse
   * @return parsed URI reference
   * @throws java.net.MalformedURLException when URI reference is malformed
   */
  public static IUriReference parseUriReference(String externalForm)
    throws java.net.MalformedURLException {
    return new PreUri(externalForm).getUriReference();
  }

  // ---------------------- private ----------------------------------------

  private static class PreUri {
    private final String scheme;
    private final String authority;
    private final String path;
    private final String query;
    private final String fragment;

    public PreUri(String externalForm)
      throws java.net.MalformedURLException {
      if (externalForm == null) {
        throw new java.net.MalformedURLException("external form is null");
      }

      String work = externalForm;
      int index = work.indexOf('#');
      if (index >= 0) {
        fragment = tail(work, index);
        work = work.substring(0, index);
      }
      else {
        fragment = null;
      }

      if ((index = work.indexOf('?')) >= 0) {
        query = tail(work, index);
        work = work.substring(0, index);
      }
      else {
        query = null;
      }

      if (work.indexOf('\\') >= 0) {
        // We convert the \ into / and assume
        // that this externalForm came from a Windows host. Internet
        // Explorer accepts Uris with \ even though this is not
        // an allowed character according to RFC 2396.
        //
        work = work.replace('\\', '/');
      }

      if ( ! work.startsWith("//")) {
        // no net_path production
        index = work.indexOf(':');
        if (index > 0) {
          scheme = work.substring(0, index).toLowerCase();
          work = tail(work, index);
        }
        else {
          // no scheme
          scheme = null;
        }
      }
      else {
        // net_path
        scheme = null;
      }

      if (work.startsWith("//")) {
        // authority
        index = work.indexOf('/', 2);
        if (index > 2) {
          authority = work.substring(2, index);
          work = work.substring(index);
        }
        else {
          authority = tail(work, 1);
          work = "";
        }
      }
      else {
        // no authority
        authority = null;
      }

      path = work;

      // Some last sanity checks on the parsed components
      //
      if (authority != null) {
        if (path.length() > 0 && !path.startsWith("/")) {
          throw new java.net.MalformedURLException("uri with authority requires absolute path");
        }
      }
      else {
/*        if (scheme != null && path.length() == 0 && query == null && fragment == null) {
          throw new java.net.MalformedURLException("a scheme alone is no valid uri or uri reference");
        }
*/      }
    }

    public IUri getUri()
      throws java.net.MalformedURLException {

      if (fragment != null) {
        throw new java.net.MalformedURLException("illegal fragment identifier");
      }
      return internalGetUri();
    }

    public IUriReference getUriReference()
      throws java.net.MalformedURLException {
      if (scheme == null) {
        return new UriReference(authority, path, query, fragment);
      }
      else if (authority != null) {
        // scheme + authority != null, is an absolute uri reference
        return new UriReference(internalGetUri(), fragment);
      }
      else {
        // scheme != null + authority == null, should be a non-hierarchical
        // absolute uri reference, but there are degenerate cases as mentioned
        // in RFC 2396, page 32.
        //
        try {
          return new UriReference(internalGetUri(), fragment);
        }
        catch (java.net.MalformedURLException ex) {
            //$JL-EXC$          
          return new UriReference(scheme, null, path, query, fragment);
        }
      }
    }

    private IUri internalGetUri()
      throws java.net.MalformedURLException {
      if (scheme == null) {
        throw new java.net.MalformedURLException("scheme missing");
      }

      try {
        if (scheme.equals("http") || scheme.equals("https")) {
          if (authority == null) {
            throw new java.net.MalformedURLException("authority missing");
          }

          return new HttpUrl(scheme, authority, path, query);
        }
        else if (scheme.equals("resource")) {
          if (authority == null) {
            throw new java.net.MalformedURLException("authority missing");
          }

          return new ResourceUrl(scheme, authority, path, query);
        }
        else {
          StringBuffer sb = new StringBuffer(128);
          if (authority != null) {
            sb.append("//").append(authority);
          }
          if (path != null) {
            sb.append(path);
          }
          if (query != null) {
            sb.append('?').append(query);
          }

          if (scheme.equals("data")) {
            return new DataUri(sb.toString());
          }
          else if (scheme.equals("opaquelocktoken")) {
            return new OpaqueLockTokenUri(sb.toString());
          }
          else {
            return new GenericUri(scheme, sb.toString());
          }
        }

      }
      catch (IllegalArgumentException ex) {
            //$JL-EXC$        
        throw new java.net.MalformedURLException(ex.getMessage());
      }
    }

    private String tail(String s, int splitIndex) {
      ++splitIndex;
      return (splitIndex < s.length()) ? s.substring(splitIndex) : "";
    }
  }
}
