/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.uri;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Properties;

/**
 * An URI (better WcmPath) is an identifier for a WCM resource. <p>
 *
 * An URI holds all information to identify a WCM resource and retrieve it via
 * the WCM Framework. Each WCM resource has its unique URI. <p>
 *
 * A URI has a path-like structure, like a path in a file system and follows the
 * following production: <pre>
 *   URI         ::= '/' pathsegment | '/' pathsegment '?' query
 *   pathsegment ::= name | name '/' pathsegment | empty
 *   name        ::= any char except '/'
 *   query       ::= parameter | parameter '&' query | empty
 *   parameter   ::= varname | varname '=' value
 *   varname     ::= any char except '=', '&'
 *   value       ::= any character except '&'
 * </pre> Equality is defined as ignoring trailing '/' characters. Thus, the
 * URIs "/docs/" and /docs" are considered equal. There is special handling for
 * the root collection "/", which is not equal to the empty URI "". <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author frank.renkes@sapportals.com
 * @author stefan.eissing@greenbytes.de
 * @deprecated as of EP 5.0 SP3, replaced by {@link RID}
 */
public final class URI implements Comparable {

  private static URI emptyURI = new URI("");
  private static URI rootURI = new URI("/");

  private String m_uri;
  private final String m_path;
  private final int m_pathlen;
  private final String m_query;
  private final boolean m_endsWithSlash;
  private final boolean m_isAbsolute;
  private int m_hashCode;
  private URI m_root;
  private URI m_name;
  private URI m_parent;

  /**
   * Create a new URI with given path and query string
   *
   * @param path of URI
   * @param query part or URI (can be null)
   */
  public URI(String path, String query) {
    m_path = (path == null) ? "" : path;
    m_pathlen = m_path.length();
    if (m_pathlen > 0) {
      m_endsWithSlash = m_path.charAt(m_pathlen - 1) == '/';
      m_isAbsolute = m_path.charAt(0) == '/';
    }
    else {
      m_endsWithSlash = false;
      m_isAbsolute = false;
    }
    m_query = query;
    if (m_query == null) {
      m_uri = m_path;
    }
  }

  /**
   * Create a new UTI with given path, without query
   *
   * @param uri TBD: Description of the incoming method parameter
   */
  public URI(String uri) {
    // sei: I'd like to remove the recognition of the query string
    // here. However in the repository framework, the ResourceFactory
    // round-trips URI and URI.toString()
    // So:              URI("/test", "a=b")
    // toString:        "/test?a=b"
    // new URI(string); URI("/test?a=b", null)
    // but correct is   URI("/test", "a=b")
    //
    String path = (uri != null) ? uri : "";
    m_uri = path;
    String query = null;
    if (uri != null) {
      int index = path.lastIndexOf('?');
      if (index >= 0) {
        query = path.substring(index + 1);
        path = path.substring(0, index);
      }
    }
    m_path = path;
    m_pathlen = m_path.length();
    if (m_pathlen > 0) {
      m_endsWithSlash = m_path.charAt(m_pathlen - 1) == '/';
      m_isAbsolute = m_path.charAt(0) == '/';
    }
    else {
      m_endsWithSlash = false;
      m_isAbsolute = false;
    }
    m_query = query;
  }

  /**
   * @return TBD: Description of the outgoing return value
   * @deprecated as of NW04. URI objects are never encoded
   */
  public URI encode() {
    return new URI(URICodec.Encode(m_path));
  }

  /**
   * @return TBD: Description of the outgoing return value
   * @deprecated as of NW04. URI objects are never decoded
   */
  public URI decode() {
    return new URI(URICodec.Decode(m_path));
  }

  /**
   * Convert this URI to a representation usable in W3C Uri References.
   * Non-ascii and reserved characters are encoded.<p>
   *
   * Note that such an encoded string cannot be used to construct an new URI
   * object! The path in URI objects is never, ever encoded.
   *
   * @return URI in W3C string representation
   */
  public String toExternalForm() {
    if (m_query != null) {
      StringBuffer sb = new StringBuffer(256);
      sb.append(URICodec.EncodePath(m_path));
      sb.append('?');
      sb.append(m_query);
      return sb.toString();
    }
    else {
      return URICodec.EncodePath(m_path);
    }
  }

  /**
   * Return the path component of this URI.
   *
   * @return the path component
   */
  public String getPath() {
    return m_path;
  }

  /**
   * Return the query component of this URI (null if it does not exist).
   *
   * @return the query component
   */
  public String getQuery() {
    return m_query;
  }

  /**
   * Get the query parameters of the URI. Will return an empty Properties object
   * if URI has no query part. Modifying the {@link Properties} object has no
   * effect on the URI itself.
   *
   * @return query parameters as {@link Properties}
   */
  public Properties getQueryParameter() {
    if (m_query != null && m_query.length() > 0) {
      return URICodec.DecodeQuery(m_query);
    }
    else {
      return new Properties();
    }
  }

  /**
   * Get URI of toplevel collection (below root) of this URI. <p>
   *
   * The root URI of a toplevel collection is the root collection URI. The root
   * URI of the root URI is the root URI itself.
   *
   * @return URI of toplevel collection below root
   */
  public URI root() {
    if (null != m_root) {
      return m_root;
    }
    if (0 == m_pathlen) {
      return this;
    }

    int pos = m_path.indexOf('/', 1);
    m_root = (pos != -1) ? new URI(m_path.substring(0, pos)) : rootURI;

    return m_root;
  }

  /**
   * Get the name of the resource, designated by this URI. If the URI has query
   * parameter, these are discarded.
   *
   * @return the name of the resource, designated by this URI
   */
  public URI name() {
    if (null != m_name) {
      return m_name;
    }
    if (0 == m_pathlen) {
      return this;
    }

    if (m_endsWithSlash) {
      int pos = m_path.lastIndexOf('/', m_pathlen - 2);
      m_name = new URI(m_path.substring(pos + 1, m_pathlen - 1));
    }
    else {
      int pos = m_path.lastIndexOf('/');
      m_name = (pos != -1) ? new URI(m_path.substring(pos + 1)) : this;
    }

    return m_name;
  }


  /**
   * Get the URI of the parent collection of this URI
   *
   * @return the URI of the parent collection
   */
  public URI parent() {
    if (null != m_parent) {
      return m_parent;
    }
    m_parent = removeName();
    return m_parent;
  }

  /**
   * Get the extension part of the resource name. Returns the empty string if
   * there is no extension.
   *
   * @return the extension part of the resource name
   */
  public String extension() {
    if (m_pathlen == 0) {
      return "";
    }

    URI n = name();
    int pos = n.m_path.lastIndexOf('.');
    if (pos != -1 && pos < n.m_pathlen - 1) {
      return n.m_path.substring(pos + 1);
    }
    else {
      return "";
    }
  }

  /**
   * Get a new URI with the toplevel collection removed
   *
   * @return URI with toplevel collection removed
   */
  public URI removeRoot() {
    if (m_pathlen == 0) {
      return this;
    }

    int start = m_isAbsolute ? 1 : 0;
    int pos = m_path.indexOf('/', start);

    return (pos == -1 || pos == m_pathlen - 1) ? URI.emptyURI : new URI(m_path.substring(pos), m_query);
  }

  /**
   * Remove the name of the resource, same as parent()
   *
   * @return parent of the resource
   */
  public URI removeName() {
    if (m_pathlen == 0) {
      return this;
    }

    int pos;
    if (m_endsWithSlash) {
      pos = m_path.lastIndexOf('/', m_pathlen - 2);
    }
    else {
      pos = m_path.lastIndexOf('/');
    }

    switch (pos) {
      case 0:
        return URI.rootURI;
      case -1:
        return URI.emptyURI;
      default:
        return new URI(m_path.substring(0, pos));
    }
  }

  /**
   * Remove the (optional) extension of the resource name. Query parameters are
   * discarded.
   *
   * @return uri with extension of name removed
   */
  public URI removeExtension() {
    if (m_pathlen == 0 || m_endsWithSlash) {
      return this;
    }

    int spos = m_path.lastIndexOf('/');
    int pos = m_path.lastIndexOf('.');

    if (pos > -1 && pos > spos) {
      return new URI(m_path.substring(0, pos));
    }
    else {
      return this;
    }
  }

  /**
   * Remove a possible trailing slash from the URI. Query parameters are
   * preserved.
   *
   * @return URI without trailing slash
   */
  public URI removeTrailingSlash() {
    return m_endsWithSlash ?
      new URI(m_path.substring(0, m_pathlen - 1), m_query) : this;
  }

  /**
   * Return if URI is absolute, e.g. start with a slash.
   *
   * @return if URI is absolute path
   */
  public boolean isAbsolute() {
    return m_isAbsolute;
  }

  /**
   * Return if URI is root collection.
   *
   * @return if this is the root collection
   */
  public boolean isRoot() {
    return m_pathlen == 1 && m_isAbsolute;
  }

  /**
   * Determine if this URI is an ancestor (parent or parent's parent, etc.) of
   * the given child URI.
   *
   * @param child to test against
   * @return if this URI is ancestor of child
   */
  public boolean isAncestorOf(URI child) {
    if (m_pathlen < child.m_pathlen && m_isAbsolute == child.m_isAbsolute) {
      if (child.m_path.startsWith(m_path)) {
        return m_endsWithSlash || child.m_path.charAt(m_pathlen) == '/';
      }
    }
    return false;
  }

  /**
   * Return if this URI ends with a slash.
   *
   * @return if this URI ends with a slash
   */
  public boolean endsWithSlash() {
    return m_endsWithSlash;
  }

  /**
   * Concatenate this URI with the given URI. Treats this URI as if it ends with
   * a slash and the parameter URI as if it starts without a slash.
   *
   * @param uri to append to this
   * @return new URI as concatenation
   */
  public URI add(String uri) {
    if (m_pathlen == 0) {
      return new URI(uri);
    }
    int ulen = uri.length();
    if (ulen == 0) {
      return this;
    }

    String query = null;
    int index = uri.lastIndexOf('?');
    if (index >= 0) {
      query = (index < ulen) ? uri.substring(index + 1) : "";
      uri = uri.substring(0, index);
      ulen = index;
    }

    StringBuffer sb = new StringBuffer(m_pathlen + ulen + 1);
    sb.append(m_path);

    if (m_endsWithSlash) {
      if (uri.charAt(0) == '/') {
        sb.setLength(m_pathlen - 1);
      }
    }
    else {
      if (uri.charAt(0) != '/') {
        sb.append('/');
      }
    }
    sb.append(uri);

    return new URI(sb.toString(), query);
  }

  /**
   * Concatenate this URI with the given URI. Treats this URI as if it ends with
   * a slash and the parameter URI as if it starts without a slash.
   *
   * @param uri to append to this
   * @return new URI as concatenation
   */
  public URI add(URI uri) {
    if (m_pathlen == 0) {
      return uri;
    }
    if (uri.m_pathlen == 0) {
      return new URI(m_path, uri.m_query);
    }

    StringBuffer sb = new StringBuffer(m_pathlen + uri.m_pathlen + 1);
    sb.append(m_path);

    if (m_endsWithSlash) {
      if (uri.m_isAbsolute) {
        sb.setLength(m_pathlen - 1);
      }
    }
    else {
      if (!uri.m_isAbsolute) {
        sb.append('/');
      }
    }
    sb.append(uri.m_path);

    return new URI(sb.toString(), uri.m_query);
  }

  /**
   * Splits the URI into its path components
   *
   * @return List of {@link String}s
   */
  public List split() {
    if (m_pathlen == 0) {
      return Collections.EMPTY_LIST;
    }
    return URI.splitString(m_path, m_pathlen, '/');
  }

  /**
   * Returns string representation of this URI
   *
   * @return this URI as string
   */
  public String toString() {
    if (m_uri == null) {
      if (m_query != null) {
        m_uri = m_path + '?' + m_query;
      }
      else {
        m_uri = m_path;
      }
    }
    return m_uri;
  }

  /**
   * Determine if this URI equals other.
   *
   * @param other URI
   * @return if this URI equals other
   */
  public boolean equals(URI other) {
    if (this == other) {
      return true;
    }
    if (other == null) {
      return false;
    }

    if (hashCode() != other.hashCode()) {
      return false;
    }

    if (this.m_pathlen == other.m_pathlen) {
      return this.m_path.equals(other.m_path) && equalsQuery(other);
    }
    else if (this.m_endsWithSlash) {
      if (!other.m_endsWithSlash && this.m_pathlen == other.m_pathlen + 1) {
        return this.m_path.startsWith(other.m_path) && equalsQuery(other);
      }
    }
    else if (other.m_endsWithSlash) {
      if (other.m_pathlen == this.m_pathlen + 1) {
        return other.m_path.startsWith(this.m_path) && equalsQuery(other);
      }
    }
    return false;
  }

  /**
   * Determine if this URI equals other.
   *
   * @param other URI (string representation)
   * @return if this URI equals other
   */
  public boolean equals(String other) {
    if (other != null) {
      String query = null;
      int index = other.lastIndexOf('?');
      if (index >= 0) {
        query = (index < other.length()) ? other.substring(index + 1) : "";
        other = other.substring(0, index);
      }
      return equals(new URI(other, query));
    }
    return false;
  }

  /**
   * Determine if this URI equals other. Equality is defined for instances of
   * URI only.
   *
   * @param other
   * @return if this URI equals other
   */
  public boolean equals(Object other) {
    if (this == other) {
      return true;
    }

    if (other instanceof URI) {
      return equals((URI)other);
    }
    // sei: not a good idea, since this relation is not reflexiv
    //else if (other instanceof String) {
    //  return equals((String)other);
    //}
    else if (other instanceof RID) {
      throw new RuntimeException("comparing URI to instance of RID");
    }
    else {
      return false;
    }
  }

  public int hashCode() {
    if (m_hashCode == 0) {
      m_hashCode = (m_pathlen > 1 && m_endsWithSlash) ?
        m_path.substring(0, m_pathlen - 1).hashCode() : m_path.hashCode();
      if (m_query != null) {
        m_hashCode += m_query.hashCode();
      }
    }
    return m_hashCode;
  }

  /**
   * Return length of URI in string representation
   *
   * @return length of URI in string representation
   */
  public int length() {
    if (m_query != null) {
      return m_pathlen + 1 + m_query.length();
    }
    return m_pathlen;
  }

  /**
   * Resolve relative URIs
   *
   * @param uri An URI, which may be relative or absolute
   * @return TBD: Description of the outgoing return value
   */
  public URI resolveRelativeURI(String uri) {

    // absolute href (e.g. http:// c:\ c:/).
    if (m_pathlen > 0 && uri.charAt(0) != '/' && uri.indexOf(":/") < 0 && uri.indexOf(":\\", 1) < 0) {

      // relative uri. Append it to the base url
      String path;
      if (m_endsWithSlash) {
        path = m_path;
      }
      else {
        path = parent().toString();
        if (!path.endsWith("/")) {
          path += "/";
        }
      }
      String newUri = path.concat(uri);

      //  remove all "./" where "." is a complete path segment
      int index = -1;
      while ((index = newUri.indexOf("/./")) != -1) {
        newUri = path.substring(0, index + 1).concat(newUri.substring(index + 3));
      }

      // remove all "<segment>/../" where "<segment>" is a complete
      // path segment not equal to ".."
      index = -1;
      int segIndex = -1;
      String tempString = null;

      while ((index = newUri.indexOf("/../")) > 0) {
        tempString = newUri.substring(0, newUri.indexOf("/../"));
        segIndex = tempString.lastIndexOf('/');
        if (segIndex != -1) {
          if (!tempString.substring(segIndex++).equals("..")) {
            newUri = newUri.substring(0, segIndex).concat(newUri.substring(index + 4));
          }
        }
      }
      return new URI(newUri);
    }
    else {
      return new URI(uri);
    }
  }

  /**
   * Create a new URI by setting the given parameters as query part of this URI.
   *
   * @param parameter to set to query
   * @return new URI with parameter in query
   * @deprecated as of NW04. Use setQueryParameter
   */
  public URI addQueryParameter(Properties parameter) {
    return setQueryParameter(parameter);
  }

  /**
   * Create a new URI by setting the given parameters as query part of this URI.
   *
   * @param parameter to set to query
   * @return new URI with parameter in query
   */
  public URI setQueryParameter(Properties parameter) {
    return new URI(m_path, URICodec.EncodeQuery(parameter));
  }

  // --- Comparable interface ---

  public int compareTo(Object o) {
    String uri = m_path;
    String s2 = null;

    if (o instanceof String) {
      s2 = (String)o;
    }
    else if (o instanceof URI) {
      s2 = ((URI)o).m_path;
    }
    else {
      throw new java.lang.IllegalArgumentException("URI cannot be compared to " + o.getClass().getName());
    }

    final int o_len = s2.length();
    if (m_pathlen != o_len) {
      if ((m_pathlen > 1) && (o_len > 1)) {
        if (m_endsWithSlash) {
          uri = uri.substring(0, m_pathlen - 1);
        }
        if (s2.charAt(o_len - 1) == '/') {
          s2 = s2.substring(0, o_len - 1);
        }
      }
    }

    return uri.compareTo(s2);
  }

  // ----------------------------- private -----------------------------------

  private static List splitString(final String str, final int len, final int delim) {

    if (len == 0) {
      return Collections.EMPTY_LIST;
    }

    List names = new ArrayList(10);

    int i = (str.charAt(0) == delim) ? 1 : 0;
    int last = i;
    for (; i < len; ++i) {
      if (str.charAt(i) == delim) {
        names.add(str.substring(last, i));
        last = i + 1;
      }
    }

    if (last < i) {
      names.add(str.substring(last));
    }

    return names;
  }

  private URI() {
    this(null);
  }

  private boolean equalsQuery(URI other) {
    if (m_query != null) {
      return m_query.equals(other.m_query);
    }
    else {
      return other.m_query == null;
    }
  }
}
