package com.sapportals.wcm.util.uri;

import com.sapportals.wcm.WcmException;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Properties;

/**
 * A RID is an identifier for a CM resource. It holds all information to
 * identify a CM resource and retrieve it via the CM Framework. Each CM resource
 * has its unique RID. The first pathsegment must be a valid name of a
 * repository (prefix). <p>
 *
 * A RID has a path-like structure, like a path in a file system and follows the
 * following production: <pre>
 *   RID         ::= '/' pathsegment | '/' pathsegment '?' query
 *   pathsegment ::= name | name '/' pathsegment | empty
 *   name        ::= any char except '/'
 *   query       ::= parameter | parameter '&' query | empty
 *   parameter   ::= varname | varname '=' value
 *   varname     ::= any char except '=', '&'
 *   value       ::= any character except '&'
 * </pre> Equality is defined as ignoring trailing '/' characters. Thus, the
 * RIDs "/docs/" and /docs" are considered equal. There is special handling for
 * the root collection "/", which is not equal to the empty RID "". <p>
 *
 * Copyright (c) SAP AG 2001-2004
 *
 * @author frank.renkes@sapportals.com
 * @author stefan.eissing@greenbytes.de
 * @author m.breitenfelder@sapportals.com
 */
public final class RID implements Comparable, Serializable {

  public final static char PATH_SEPARATOR = '/';

  private static RID emptyRID = RID.getRID("");
  private static RID rootRID = RID.getRID(new String(new char[]{RID.PATH_SEPARATOR}));

  private String uri;
  private final String path;
  private final int pathlen;
  private final String query;
  private final boolean endsWithSlash;
  private final boolean isAbsolute;
  private int hashCode;
  private RID root;
  private RID name;
  private RID parent;

// stefan.eissing@greenbytes.de: static caching of RIDs disabled for now
// Problems that need to be addressed:
// - needs synchronization for read *and* write accesses
// - SoftReferences make garbage collection more difficult and less efficient
// - RID("a?", null) and RID("a?") have same key, but different external form
//
//	private static HashMap ridCache;

  public static RID getRID(String path, String query) {
    return new RID(path, query);
//		if (null == ridCache) ridCache = new HashMap();
//
//		String key;
//
// 		if (query != null && query.length() > 0) {
//			key = path + "?" + query;
//		} else {
//			key = path;
//		}
//
//		RidReference ridRef = (RidReference)ridCache.get(key);
//		RID 		 rid;
//
//		if (ridRef != null) {
//			if (null != (rid = ridRef.getRID())) {
//				return rid;
//			}
//		}
//
//		RID newRID = new RID(path, query);
//
//		ridCache.put(key, new RidReference(newRID));
//
//		return newRID;
  }

  public static RID getRID(String uri) {
    return new RID(uri);
//    if (null == ridCache) ridCache = new HashMap();
//
//    RidReference ridRef = (RidReference)ridCache.get(uri);
//    RID 		 rid;
//
//    if (ridRef != null) {
//      if (null != (rid = ridRef.getRID())) {
//        return rid;
//      }
//    }
//
//    RID newRID = new RID(uri);
//
//    ridCache.put(uri, new RidReference(newRID));
//
//    return newRID;
  }


  /**
   * Create a new RID with given path and query string
   *
   * @param path of RID
   * @param query part or RID (can be null)
   * @deprecated as of NW04. Use RID.getRID(path, query)
   */
  public RID(String path, String query) {
    this.path = (path == null) ? "" : path;
    this.pathlen = this.path.length();
    if (this.pathlen > 0) {
      this.endsWithSlash = this.path.charAt(this.pathlen - 1) == RID.PATH_SEPARATOR;
      this.isAbsolute = this.path.charAt(0) == RID.PATH_SEPARATOR;
    }
    else {
      this.endsWithSlash = false;
      this.isAbsolute = false;
    }
    this.query = query;
    if (this.query == null) {
      this.uri = this.path;
    }
  }

  /**
   * Create a new RID from a string (possibly including a query part). <p>
   *
   *
   *
   * @param uri TBD: Description of the incoming method parameter
   * @deprecated as of NW04. Use RID.getRID(uri)
   */
  public RID(String uri) {
    // sei: I'd like to remove the recognition of the query string
    // here. However in the repository framework, the ResourceFactory
    // round-trips RID and RID.toString()
    // So:              RID("/test", "a=b")
    // toString:        "/test?a=b"
    // new RID(string); RID("/test?a=b", null)
    // but correct is   RID("/test", "a=b")
    //
    String path = (uri != null) ? uri : "";
    this.uri = path;
    String query = null;
    if (uri != null) {
      int index = path.lastIndexOf('?');
      if (index >= 0) {
        query = path.substring(index + 1);
        path = path.substring(0, index);
      }
    }
    this.path = path;
    this.pathlen = this.path.length();
    if (this.pathlen > 0) {
      this.endsWithSlash = this.path.charAt(this.pathlen - 1) == RID.PATH_SEPARATOR;
      this.isAbsolute = this.path.charAt(0) == RID.PATH_SEPARATOR;
    }
    else {
      this.endsWithSlash = false;
      this.isAbsolute = false;
    }
    this.query = query;
  }

  /**
   * @return TBD: Description of the outgoing return value
   * @deprecated as of NW04. RID objects are never encoded
   */
  public RID encode() {
    return RID.getRID(URICodec.Encode(this.path));
  }

  /**
   * @return TBD: Description of the outgoing return value
   * @deprecated as of NW04. RID objects are never decoded
   */
  public RID decode() {
    return RID.getRID(URICodec.Decode(this.path));
  }

  /**
   * Convert this RID to a representation usable in RFC2396 URI references.
   * Non-ASCII and reserved characters are encoded.<p>
   *
   * Note that such an encoded string cannot be used to construct an new RID
   * object! The path in RID objects is never, ever encoded.
   *
   * @returns RID suitably escaped as RFC2396 URI reference
   */
  private String toExternalFormCache = null;

  public String toExternalForm() {
    if (toExternalFormCache == null) {
      if (this.query != null) {
        StringBuffer sb = new StringBuffer(256);
        sb.append(URICodec.EncodePath(this.path));
        sb.append('?');
        sb.append(this.query);
        toExternalFormCache = sb.toString();
      }
      else {
        toExternalFormCache = URICodec.EncodePath(this.path);
      }
    }
    return toExternalFormCache;
  }

  /**
   * Return the path component of this RID.
   *
   * @return the path component
   */
  public String getPath() {
    return this.path;
  }

  /**
   * Return the query component of this RID (null if it does not exist).
   *
   * @return the query component
   */
  public String getQuery() {
    return this.query;
  }

  /**
   * Get the query parameters of the RID. Will return an empty Properties object
   * if RID has no query part. Modifying the {@link Properties} object has no
   * effect on the RID itself.
   *
   * @return query parameters as {@link Properties}
   */
  public Properties getQueryParameter() {
    if (this.query != null && this.query.length() > 0) {
      return URICodec.DecodeQuery(this.query);
    }
    else {
      return new Properties();
    }
  }

  /**
   * Get RID of toplevel collection (below root) of this RID. <p>
   *
   * The root RID of a toplevel collection is the root collection RID. The root
   * RID of the root RID is the root RID itself.
   *
   * @return RID of toplevel collection below root
   */
  public RID root() {
    if (null != this.root) {
      return this.root;
    }
    if (0 == this.pathlen) {
      return this;
    }

    int pos = this.path.indexOf(RID.PATH_SEPARATOR, 1);
    this.root = (pos != -1) ? RID.getRID(this.path.substring(0, pos)) : rootRID;

    return this.root;
  }

  /**
   * Get the name of the resource, designated by this RID. If the RID has query
   * parameter, these are discarded.
   *
   * @return the name of the resource, designated by this RID
   */
  public RID name() {
    if (null != this.name) {
      return this.name;
    }
    if (0 == this.pathlen) {
      return this;
    }

    if (this.endsWithSlash) {
      int pos = this.path.lastIndexOf(RID.PATH_SEPARATOR, this.pathlen - 2);
      this.name = RID.getRID(this.path.substring(pos + 1, this.pathlen - 1), null);
    }
    else {
      int pos = this.path.lastIndexOf(RID.PATH_SEPARATOR);
      this.name = (pos != -1) ? RID.getRID(this.path.substring(pos + 1), null) : this;
    }

    return this.name;
  }


  /**
   * Get the RID of the parent collection of this RID
   *
   * @return the RID of the parent collection
   */
  public RID parent() {
    if (null != this.parent) {
      return this.parent;
    }
    this.parent = removeName();
    return this.parent;
  }

  /**
   * Get the extension part of the resource name. Returns the empty string if
   * there is no extension.
   *
   * @return the extension part of the resource name
   */
  public String extension() {
    if (this.pathlen == 0) {
      return "";
    }

    RID n = name();
    int pos = n.path.lastIndexOf('.');
    if (pos != -1 && pos < n.pathlen - 1) {
      return n.path.substring(pos + 1);
    }
    else {
      return "";
    }
  }

  /**
   * Get a new RID with the toplevel collection removed
   *
   * @return RID with toplevel collection removed
   */
  public RID removeRoot() {
    if (this.pathlen == 0) {
      return this;
    }

    int start = this.isAbsolute ? 1 : 0;
    int pos = this.path.indexOf(RID.PATH_SEPARATOR, start);

    return (pos == -1 || pos == this.pathlen - 1) ? RID.emptyRID : RID.getRID(this.path.substring(pos), this.query);
  }

  /**
   * Remove the name of the resource, same as parent()
   *
   * @return parent of the resource
   */
  public RID removeName() {
    if (this.pathlen == 0) {
      return this;
    }

    int pos;
    if (this.endsWithSlash) {
      pos = this.path.lastIndexOf(RID.PATH_SEPARATOR, this.pathlen - 2);
    }
    else {
      pos = this.path.lastIndexOf(RID.PATH_SEPARATOR);
    }

    switch (pos) {
      case 0:
        return RID.rootRID;
      case -1:
        return RID.emptyRID;
      default:
        return RID.getRID(this.path.substring(0, pos));
    }
  }

  /**
   * Remove the (optional) extension of the resource name. Query parameters are
   * discarded.
   *
   * @return uri with extension of name removed
   */
  public RID removeExtension() {
    if (this.pathlen == 0 || this.endsWithSlash) {
      return this;
    }

    int spos = this.path.lastIndexOf(RID.PATH_SEPARATOR);
    int pos = this.path.lastIndexOf('.');

    if (pos > -1 && pos > spos) {
      return new RID(this.path.substring(0, pos));
    }
    else {
      return this;
    }
  }

  /**
   * Remove a possible trailing slash from the RID. Query parameters are
   * preserved.
   *
   * @return RID without trailing slash
   */
  public RID removeTrailingSlash() {
    return this.endsWithSlash ?
      RID.getRID(this.path.substring(0, this.pathlen - 1), this.query) : this;
  }

  /**
   * Return if RID is absolute, e.g. start with a slash.
   *
   * @return if RID is absolute path
   */
  public boolean isAbsolute() {
    return this.isAbsolute;
  }

  /**
   * Return if RID is root collection.
   *
   * @return if this is the root collection
   */
  public boolean isRoot() {
    return this.pathlen == 1 && this.isAbsolute;
  }

  /**
   * Determine if this RID is an ancestor (parent or parent's parent, etc.) of
   * the given child RID.
   *
   * @param child to test against
   * @return if this RID is ancestor of child
   */
  public boolean isAncestorOf(RID child) {
    if (this.pathlen < child.pathlen && this.isAbsolute == child.isAbsolute) {
      if (child.path.startsWith(this.path)) {
        return this.endsWithSlash || child.path.charAt(this.pathlen) == RID.PATH_SEPARATOR;
      }
    }
    return false;
  }

  /**
   * Determine if this RID is an ancestor (parent or parent's parent, etc.) of
   * the given child RID <em>or</em> the same.
   *
   * @param child to test against
   * @return if this RID is ancestor of child
   */
  public boolean isAncestorOfOrSelf(RID child) {
    if (this.equals(child)) {
      return true;
    }
    else if (this.pathlen < child.pathlen && this.isAbsolute == child.isAbsolute) {
      if (child.path.startsWith(this.path)) {
        return this.endsWithSlash || child.path.charAt(this.pathlen) == RID.PATH_SEPARATOR;
      }
    }
    return false;
  }

  /**
   * Return if this RID ends with a slash.
   *
   * @return if this RID ends with a slash
   */
  public boolean endsWithSlash() {
    return this.endsWithSlash;
  }

  /**
   * Concatenate this RID with the given RID. Treats this RID as if it ends with
   * a slash and the parameter RID as if it starts without a slash. <p>
   *
   * Note: do not use this method to add path segments, as the parameter is
   * parsed as legal RID, not as a legal RID segment; use {@link
   * RID#addPathSegment(String)} instead.
   *
   * @param uri to append to this
   * @return new RID as concatenation
   */
  public RID add(String uri) {
    if (this.pathlen == 0) {
      return RID.getRID(uri);
    }
    int ulen = uri.length();
    if (ulen == 0) {
      return this;
    }

    String query = null;
    int index = uri.lastIndexOf('?');
    if (index >= 0) {
      query = (index < ulen) ? uri.substring(index + 1) : "";
      uri = uri.substring(0, index);
      ulen = index;
    }

    StringBuffer sb = new StringBuffer(this.pathlen + ulen + 1);
    sb.append(this.path);

    if (this.endsWithSlash) {
      if (uri.charAt(0) == RID.PATH_SEPARATOR) {
        sb.setLength(this.pathlen - 1);
      }
    }
    else {
      if (ulen == 0 || (uri.charAt(0) != RID.PATH_SEPARATOR)) {
        sb.append(RID.PATH_SEPARATOR);
      }
    }
    sb.append(uri);

    return RID.getRID(sb.toString(), query);
  }

  /**
   * Concatenate this RID with the given RID. Treats this RID as if it ends with
   * a slash and the parameter RID as if it starts without a slash.
   *
   * @param uri to append to this
   * @return new RID as concatenation
   */
  public RID add(RID uri) {
    if (this.pathlen == 0) {
      return uri;
    }
    if (uri.pathlen == 0) {
      return RID.getRID(this.path, uri.query);
    }

    StringBuffer sb = new StringBuffer(this.pathlen + uri.pathlen + 1);
    sb.append(this.path);

    if (this.endsWithSlash) {
      if (uri.isAbsolute) {
        sb.setLength(this.pathlen - 1);
      }
    }
    else {
      if (!uri.isAbsolute) {
        sb.append(RID.PATH_SEPARATOR);
      }
    }
    sb.append(uri.path);

    return RID.getRID(sb.toString(), uri.query);
  }

  /**
   * Splits the RID into its path components
   *
   * @return List of {@link String}s
   */
  public List split() {
    if (this.pathlen == 0) {
      return Collections.EMPTY_LIST;
    }
    return splitString(this.path, this.pathlen, RID.PATH_SEPARATOR);
  }

  /**
   * Returns string representation of this RID
   *
   * @return this RID as string
   */
  public String toString() {
    if (this.uri == null) {
      if (this.query != null) {
        this.uri = this.path + '?' + this.query;
      }
      else {
        this.uri = this.path;
      }
    }
    return this.uri;
  }

  /**
   * Determine if this RID equals other.
   *
   * @param other RID
   * @return if this RID equals other
   */
  public boolean equals(RID other) {
    if (this == other) {
      return true;
    }
    if (other == null) {
      return false;
    }

    if (hashCode() != other.hashCode()) {
      return false;
    }

    if (this.pathlen == other.pathlen) {
      return this.path.equals(other.path) && equalsQuery(other);
    }
    else if (this.endsWithSlash) {
      if (!other.endsWithSlash && this.pathlen == other.pathlen + 1) {
        return this.path.startsWith(other.path) && equalsQuery(other);
      }
    }
    else if (other.endsWithSlash) {
      if (other.pathlen == this.pathlen + 1) {
        return other.path.startsWith(this.path) && equalsQuery(other);
      }
    }
    return false;
  }

  /**
   * Determine if this RID equals other.
   *
   * @param other RID (string representation)
   * @return if this RID equals other
   */
  public boolean equals(String other) {
    if (other != null) {
      String query = null;
      int index = other.lastIndexOf('?');
      if (index >= 0) {
        query = (index < other.length()) ? other.substring(index + 1) : "";
        other = other.substring(0, index);
      }
      return equals(RID.getRID(other, query));
    }
    return false;
  }

  /**
   * Determine if this RID equals other. Equality is defined for instances of
   * RID only.
   *
   * @param other
   * @return if this RID equals other
   */
  public boolean equals(Object other) {
    if (this == other) {
      return true;
    }

    if (other instanceof RID) {
      return equals((RID)other);
    }
    // sei: not a good idea, since this relation is not reflexiv
    //else if (other instanceof String) {
    //  return equals((String)other);
    //}
    else if (other instanceof URI) {
      throw new RuntimeException("comparing RID to instance of URI");
    }
    else {
      return false;
    }
  }

  public int hashCode() {
    if (this.hashCode == 0) {
      this.hashCode = (this.pathlen > 1 && this.endsWithSlash) ?
        this.path.substring(0, this.pathlen - 1).hashCode() : this.path.hashCode();
      if (this.query != null) {
        this.hashCode += this.query.hashCode();
      }
    }
    return this.hashCode;
  }

  /**
   * Return length of RID in string representation
   *
   * @return length of RID in string representation
   */
  public int length() {
    if (this.query != null) {
      return this.pathlen + 1 + this.query.length();
    }
    return this.pathlen;
  }

  private final static String RID_PATH_SEPARATOR = String.valueOf(RID.PATH_SEPARATOR);

  /**
   * Resolve relative RIDs
   *
   * @param uri An RID, which may be relative or absolute
   * @return TBD: Description of the outgoing return value
   */
  public RID resolveRelativeRID(String uri) {

    // absolute href (e.g. http:// c:\ c:/).
    if (this.pathlen > 0 && !(uri.startsWith(RID.RID_PATH_SEPARATOR)) && uri.indexOf(":/") < 0 && uri.indexOf(":\\", 1) < 0) {

      // relative uri. Append it to the base url
      StringBuffer sb = new StringBuffer(128);
      if (this.endsWithSlash) {
        sb.append(this.path);
      }
      else {
        sb.append(parent().toString());
        if (sb.charAt(sb.length() - 1) != RID.PATH_SEPARATOR) {
          sb.append(RID.PATH_SEPARATOR);
        }
      }

      RID tmp = RID.getRID(uri);
      String query = tmp.getQuery();
      sb.append(tmp.getPath());

      sb = HttpUrl.normalizePath(sb);

      return RID.getRID(sb.toString(), query);
//      //  remove all "./" where "." is a complete path segment
//      int index = -1;
//      while ((index = newUri.indexOf("/./")) != -1) {
//        newUri = path.substring(0, index+1).concat(newUri.substring(index+3));
//      }
//
//      // remove all "<segment>/../" where "<segment>" is a complete
//      // path segment not equal to ".."
//      index = -1;
//      int segIndex = -1;
//      String tempString = null;
//
//      while ((index = newUri.indexOf("/../")) > 0) {
//        tempString = newUri.substring(0, newUri.indexOf("/../"));
//        segIndex = tempString.lastIndexOf('/');
//        if (segIndex != -1) {
//          if (!tempString.substring(segIndex++).equals("..")) {
//            newUri = newUri.substring(0, segIndex).concat(newUri.substring(index+4));
//          }
//        }
//      }
//      return new RID(newUri);
    }
    else {
      return RID.getRID(uri);
    }
  }

  /**
   * Adds a path segment.
   *
   * @param segment String containing the new path segment
   * @return new RID
   * @throws WcmException if segment contains characters that aren't allowed in
   *      path segments
   */
  public RID addPathSegment(String segment)
    throws WcmException {
    if (segment.indexOf(RID.PATH_SEPARATOR) >= 0) {
      throw new WcmException("illegal character for RID path segment");
    }

    StringBuffer sb = new StringBuffer(this.pathlen + segment.length() + 1);
    sb.append(this.path);

    if (!this.endsWithSlash) {
      sb.append(RID.PATH_SEPARATOR);
    }
    sb.append(segment);

    return RID.getRID(sb.toString(), null);
  }

  /**
   * Create a new RID by setting the given parameters as query part of this RID.
   *
   * @param parameter to set to query
   * @return new RID with parameter in query
   * @deprecated as of NW04. Use setQueryParameter
   */
  public RID addQueryParameter(Properties parameter) {
    return setQueryParameter(parameter);
  }

  /**
   * Create a new RID by setting the given parameters as query part of this RID.
   *
   * @param parameter to set to query
   * @return new RID with parameter in query
   */
  public RID setQueryParameter(Properties parameter) {
    return RID.getRID(this.path, URICodec.EncodeQuery(parameter));
  }

  // --- Comparable interface ---

  public int compareTo(Object o) {
    String uri = this.path;
    String s2 = null;

    if (o instanceof String) {
      s2 = (String)o;
    }
    else if (o instanceof RID) {
      s2 = ((RID)o).path;
    }
    else {
      throw new java.lang.IllegalArgumentException("RID cannot be compared to " + o.getClass().getName());
    }

    final int o_len = s2.length();
    if (this.pathlen != o_len) {
      if ((this.pathlen > 1) && (o_len > 1)) {
        if (this.endsWithSlash) {
          uri = uri.substring(0, this.pathlen - 1);
        }
        if (s2.charAt(o_len - 1) == RID.PATH_SEPARATOR) {
          s2 = s2.substring(0, o_len - 1);
        }
      }
    }

    return uri.compareTo(s2);
  }

  // ----------------------------- private -----------------------------------

  private static List splitString(final String str, final int len, final int delim) {

    if (len == 0) {
      return Collections.EMPTY_LIST;
    }

    List names = new ArrayList(10);

    int i = (str.charAt(0) == delim) ? 1 : 0;
    int last = i;
    for (; i < len; ++i) {
      if (str.charAt(i) == delim) {
        names.add(str.substring(last, i));
        last = i + 1;
      }
    }

    if (last < i) {
      names.add(str.substring(last));
    }

    return names;
  }

  private RID() {
    this(null);
  }

  private boolean equalsQuery(RID other) {
    if (this.query != null) {
      return this.query.equals(other.query);
    }
    else {
      return other.query == null;
    }
  }
}
