/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.uri;

import java.net.MalformedURLException;

/**
 * A <a href="http://greenbytes.de/tech/webdav/rfc2518.html#rfc.section.6.4">
 * RFC2518</a> IUri implementation. <pre>
 *      lturl      := "opaquelocktoken:" UUID [ext]
 *      ext        := path
 * </pre> <p>
 *
 * Copyright (c) SAP AG 2001-2004
 * @author julian.reschke@greenbytes.de
 * @version $Id: OpaqueLockTokenUri.java,v 1.2 2004/12/28 17:58:55 jre Exp $
 */

public class OpaqueLockTokenUri implements IUri {

  private final static String SCHEME = "opaquelocktoken";
  private final String remainder;
  private int hashCode;

  public OpaqueLockTokenUri(String remainder)
    throws IllegalArgumentException {
    this.remainder = remainder;
  }

  protected IUri newInstance(String remainder)
    throws IllegalArgumentException {
    return new OpaqueLockTokenUri(remainder);
  }

  public String getRemainder() {
    return remainder;
  }

  public String getScheme() {
    return SCHEME;
  }

  public int hashCode() {
    if (this.hashCode == 0) {
      this.hashCode = this.toString().hashCode();
    }
    return this.hashCode;
  }

  public String toExternalForm() {
    return SCHEME + ":" + remainder;
  }

  public String toString() {
    return toExternalForm();
  }

  public IUri resolve(IUriReference ref)
    throws IllegalArgumentException {
    throw new IllegalArgumentException("resolution not supported for opaquelocktoken scheme");
  }

  public void checkSyntax() throws MalformedURLException {
    
    if (this.remainder.length() < 32) {
      throw new MalformedURLException("opaquelocktoken: scheme-specific part must start with UUID");
    }
    
    for (int i = 0; i < 32; i++) {
      char c = this.remainder.charAt(i);
      
      switch (i) {
        case 8:
        case 13:
        case 18:
        case 23: if (c != '-') {
            throw new MalformedURLException("opaquelocktoken: character at position " + i + " must be a minus");
          }
          break;
          
        default:
          if (! ((c >= '0' && c <= '9') || (c >= 'a' && c <= 'f') || (c >= 'A' && c <= 'F'))) {
            throw new MalformedURLException("opaquelocktoken: character at position " + i + " must be a hex digit");
          }
          break;
      }
    }
    
    // TODO: check legal path characters for remainder
    
  }
  
  public boolean equals(Object other) {
    if (other instanceof IUri) {
      return this.equals((IUri)other);
    }
    return false;
  }

  public boolean equals(IUri other) {
    if (other == null) {
      return false;
    }
    if (other == this) {
      return true;
    }
    return SCHEME.equalsIgnoreCase(other.getScheme())
       && this.remainder.equals(other.getRemainder());
  }
  
  
  
}
