/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.uri;

/**
 * A <a href="http://www.ietf.org/rfc/rfc2396.txt">RFC2396</a> URI reference
 * interface. <p>
 *
 * urireference ::= refpart fragment refpart ::= uri | relativeref fragment ::=
 * '#' uri-characters | empty relativeref ::= hierarchical_part | opaque_part
 * hierarchical_part ::= pathsegment query | [ netpath ] abspath query netpath
 * ::= '//' authority <p>
 *
 * The opaque part is not directly mapped to this interface, but is the
 * composition of path + query. This is due to the fact that at construction
 * time of a relatice uri reference, it is not clear if the reference is part of
 * an opaque or hierarchical uri scheme. Base uri classes, not hierarchical,
 * which have to resolve references, have to concatenate path and query to get
 * the opaque part of the reference. <p>
 *
 * There are degenrate cases of relative uri reference with a scheme, but
 * without authority component. See RFC 2396, page 32 as an example. The use is
 * discouraged. Copyright (c) SAP AG 2001-2002
 *
 * @author julian.reschke@greenbytes.de
 * @author stefan.eissing@greenbytes.de
 * @version $Id: IUriReference.java,v 1.4 2001/11/22 14:16:42 sei Exp $
 */

public interface IUriReference {

  /**
   * Create a new uri reference by appending the given path to any existing path
   * in the uri reference. The path <b>must</b> be uri encoded or the created
   * reference will not be valid. The given path is appended by prefixing a
   * slash character, if necessary.
   *
   * @param path uri encoded path to be appended
   * @return an uri reference with path appended
   * @exception UnsupportedOperationException Exception raised in failure
   *      situation
   * @throw UnsupportedOperationException if reference holds non-hierarchical
   *      uri scheme
   */
  public IUriReference appendPath(String path)
    throws UnsupportedOperationException;

  /**
   * Gets a new URI reference resulting from setting the query of this URI
   * reference to the given query. The query string must be in RFC 2396 encoded
   * form. See {@link UriQuery} for manipulation of query strings.
   *
   * @param query the encoded query string to be set (may be <code>null</code> )
   * @return an uri reference with query set to given query
   * @throw UnsupportedOperationException if reference holds uri scheme without
   *      queries
   */
  public IUriReference setQuery(String query);

  /**
   * Get the fragment identifier (without '#') of the uri
   *
   * @return fragment identifier or empty string
   */
  public String getFragmentIdentifier();

  /**
   * Get the scheme of the reference
   *
   * @return scheme of reference or null
   */
  public String getScheme();

  /**
   * Get the authority of the reference
   *
   * @return authority of reference or null
   */
  public String getAuthority();

  /**
   * Get the path of the reference
   *
   * @return path of reference or empty string
   */
  public String getPath();

  /**
   * Get the query of the reference
   *
   * @return query of reference or null
   */
  public String getQuery();

  /**
   * Get the uri of the reference if absolute
   *
   * @return uri of reference or null
   */
  public IUri getUri();

  /**
   * Determine if the reference is absolute
   *
   * @return if reference is absolute
   */
  public boolean isAbsolute();

  /**
   * With base Uri, resolve this reference to an absolute reference. If this
   * reference is already absolute, this is the identity operation.
   *
   * @param base uri to resolve reference with
   * @return absolute uri reference
   */
  public IUriReference resolveWith(IUri base);

  /**
   * Returns a string representation according to RFC 2396 of this uri
   * reference.
   *
   * @return string in uri reference format
   */
  public String toExternalForm();

}// interface IUriReference
