/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.uri;

/**
 * A <a href="http://www.ietf.org/rfc/rfc2396.txt">RFC2396</a> IUri
 * implementation. <pre>
 *     Uri ::= scheme ':' opaque_part
 * </pre> <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author julian.reschke@greenbytes.de
 * @author stefan.eissing@greenbytes.de
 * @version $Id: GenericUri.java,v 1.3 2001/11/22 14:16:41 sei Exp $
 */

public class GenericUri implements IUri {

  private final String m_scheme;
  private final String m_remainder;
  private String m_full;
  private int m_hashCode;

  public GenericUri(String scheme, String remainder)
    throws IllegalArgumentException {

    if (scheme == null) {
      throw new IllegalArgumentException("scheme must be non-null");
    }

    m_scheme = scheme.toLowerCase();
    m_remainder = (remainder != null) ? remainder : "";
  }

  /**
   * Get the scheme of this Uri (without ':').
   *
   * @return scheme of this Uri
   */
  public String getScheme() {
    return m_scheme;
  }

  /**
   * Get the remainder of this Uri (opaque part after scheme without ':').
   *
   * @return remainder of this Uri
   */
  public String getRemainder() {
    return m_remainder;
  }

  /**
   * Resolve the uri reference in the context of this Uri.
   *
   * @param ref reference to resolve
   * @return resolved reference as absolute uri without fragment identifier
   * @exception IllegalArgumentException Exception raised in failure situation
   */
  public IUri resolve(IUriReference ref)
    throws IllegalArgumentException {
    if (ref.isAbsolute()) {
      return ref.getUri();
    }
    else {
      String rscheme = ref.getScheme();
      if (rscheme != null && !m_scheme.equalsIgnoreCase(rscheme)) {
        throw new IllegalArgumentException("scheme does not match");
      }
      String rauth = ref.getAuthority();
      if (rauth != null) {
        throw new IllegalArgumentException("reference has authority");
      }
    }

    String rquery = ref.getQuery();
    return new GenericUri(m_scheme, ref.getPath() + (rquery != null ? rquery : ""));
  }

  public boolean equals(Object other) {
    if (other instanceof IUri) {
      return this.equals((IUri)other);
    }
    return false;
  }

  public boolean equals(IUri other) {
    if (other == null) {
      return false;
    }
    if (other == this) {
      return true;
    }
    return m_scheme.equalsIgnoreCase(other.getScheme())
       && m_remainder.equals(other.getRemainder());
  }

  public int hashCode() {
    if (m_hashCode == 0) {
      m_hashCode = this.toString().hashCode();
    }
    return m_hashCode;
  }

  public String toString() {
    if (m_full == null) {
      m_full = m_scheme + ":" + m_remainder;
    }
    return m_full;
  }

  public String toExternalForm() {
    return toString();
  }

  /**
   * Construct a new IUri with the scheme of this and the given remainder. used
   * when generic uri resolves references. This method should be overridden by
   * subclasses to construct their own class.
   *
   * @param remainder new opaque part of uri
   * @return new IUri instance
   * @exception IllegalArgumentException Exception raised in failure situation
   * @throws java.lang.IllegalArgumentException when remainder is not legal
   */
  protected IUri newInstance(String remainder)
    throws IllegalArgumentException {
    return new GenericUri(m_scheme, remainder);
  }
}
