/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.uri;

import com.sapportals.wcm.util.base64.Base64Decoder;

import java.util.StringTokenizer;

/**
 * A <a href="http://www.ietf.org/rfc/rfc2397.txt">RFC2397</a> IUri
 * implementation. <pre>
 *      dataurl    := "data:" [ mediatype ] [ ";base64" ] "," data
 *      mediatype  := [ type "/" subtype ] *( ";" parameter )
 *      data       := *urlchar
 *      parameter  := attribute "=" value
 * </pre> <p>
 *
 * Copyright (c) SAP AG 2001-2003
 *
 * @author julian.reschke@greenbytes.de
 * @version $Id: DataUri.java,v 1.5 2003/02/17 14:24:04 jre Exp $
 */

public class DataUri extends GenericUri {

  // private static final com.sap.tc.logging.Location log = com.sap.tc.logging.Location.getLocation(DataUri.class.getName());

  private String typeAndSubType;
  private String encoding;
  private byte[] data = null;
  private boolean isBase64 = false;

  public DataUri(String remainder)
    throws IllegalArgumentException {
    super("data", remainder);

    int pComma = remainder.indexOf(",");
    if (pComma < 0) {
      throw new IllegalArgumentException();
    }

    String prefix = remainder.substring(0, pComma);

    this.typeAndSubType = "text/plain";
    this.encoding = "US-ASCII";

    StringTokenizer st = new StringTokenizer(prefix, ";");
    if (st.hasMoreTokens()) {
      this.typeAndSubType = st.nextToken().trim();
      while (st.hasMoreTokens()) {
        String param = st.nextToken().trim();

        if (param.equalsIgnoreCase("base64")) {
          this.isBase64 = true;
        }
        else if (param.toLowerCase().startsWith("charset=")) {
          this.encoding = param.substring("charset=".length());
        }
      }
    }
  }

  public byte[] getData() {
    if (this.data == null) {

      String remainder = getRemainder();
      int pComma = remainder.indexOf(",");
      String data = remainder.substring(pComma + 1);

      if (this.isBase64) {
        try {
          this.data = Base64Decoder.decode(data).getBytes("8859_1");
        }
        catch (java.io.UnsupportedEncodingException ex) {
            //$JL-EXC$          
          throw new RuntimeException();
        }
      }
      else {
        byte[] tmp = new byte[data.length()];
        int j = 0;
        int i = 0;

        while (i < data.length()) {
          char c = data.charAt(i++);

          if (c == '%') {
            try {
              tmp[j++] = (byte)Integer.parseInt(data.substring(i, i + 2), 16);
              i += 2;
            }
            catch (java.lang.IndexOutOfBoundsException ex) {
            //$JL-EXC$              
              throw new IllegalArgumentException(ex.getMessage());
            }
          }
          else {
            tmp[j++] = (byte)c;
          }

        }

        this.data = new byte[j];
        System.arraycopy(tmp, 0, this.data, 0, j);
      }
    }

    return this.data;
  }

  public String getEncoding() {
    return this.encoding;
  }

  public String getContentType() {
    return this.typeAndSubType;
  }

  protected IUri newInstance(String remainder)
    throws IllegalArgumentException {
    return new DataUri(remainder);
  }
}
