/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.systemconfig;

import com.sapportals.portal.security.usermanagement.*;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.util.usermanagement.WPUMFactory;

import java.util.*;

/**
 * The (default) implementation of an <code>ISystemPrincipal</code> . <br>
 *
 */
public class SystemPrincipal implements ISystemPrincipal {

  private final IUMPrincipal m_principal;
  private final boolean m_isLockBreaker, m_hasServiceAcls;
  private final Set m_mappedPermissions;
  private final boolean m_canChangePermissions;

  /**
   * Construct a SystemPrincipal
   *
   * @param principalId TBD: Description of the incoming method parameter
   * @param principalType TBD: Description of the incoming method parameter
   * @param isLockBreaker TBD: Description of the incoming method parameter
   * @param hasServiceAcls TBD: Description of the incoming method parameter
   * @param mappedPermissions TBD: Description of the incoming method parameter
   * @param canChangePermissions TBD: Description of the incoming method
   *      parameter
   * @exception WcmException Exception raised in failure situation
   * @exception UserManagementException Exception raised in failure situation
   */
  public SystemPrincipal(String principalId, int principalType, boolean isLockBreaker, boolean hasServiceAcls,
    Set mappedPermissions, boolean canChangePermissions)
    throws WcmException, UserManagementException {

    // get the principal from the usermanagement
    m_principal = getUMPrincipal(principalId, principalType);
    if (m_principal == null) {
      throw new WcmException("system principal " + principalId + " does not exist - check WCM configuration");
    }

    // assign member variables
    m_isLockBreaker = isLockBreaker;
    m_hasServiceAcls = hasServiceAcls;
    m_mappedPermissions = mappedPermissions;
    m_canChangePermissions = canChangePermissions;
  }

  /**
   * @return the principal from the usermanagement
   */
  public IUMPrincipal getPrincipal() {
    return m_principal;
  }

  /**
   * @return true iff the principal is allowed to break foreign locks on
   *      resources
   */
  public boolean isLockBreaker() {
    return m_isLockBreaker;
  }

  /**
   * @return true iff the principal is should have all service acls    
   */
  public boolean hasServiceAcls() {
    return m_hasServiceAcls;
  }
  /**
   * @return a list of strings containing the names of CM framework permissions
   *      which are granted to the principal on each resource
   */
  public Set getMappedPermissions() {
    return m_mappedPermissions;
  }

  /**
   * @return true iff the principal is allowed to change permissions on each
   *      resource
   */
  public boolean canChangePermissions() {
    return m_canChangePermissions;
  }

  /**
   * @return aclPermissions
   * @deprecated as of NW04.
   */
  public List getAclPermissions() {
    return new LinkedList();
  }

  /**
   * @return aclOwner
   * @deprecated as of NW04.
   */
  public boolean isAclOwner() {
    return false;
  }

  /**
   * @return appLogViewer
   * @deprecated as of NW04.
   */
  public boolean isAppLogViewer() {
    return false;
  }

  /**
   * @param id TBD: Description of the incoming method parameter
   * @param type TBD: Description of the incoming method parameter
   * @return a principal object from the usermanagament
   * @exception UserManagementException Exception raised in failure situation
   */
  private IUMPrincipal getUMPrincipal(String id, int type)
    throws UserManagementException {
    switch (type) {
      case IUMPrincipal.IUSER:
        return getUser(id);
      case IUMPrincipal.IGROUP:
        return getGroup(id);
      case IUMPrincipal.IROLE:
        return getRole(id);
    }

    return null;
  }

  /**
   * @param id TBD: Description of the incoming method parameter
   * @return a user object from the usermanagement
   * @exception UserManagementException Exception raised in failure situation
   */
  private IUser getUser(String id)
    throws UserManagementException {
    IUserFactory userFactory = WPUMFactory.getUserFactory();
    return userFactory == null ? null : userFactory.getUser(id);
  }

  /**
   * @param id TBD: Description of the incoming method parameter
   * @return a group object from the usermanagement
   * @exception UserManagementException Exception raised in failure situation
   */
  private IGroup getGroup(String id)
    throws UserManagementException {
    IGroupFactory groupFactory = WPUMFactory.getGroupFactory();
    return groupFactory == null ? null : groupFactory.getGroup(id);
  }

  /**
   * @param id TBD: Description of the incoming method parameter
   * @return a role object from the usermanagement
   * @exception UserManagementException Exception raised in failure situation
   */
  private IRole getRole(String id)
    throws UserManagementException {
    IRoleFactory roleFactory = WPUMFactory.getRoleFactory();
    return roleFactory == null ? null : roleFactory.getRole(id);
  }
}
