/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.systemconfig;

import java.security.AccessController;
import java.security.PrivilegedExceptionAction;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import com.sapportals.config.fwk.Configuration;
import com.sapportals.config.fwk.IConfigClientContext;
import com.sapportals.config.fwk.IConfigManager;
import com.sapportals.config.fwk.IConfigPlugin;
import com.sapportals.config.fwk.IConfigurable;
import com.sapportals.config.fwk.IMutableConfigurable;
import com.sapportals.portal.security.usermanagement.IGroup;
import com.sapportals.portal.security.usermanagement.IRole;
import com.sapportals.portal.security.usermanagement.IUMPrincipal;
import com.sapportals.portal.security.usermanagement.IUser;
import com.sapportals.wcm.util.config.ConfigCrutch;
import com.sapportals.wcm.util.string.StrUtil;
import com.sapportals.wcm.WcmException;


/**
 * Main entry point for retrieving the <code>ISystemPrincipal</code>s. <br>
 */
public class SystemConfig {

  // =========
  // CONSTANTS ================================================================
  // =========

  private final static String ACL_PERMISSION_READ = "read";
  private final static String ACL_PERMISSION_WRITE = "write";
  private final static String ACL_PERMISSION_READWRITE = "readwrite";
  private final static String ACL_PERMISSION_FULLCONTROL = "fullcontrol";

  private final static String PERMISSION_READ = "read";
  private final static String PERMISSION_WRITE = "write";
  private final static String PERMISSION_DELETE = "delete";

  private final static String CONFIG_UPDATED = "updated_to_fwk_permissions";
  private final static String CONFIG_ACL_PERMISSIONS = "acl_permissions";
  private final static String CONFIG_ACL_OWNER = "acl_owner";
  private final static String CONFIG_PERMISSIONS = "permissions";
  private final static String CONFIG_CHANGE_PERMISSIONS = "change_permissions";
  private final static String CONFIG_LOCK_BREAKER = "lock_breaker";
  private final static String CONFIG_SERVICEACLS = "serviceacls";
  private final static String CONFIG_PRINCIPAL_NAME = "principal_name";

  private final static String CFG_PLUGIN_CM_UTILITIES_SYSTEM_CONFIG = "/cm/utilities/system_config";
  private final static String CFG_PLUGIN_CM_UTILITIES_SYSTEM_CONFIG_MAPPINGS = "/cm/utilities/system_config/permission_mappings";

  private final static String SYSTEM_USERS_CONFIG_CLASS = "SystemUser";
  private final static String SYSTEM_GROUPS_CONFIG_CLASS = "SystemGroup";
  private final static String SYSTEM_ROLES_CONFIG_CLASS = "SystemRole";

  private final static char CFGFWK_ENUM_LIST_SEPARATOR = ',';
  private final static char CFGFWK_STRING_LIST_SEPARATOR = ',';

  private final static String PERM_CACHE_IN_USER = "--km-permissions--";
  private final static String CACL_CACHE_IN_USER = "--km-change-permission--";
  private final static String HASSACL_CACHE_IN_USER = "--km-has-service-acls--";

  private final static String UNDEFINED_PRINCIPAL_NAME = "undefined";


  // ================
  // STATIC VARIABLES =========================================================
  // ================

  private final static com.sap.tc.logging.Location s_log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.systemconfig.SystemConfig.class);

  private static SystemConfig s_systemConfig = null;


  // ==================
  // INSTANCE VARIABLES =======================================================
  // ==================

  private SystemPrincipalList m_systemUsers;
  private SystemPrincipalList m_systemGroups;
  private SystemPrincipalList m_systemRoles;
  private SystemPrincipalList m_systemPrincipals;

  private final HashMap m_permissions = new HashMap();

  private final static long containsPrincipalCacheLifetime = 60000;// ms ~ 1 min
  static long containsPrincipalCacheLastRefresh = 0;
  static Hashtable containsPrincipalCache = new Hashtable();


  // ===============
  // PRIVATE METHODS ==========================================================
  // ===============

  // ------------------------------------------------------------------------
  /**
   * get the hashkey for a (IUMPrincipal,IUMPrincipal) tuple
   */
  private String getPrincipalsHashKey(IUMPrincipal p1,
                                      IUMPrincipal p2) {

    StringBuffer buffer = new StringBuffer(50);

    // avoid unnessecary object generation, so convert int -> char

    char p1type = '-';
    switch (p1.getType()) {
      case IUMPrincipal.IROLE:  p1type = 'R'; break;
      case IUMPrincipal.IGROUP: p1type = 'G'; break;
      case IUMPrincipal.IUSER:  p1type = 'U'; break;
    }

    char p2type = '-';
    switch (p2.getType()) {
      case IUMPrincipal.IROLE:  p2type = 'R'; break;
      case IUMPrincipal.IGROUP: p2type = 'G'; break;
      case IUMPrincipal.IUSER:  p2type = 'U'; break;
    }

    buffer.append(p1type);
    buffer.append(p1.getId());
    buffer.append(p2type);
    buffer.append(p2.getId());

    return buffer.toString();

  }


  // ------------------------------------------------------------------------
  /**
   * check if a given principal contains or is equal to another principal 
   * @return true if principal p1 equals or contains principal p2 (as
   *        group or role member, recursive check)
   */
  private boolean containsPrincipal(IUMPrincipal p1,
                                    IUMPrincipal p2) {

    if (   ( p1 == p2 )
        || ( p1.equals(p2) )
       ) {
      // check the cheap case first
      return true;
    }

    // check if cache is expired and clear if necessary
    long now = System.currentTimeMillis();
    if (now - containsPrincipalCacheLastRefresh > containsPrincipalCacheLifetime) {
      containsPrincipalCache.clear();
      containsPrincipalCacheLastRefresh = now;
    }

    String hashkey = getPrincipalsHashKey(p1, p2);
    Boolean cachedResult = (Boolean)containsPrincipalCache.get(hashkey);
    if (cachedResult != null) {
      return cachedResult.booleanValue();
    }

    // result not found in cache, so check and put into cache    
    boolean result = false;
    switch (p2.getType()) {
      case IUMPrincipal.IUSER:
        if (   (p1.getType() == IUMPrincipal.IGROUP && ((IGroup)p1).containsUser(p2.getId()))
            || (p1.getType() == IUMPrincipal.IROLE && ((IRole)p1).containsUser(p2.getId()))
           ) {
          result = true;
        }
        break;
      case IUMPrincipal.IGROUP:
        if (   (p1.getType() == IUMPrincipal.IGROUP && ((IGroup)p1).containsGroup(p2.getId()))
            || (p1.getType() == IUMPrincipal.IROLE && ((IRole)p1).containsGroup(p2.getId()))
           ) {
          result = true;
        }
        break;
    }
    containsPrincipalCache.put(hashkey, result ? Boolean.TRUE : Boolean.FALSE);

    return result;
    
  }


  // ------------------------------------------------------------------------
  /**
   * get the system principals for a given type from the given configurable.
   * @return a list of system principals of a specific type
   * @exception Exception Exception raised in failure situation
   */
  private List getSystemPrincipals(IConfigurable[] configurables,
                                   String configClass,
                                   int type)
                            throws Exception {

    // create result list
    LinkedList result = new LinkedList();

    // loop over the configurables
    for (int i = 0; i < configurables.length; i++) {
      // check if config class matches the requested type
      if (configClass.equals(configurables[i].getConfigClass().getName())) {
        // get system principal ID
        String id = configurables[i].getPropertyValue(CONFIG_PRINCIPAL_NAME);
        try {
          SystemPrincipal principal = new SystemPrincipal(
                                                         id,
                                                         type,
                                                         getLockBreakerFlag(configurables[i]),
                                                         getServiceAclsFlag(configurables[i]),
                                                         getMappedPermissions(configurables[i]),
                                                         getChangePermissionsFlag(configurables[i])
                                                         ); 
          // create system principal object
          result.add(principal);
        }
        catch (Exception e) {
            //$JL-EXC$          
          s_log.errorT("getSystemPrincipals(283)", "Failed to create system principal - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
        }
      }
    }

    return result;
    
  }


  // ------------------------------------------------------------------------
  /**
   * get the system principals for a given type from config.
   * @return a list of system principals for the specified type
   * @exception Exception Exception raised in failure situation
   */
  private List getSystemPrincipalsFromConfig(String configClass,
                                             int type)
                                      throws Exception {

    // retrieve system principals configurables from configuration framework
    IConfigClientContext context = IConfigClientContext.createContext(ConfigCrutch.getConfigServiceUser());
    IConfigManager cfgManager = Configuration.getInstance().getConfigManager(context);
    IConfigPlugin plugin = cfgManager.getConfigPlugin(CFG_PLUGIN_CM_UTILITIES_SYSTEM_CONFIG);
    IConfigurable[] configurables = plugin.getConfigurables();

    // generate system principal list from the configurables
    return getSystemPrincipals(configurables, configClass, type);
    
  }


  // ------------------------------------------------------------------------
  /**
   * get the mapped framework permissions from a given configurable.
   * @return the CM framework permissions of a system principal as string array
   * @exception Exception Exception raised in failure situation
   */
  private HashSet getMappedPermissions(IConfigurable configurable)
                                throws Exception {

    // get the system config permissions in a string
    String permissionsString = configurable.getPropertyValue(CONFIG_PERMISSIONS);

    // if no permissions are assigned to the system principal, return an empty array (no mapped CM framework permissions)
    if (permissionsString == null) {
      return new HashSet();
    }

    // make array from string components (system config permissions)
    String[] permissionsArray = StrUtil.extractStringComponentsToArray(permissionsString, CFGFWK_ENUM_LIST_SEPARATOR, true, '\\');

    // get the permission mappings (a system config permissions can be mapped to multiple CM framework permissions)
    HashMap mappings = getPermissionMappings();

    // create a set to collect the mapped permissions
    HashSet result = new HashSet();

    // insert the mappings to the set (for all system config permissions)
    for (int i = 0; i < permissionsArray.length; i++) {
      // get the mappings for the current system config permissions
      String[] mappedPermissions = (String[])mappings.get(permissionsArray[i]);
      if (mappedPermissions != null) {
        // insert the mapped permissions (CM framework permissions) to the set with all mapped permissions
        for (int k = 0; k < mappedPermissions.length; k++) {
          result.add(mappedPermissions[k]);
        }
      }
    }

    return result;

  }


  // ------------------------------------------------------------------------
  /**
   * get the mapping for system config permissions to CM framework
   * permissions.
   * @return a hash map containing the mappings of system config permissions to
   *      CM framework permissions as they are configured in the configuration
   *      framework key: system config permission name (string) value: mapped CM
   *      framework permissions (string array)
   * @exception Exception Exception raised in failure situation
   */
  private HashMap getPermissionMappings()
                                 throws Exception {

    final String CONFIG_FROM = "from";
    final String CONFIG_TO = "to";

    // create result
    HashMap result = new HashMap();

    // get all configurables (configured permission mappings)
    IConfigClientContext context = IConfigClientContext.createContext(ConfigCrutch.getConfigServiceUser());
    IConfigManager cfgManager = Configuration.getInstance().getConfigManager(context);
    IConfigPlugin plugin = cfgManager.getConfigPlugin(CFG_PLUGIN_CM_UTILITIES_SYSTEM_CONFIG_MAPPINGS);
    IConfigurable[] configurables = plugin.getConfigurables();

    // loop over the mappings
    for (int i = 0; i < configurables.length; i++) {
      // get the mapping values
      String from = configurables[i].getPropertyValue(CONFIG_FROM);
      String to = configurables[i].getPropertyValue(CONFIG_TO);
      // put the mapping to the result
      if (from != null && to != null) {
        String[] toArray = StrUtil.extractStringComponentsToArray(to, CFGFWK_STRING_LIST_SEPARATOR, true);
        result.put(from, toArray);
      }
    }

    return result;
    
  }


  // ------------------------------------------------------------------------
  /**
   * get the 'change_permission' attribute value from a given configurable.
   * @return the value of the change_permissions attribute of the configurable
   * @exception Exception Exception raised in failure situation
   */
  private boolean getChangePermissionsFlag(IConfigurable configurable)
                                    throws Exception {
    String changePermissionFlag = configurable.getPropertyValue(CONFIG_CHANGE_PERMISSIONS);
    if (changePermissionFlag == null) {
      return false;
    }
    return changePermissionFlag.equals(Boolean.TRUE.toString());
  }


  // ------------------------------------------------------------------------
  /**
   * get the 'lock_breaker' attribute value from a given configurable.
   * @return the value of the lock_breaker attribute of the configurable
   * @exception Exception Exception raised in failure situation
   */
  private boolean getLockBreakerFlag(IConfigurable configurable)
                              throws Exception {
    String lockBreakerFlag = configurable.getPropertyValue(CONFIG_LOCK_BREAKER);
    if (lockBreakerFlag == null) {
      return false;
    }
    return lockBreakerFlag.equals(Boolean.TRUE.toString());       
  }

  /**
   * get the 'lock_breaker' attribute value from a given configurable.
   * @return the value of the lock_breaker attribute of the configurable
   * @exception Exception Exception raised in failure situation
   */
  private boolean getServiceAclsFlag(IConfigurable configurable)
                                     throws Exception {
    String serviceAclsFlag = configurable.getPropertyValue(CONFIG_SERVICEACLS);
    if (serviceAclsFlag == null) {
      return false;
    }
    return serviceAclsFlag.equals(Boolean.TRUE.toString());    
  }

  // ------------------------------------------------------------------------
  /**
   * Map the old acl_permissions contained in aclPermissionsString (separated by
   * colons) to the new system principal permissions
   *
   * @param aclPermissionsString TBD: Description of the incoming method
   *      parameter
   * @param isOwner TBD: Description of the incoming method parameter
   * @return a string containing the system principal permissions separated by
   *      colons
   */
  private String mapAclPermissionsToSystemConfigPermissions(String aclPermissionsString,
                                                            boolean isOwner) {

    // create a set to collect the mapped permissions
    HashSet permissions = new HashSet();

    if (isOwner) {
      // if the system principal is an owner, all permissions are granted
      permissions.add(PERMISSION_READ);
      permissions.add(PERMISSION_WRITE);
      permissions.add(PERMISSION_DELETE);
    }
    else {
      // get array with acl_permissions
      String[] aclPermissionsArray = StrUtil.extractStringComponentsToArray(aclPermissionsString, CFGFWK_ENUM_LIST_SEPARATOR, true);

      // map every single acl_permission to the corresponding framework permissions
      for (int i = 0; i < aclPermissionsArray.length; i++) {
        if (aclPermissionsArray[i].equals(ACL_PERMISSION_READ)) {
          permissions.add(PERMISSION_READ);
        }
        if (aclPermissionsArray[i].equals(ACL_PERMISSION_WRITE)) {
          permissions.add(PERMISSION_WRITE);
        }
        if (aclPermissionsArray[i].equals(ACL_PERMISSION_READWRITE)) {
          permissions.add(PERMISSION_READ);
          permissions.add(PERMISSION_WRITE);
        }
        if (aclPermissionsArray[i].equals(ACL_PERMISSION_FULLCONTROL)) {
          permissions.add(PERMISSION_READ);
          permissions.add(PERMISSION_WRITE);
          permissions.add(PERMISSION_DELETE);
        }
      }
    }

    // generate the result string
    StringBuffer result = new StringBuffer();
    Iterator it = permissions.iterator();
    while (it.hasNext()) {
      if (result.length() > 0) {
        result.append(CFGFWK_ENUM_LIST_SEPARATOR);
      }
      result.append((String)it.next());
    }

    return result.toString();

  }


  // ------------------------------------------------------------------------
  /**
   * Update old versions of the configuration. <br>
   * 1. map old acl-permissions to new permissions property (containing the
   *    permissions read, write, and delete)
   * 2. map old acl-owner flag to new change_permissions property
   * 3. if the principal_name property is "undefined" copy the configurable
   *    name to the principal_name property
   * @exception Exception Exception raised in failure situation
   */
  private void migrateSystemConfiguration()
                                   throws Exception {

    final long CONFIG_LOCK_TIMEOUT = 10L * 60L * 1000L; // 10 minutes
    final long CONFIG_WAIT_TIMEOUT = 60L * 1000L; // 1 minute

    // get a config manager
    IConfigClientContext context = IConfigClientContext.createContext(ConfigCrutch.getConfigServiceUser());
    IConfigManager configManager = com.sapportals.config.fwk.Configuration.getInstance().getConfigManager(context);

    // get and lock the system config plugin
    IConfigPlugin plugin = configManager.getAndLockConfigPlugin(CFG_PLUGIN_CM_UTILITIES_SYSTEM_CONFIG, CONFIG_LOCK_TIMEOUT, CONFIG_WAIT_TIMEOUT);
    try {
      // get all configurables (configured system principals)
      IMutableConfigurable[] configurables = plugin.getConfigurables();
      // loop over the system principals
      for (int i = 0; i < configurables.length; i++) {
        // get the system principal property which tells us whether the principal already has been updated
        Object updatedProperty = configurables[i].getPropertyValueAsObject(CONFIG_UPDATED);
        // if the property is unset or false, update the configurable
        if (updatedProperty == null || !((Boolean)updatedProperty).booleanValue()) {
          // get the old acl_permissions property
          Object aclPermissionsPropery = configurables[i].getPropertyValueAsObject(CONFIG_ACL_PERMISSIONS);
          // set the old acl_owner flag as change_permissions flag
          boolean isOwner = false;
          Object aclOwnerPropery = configurables[i].getPropertyValueAsObject(CONFIG_ACL_OWNER);
          if (aclOwnerPropery != null && ((Boolean)aclOwnerPropery).booleanValue()) {
            configurables[i].setPropertyValue(CONFIG_CHANGE_PERMISSIONS, "true");
            isOwner = true;
          }
          // if the property exists map the acl_permissions to the new permissions which can only contain read, write, and delete
          if (aclPermissionsPropery != null) {
            String aclPermissionsString = configurables[i].getPropertyValue(CONFIG_ACL_PERMISSIONS);
            if (aclPermissionsString != null && aclPermissionsString.length() > 0) {
              configurables[i].setPropertyValue(CONFIG_PERMISSIONS, mapAclPermissionsToSystemConfigPermissions(aclPermissionsString, isOwner));
            }
          }
          // set the updated property to true
          configurables[i].setPropertyValue(CONFIG_UPDATED, "true");
          // commit the changes of the configurable
          plugin.replaceConfigurable(configurables[i]);
        }
        // if the principal_name property is "undefined" copy the configurable name to the principal_name property
        if (UNDEFINED_PRINCIPAL_NAME.equals(configurables[i].getPropertyValue(CONFIG_PRINCIPAL_NAME))) {
          // set the new value of the principal_name property to true
          configurables[i].setPropertyValue(CONFIG_PRINCIPAL_NAME, configurables[i].getIdValue());
          // commit the changes of the configurable
          plugin.replaceConfigurable(configurables[i]);
        }
      }
    }
    finally {
      plugin.unlock();
    }

  }


  // ------------------------------------------------------------------------
  /**
   * 
   */
  private SystemPrincipalList addPrincipalToList(SystemPrincipal newPrincipal,
                                                 SystemPrincipalList oldList) {
  
   List newList = new LinkedList();
   ISystemPrincipalListIterator i = oldList.iterator();
   while( i.hasNext() ) {
     ISystemPrincipal principal = i.next();
     newList.add(principal);
   }
   newList.add(newPrincipal);
   return new SystemPrincipalList(newList);
   
  }
  
  
  // ============
  // CONSTRUCTORS =============================================================
  // ============

  // ------------------------------------------------------------------------
  /**
   * Construct <code>SystemConfig</code>.
   * @exception Exception Exception raised in failure situation
   */
  private SystemConfig()
                throws Exception {

    // migrate system configuration in configuration framework (map acl permissions to system config permissions)
    migrateSystemConfiguration();

    // create lists of system principals
    List users = getSystemPrincipalsFromConfig(SYSTEM_USERS_CONFIG_CLASS, IUMPrincipal.IUSER);
    List groups = getSystemPrincipalsFromConfig(SYSTEM_GROUPS_CONFIG_CLASS, IUMPrincipal.IGROUP);
    List roles = getSystemPrincipalsFromConfig(SYSTEM_ROLES_CONFIG_CLASS, IUMPrincipal.IROLE);

    List principals = new ArrayList(users.size() + groups.size() + roles.size());
    principals.addAll(users);
    principals.addAll(groups);
    principals.addAll(roles);

    m_systemUsers = new SystemPrincipalList(users);
    m_systemGroups = new SystemPrincipalList(groups);
    m_systemRoles = new SystemPrincipalList(roles);
    m_systemPrincipals = new SystemPrincipalList(principals);

    // create inverse cache for permissions
    ISystemPrincipalListIterator iter = m_systemPrincipals.iterator();
    while (iter.hasNext()) {
      ISystemPrincipal principal = (ISystemPrincipal)iter.next();
      Set permissions = principal.getMappedPermissions();
      Iterator permiter = permissions.iterator();

      while (permiter.hasNext()) {
        String permission = (String)permiter.next();
        ArrayList pl;

        if (!m_permissions.containsKey(permission)) {
          pl = new ArrayList();
          m_permissions.put(permission, pl);
        }
        else {
          pl = (ArrayList)m_permissions.get(permission);
        }

        pl.add(principal);
      }
    }
  }


  // ==============
  // PUBLIC METHODS ===========================================================
  // ==============

  // ------------------------------------------------------------------------
  /**
   * Get an instance of the <code>SystemConfig</code> (singleton).
   * @return the instance of the <code>SystemConfig</code>
   * @exception Exception Exception raised in failure situation
   */
  public static synchronized SystemConfig getInstance()
                                               throws Exception {
    if (s_systemConfig == null) {
      s_systemConfig = new SystemConfig();
      if (s_log.beDebug()) s_log.debugT("SystemConfig instantiated!");
    }
    return s_systemConfig;
  }


  // ------------------------------------------------------------------------
  /**
   * Get a list of all system users.
   * @return list of all system users
   */
  public ISystemPrincipalList getSystemUsers() {
    return m_systemUsers;
  }


  // ------------------------------------------------------------------------
  /**
   * Get a list of all system groups.
   * @return list of all system groups
   */
  public ISystemPrincipalList getSystemGroups() {
    return m_systemGroups;
  }


  // ------------------------------------------------------------------------
  /**
   * Get a list of all system roles.
   * @return list of all system roles
   */
  public ISystemPrincipalList getSystemRoles() {
    return m_systemRoles;
  }


  // ------------------------------------------------------------------------
  /**
   * Get a list of all system principals (users, groups and roles).
   * @return list of all system principals
   */
  public ISystemPrincipalList getSystemPrincipals() {
    return m_systemPrincipals;
  }


  // ------------------------------------------------------------------------
  /**
   * Check if a given user is a system principal and has the requested
   * permission.
   * @param user the <code>IUser</code> to check
   * @param permission the <code>String</code> with the requested
   *        permission's name
   * @return <code>true</code> if the user is a system principal and
   *        has the specified permission
   */
  public boolean hasPermission(IUser user,
                               String permission) {

    HashSet permissions = (HashSet)user.getTransientAttribute(PERM_CACHE_IN_USER);

    if (permissions == null) {
      // system permissions not set on user: fill permission set as HashSet
      // and store it as transient user property
      permissions = new HashSet();
      user.setTransientAttribute(PERM_CACHE_IN_USER, permissions);
      ISystemPrincipalListIterator iter = m_systemPrincipals.iterator();
      while (iter.hasNext()) {
        ISystemPrincipal systemPrincipal = (ISystemPrincipal)iter.next();
        if (containsPrincipal(systemPrincipal.getPrincipal(), user)) {
          permissions.addAll(systemPrincipal.getMappedPermissions());
        }
      }
      user.setTransientAttribute(PERM_CACHE_IN_USER, permissions);
    }
    
    boolean ret = permissions.contains(permission);
    if (s_log.beDebug()) s_log.debugT("Check whether user '" + user.getId() + "' has permission '" + permission + "' returns " + ret);
    return ret;
    
  }


  // ------------------------------------------------------------------------
  /**
   * Check if a given user is a system principal and has the requested
   * permissions (mass call for {@link #hasPermissions(IUser, String[])}).
   * @param user the <code>IUser</code> to check
   * @param permissions an array of <code>String</code>s with the requested
   *        permissions' name
   * @return <code>true</code> if the user is a system principal and
   *        has all the specified permissions
   */
  public boolean hasPermissions(IUser user,
                                String[] permissions) {

    for (int i = 0, n = permissions.length; i < n; ++i) {
      if (!hasPermission(user, permissions[i])) {
        return false;
      }
    }

    return true;

  }


  // ------------------------------------------------------------------------
  /**
   * Check, if a given principal is a system principal which is authorized
   * to change permissions.
   * @param principal the <code>IUMPrincipal</code> to check
   * @return <code>true</code> if the user is a system principal which is
   *        authorized to change permissions.
   */
  public boolean canChangePermissions(IUMPrincipal principal) {

    IUser user = (IUser)principal;
    Boolean canChangePermmision = (Boolean)user.getTransientAttribute(CACL_CACHE_IN_USER);

    if (canChangePermmision != null) {
      return canChangePermmision.booleanValue();
    }
    
    // change permission not set on principal: fill flag as Boolean
    // and store it as transient user property
    
    try {
      for (int i = 0, n = m_systemPrincipals.principals.length; i < n; ++i) {
        ISystemPrincipal systemPrincipal = m_systemPrincipals.principals[i];
        if (containsPrincipal(systemPrincipal.getPrincipal(), principal)) {
          if (systemPrincipal.canChangePermissions()) {
            user.setTransientAttribute(CACL_CACHE_IN_USER, Boolean.TRUE);
            return true;
          }
        }
      }
    }
    catch (Exception e) {
            //$JL-EXC$
      s_log.warningT("canChangePermissions(233)", "failed to check system principals - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
    }

    user.setTransientAttribute(CACL_CACHE_IN_USER, Boolean.FALSE);
    return false;
    
  }

  /**
   * Check, if a given principal has all service acls.
   
   * @param principal the <code>IUMPrincipal</code> to check
   * @return <code>true</code> if the user is a system principal which is
   *        authorized to all service acls.
   */
  public boolean hasServiceAcls(IUMPrincipal principal) {

    IUser user = (IUser)principal;
    Boolean hasServiceAcls = (Boolean)user.getTransientAttribute(HASSACL_CACHE_IN_USER);

    if (hasServiceAcls != null) {
      return hasServiceAcls.booleanValue();
    }
    
    // change permission not set on principal: fill flag as Boolean
    // and store it as transient user property
    
    try {
      for (int i = 0, n = m_systemPrincipals.principals.length; i < n; ++i) {
        ISystemPrincipal systemPrincipal = m_systemPrincipals.principals[i];
        if (containsPrincipal(systemPrincipal.getPrincipal(), principal)) {
          if (systemPrincipal.hasServiceAcls()) {
            user.setTransientAttribute(HASSACL_CACHE_IN_USER, Boolean.TRUE);
            return true;
          }
        }
      }
    }
    catch (Exception e) {
                  //$JL-EXC$
      s_log.warningT("hasServiceAcls(797)", "failed to check system principals - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
    }

    user.setTransientAttribute(HASSACL_CACHE_IN_USER, Boolean.FALSE);
    return false;
    
  }

  // ------------------------------------------------------------------------
  /**
   * create a new system principal with the given id for the given principal
   * with the permissions specified.
   * @param configID the id for the system principal to use in the config
   * @param principal the <code>IUMPrincipal</code> to create a system principal
   *        config entry for
   * @param permissions an array of permissions that are to be granted to
   *        the system principal
   * @param canChangePermissions the value for the system principal's
   *        <code>change_permissions</code> flag 
   * @param isLockBreaker the value for the system principal's
   *        <code>lock_breaker</code> flag
   * @param hasServiceAcls the value for the system principal's
   *        <code>serviceacls</code> flag
   * @return the added system principal
   * @throws Exception if the configID or principal is empty, or if the
   *         given configID already exists
   */
  public synchronized SystemPrincipal addSystemPrincipal(String configID,
                                                         IUMPrincipal principal,
                                                         String[] permissions,
                                                         boolean canChangePermissions,
																												 boolean hasServiceAcls,
                                                         boolean isLockBreaker)
                                                  throws Exception {

    final long CONFIG_LOCK_TIMEOUT = 10L * 60L * 1000L; // 10 minutes
    final long CONFIG_WAIT_TIMEOUT = 60L * 1000L; // 1 minute

    if( configID == null ) {
      throw new WcmException("invalid parameter - config ID can not be null");
    }

    if( principal == null ) {
      throw new WcmException("invalid parameter - principal can not be null");
    }

    ISystemPrincipalListIterator iterator = m_systemPrincipals.iterator();
    while( iterator.hasNext() ) {
      ISystemPrincipal p = iterator.next();
      if( principal.equals(p.getPrincipal()) ) {
        throw new WcmException("invalid parameter - principal " + principal.getId() + " is alread a system principal");
      }
    }

    String principalType = null;
    switch( principal.getType() ) {
      case IUMPrincipal.IGROUP: principalType = "SystemGroup"; break;
      case IUMPrincipal.IROLE:  principalType = "SystemRole";  break;
      case IUMPrincipal.IUSER:  principalType = "SystemUser";  break;
    }   

    if( principalType == null ) {
      throw new WcmException("invalid parameter - principal type " + principal.getType() + " is not known for " + principal.getId());
    }
        
    // put principal to config 
    IConfigClientContext configContext = IConfigClientContext.createContext(ConfigCrutch.getConfigServiceUser());
    IConfigManager configManager = com.sapportals.config.fwk.Configuration.getInstance().getConfigManager(configContext);
    IConfigPlugin plugin = configManager.getAndLockConfigPlugin(CFG_PLUGIN_CM_UTILITIES_SYSTEM_CONFIG, CONFIG_LOCK_TIMEOUT, CONFIG_WAIT_TIMEOUT);
    IMutableConfigurable configurable = null;
    try {
      configurable = plugin.getConfigurable(configID);
      if( configurable != null ) {
        throw new WcmException("config ID " + configID + " already exists");
      }
      StringBuffer permissionList = new StringBuffer(100);
      if( permissions != null ) {
        for( int i = 0, n = permissions.length; i < n; i++ ) {
          if( i > 0 ) permissionList.append(CFGFWK_ENUM_LIST_SEPARATOR);
          permissionList.append(permissions[i]);
        }
      }
      HashMap properties = new HashMap();
      properties.put(CONFIG_PRINCIPAL_NAME, principal.getId());
      properties.put(CONFIG_LOCK_BREAKER, "" + isLockBreaker);
      properties.put(CONFIG_SERVICEACLS, "" + hasServiceAcls);
      properties.put(CONFIG_CHANGE_PERMISSIONS, "" + canChangePermissions);
      properties.put(CONFIG_PERMISSIONS, permissionList.toString());
      properties.put(CONFIG_UPDATED, Boolean.TRUE.toString());
      configurable = plugin.createConfigurable(configID, properties, principalType);
      if( configurable == null ) {
        throw new WcmException("unable to create configurable " + configID + ", type " + principalType + " for " + principal.getId());
      }
      plugin.addConfigurable(configurable);
    }
    finally {
      plugin.unlock();
    }

    SystemPrincipal sysPrincipal = new SystemPrincipal(
                                                      principal.getId(),
                                                      principal.getType(),
                                                      isLockBreaker,
																											hasServiceAcls,
                                                      getMappedPermissions(configurable),
                                                      canChangePermissions
                                                      ); 

    // put principal to list in memory 
    switch( principal.getType() ) {
      case IUMPrincipal.IGROUP: m_systemGroups = addPrincipalToList(sysPrincipal, m_systemGroups); break;
      case IUMPrincipal.IROLE:  m_systemRoles  = addPrincipalToList(sysPrincipal, m_systemRoles);  break;
      case IUMPrincipal.IUSER:  m_systemUsers  = addPrincipalToList(sysPrincipal, m_systemUsers);  break;
    }   
    addPrincipalToList(sysPrincipal, m_systemPrincipals);
    
    // add permissions to inverse cache for permissions
    Iterator pi = sysPrincipal.getMappedPermissions().iterator();
    while( pi.hasNext() ) {
      String permission = (String)pi.next();
      ArrayList pl;
      if( !m_permissions.containsKey(permission) ) {
        pl = new ArrayList();
        m_permissions.put(permission, pl);
      }
      else {
        pl = (ArrayList)m_permissions.get(permission);
      }
      pl.add(principal);
    }    
    return sysPrincipal;
    
  }

}
