/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.string;
import com.sapportals.wcm.util.uri.URICodec;

import java.util.*;

/**
 * Utility methods for strings
 */
public class StrUtil {

  private static char DELIMITER = '|';// must be a character that will be encoded by URICodec!

  /**
   * @param s TBD: Description of the incoming method parameter
   * @param separator TBD: Description of the incoming method parameter
   * @param trim TBD: Description of the incoming method parameter
   * @return all substrings of the string s which are separated by the specified
   *      separator
   */
  public static Set extractStringComponentsToSet(String s, int separator, boolean trim) {
    List list = extractStringComponents(s, separator, trim);
    HashSet result = new HashSet(list.size());
    Iterator it = list.iterator();
    while (it.hasNext()) {
      result.add(it.next());
    }
    return result;
  }

  /**
   * @param s TBD: Description of the incoming method parameter
   * @param separator TBD: Description of the incoming method parameter
   * @param trim TBD: Description of the incoming method parameter
   * @return all substrings of the string s which are separated by the specified
   *      separator
   */
  public static String[] extractStringComponentsToArray(String s, int separator, boolean trim) {
    List list = extractStringComponents(s, separator, trim);
    String[] result = new String[list.size()];
    Iterator it = list.iterator();
    int i = 0;
    while (it.hasNext()) {
      result[i++] = (String)it.next();
    }
    return result;
  }

  /**
   * @param s TBD: Description of the incoming method parameter
   * @param separator TBD: Description of the incoming method parameter
   * @param trim TBD: Description of the incoming method parameter
   * @return all substrings of the string s which are separated by the specified
   *      separator
   */
  public static LinkedList extractStringComponents(String s, int separator, boolean trim) {
    if (s == null) {
      throw new NullPointerException();
    }

    LinkedList result = new LinkedList();

    if (s.length() > 0 && s.charAt(0) == separator) {
      s = s.substring(1);
    }
    if (s.length() > 0 && s.charAt(s.length() - 1) == separator) {
      s = s.substring(0, s.length() - 1);
    }

    if (s.length() > 0) {
      int separatorPos;
      while ((separatorPos = s.indexOf(separator)) >= 0) {
        result.add(trim ? s.substring(0, separatorPos).trim() : s.substring(0, separatorPos));
        if (separatorPos + 1 < s.length()) {
          s = s.substring(separatorPos + 1);
        }
        else {
          s = "";
        }
      }
      if (s.length() > 0) {
        result.add(trim ? s.trim() : s);
      }
    }

    return result;
  }

  /**
   * @param s TBD: Description of the incoming method parameter
   * @param separators TBD: Description of the incoming method parameter
   * @param trim TBD: Description of the incoming method parameter
   * @return all substrings of the string s which are separated by the specified
   *      separators
   */
  public static Set extractStringComponentsToSet(String s, String separators, boolean trim) {
    List list = extractStringComponents(s, separators, trim);
    HashSet result = new HashSet(list.size());
    Iterator it = list.iterator();
    while (it.hasNext()) {
      result.add(it.next());
    }
    return result;
  }

  /**
   * @param s TBD: Description of the incoming method parameter
   * @param separators TBD: Description of the incoming method parameter
   * @param trim TBD: Description of the incoming method parameter
   * @return all substrings of the string s which are separated by the specified
   *      separators
   */
  public static String[] extractStringComponentsToArray(String s, String separators, boolean trim) {
    List list = extractStringComponents(s, separators, trim);
    String[] result = new String[list.size()];
    Iterator it = list.iterator();
    int i = 0;
    while (it.hasNext()) {
      result[i++] = (String)it.next();
    }
    return result;
  }

  /**
   * @param s TBD: Description of the incoming method parameter
   * @param separators TBD: Description of the incoming method parameter
   * @param trim TBD: Description of the incoming method parameter
   * @return all substrings of the string s which are separated by the specified
   *      separators
   */
  public static LinkedList extractStringComponents(String s, String separators, boolean trim) {
    if (s == null || separators == null) {
      throw new NullPointerException();
    }

    LinkedList result = new LinkedList();

    if (s.length() > 0 && separators.indexOf(s.charAt(0)) >= 0) {
      s = s.substring(1);
    }
    if (s.length() > 0 && separators.indexOf(s.charAt(s.length() - 1)) >= 0) {
      s = s.substring(0, s.length() - 1);
    }

    if (s.length() > 0) {
      do {
        int separatorPos = -1;
        for (int i = 0; i < s.length(); i++) {
          if (separators.indexOf(s.charAt(i)) >= 0) {
            separatorPos = i;
            break;
          }
        }

        if (separatorPos < 0) {
          break;
        }

        result.add(trim ? s.substring(0, separatorPos).trim() : s.substring(0, separatorPos));
        if (separatorPos + 1 < s.length()) {
          s = s.substring(separatorPos + 1);
        }
        else {
          s = "";
        }
      } while (true);

      if (s.length() > 0) {
        result.add(trim ? s.trim() : s);
      }
    }

    return result;
  }

  /**
   * @param s TBD: Description of the incoming method parameter
   * @param separators TBD: Description of the incoming method parameter
   * @param trim TBD: Description of the incoming method parameter
   * @param escapeCharacter TBD: Description of the incoming method parameter
   * @return all substrings of the string s which are separated by the specified
   *      separators and unescape
   */
  public static String[] extractStringComponentsToArray(String s, String separators, boolean trim, int escapeCharacter) {

    // list to collect the values
    LinkedList strings = new LinkedList();

    // string to accumulate the characters of the current value
    String currentValue = "";

    // loop over the characters of the csv
    for (int i = 0; i < s.length(); i++) {
      // get current character
      char c = s.charAt(i);
      // escape next character?
      if (c == escapeCharacter) {
        if (i + 1 < s.length()) {
          i++;
          currentValue += s.charAt(i);
        }
        else {
          // last character is an escape: stop
          break;
        }
      }
      else {
        // separator?
        if (separators.indexOf(c) >= 0) {
          // add the current value to the list of values
          strings.add(currentValue);
          // restart with an empty value
          currentValue = "";
        }
        else {
          // append the current character to the current value
          currentValue += c;
        }
      }
    }

    // if there where characters after the last separator, add this value to the list of values
    if (currentValue.length() > 0) {
      strings.add(currentValue);
    }

    // create the result
    String[] result = new String[strings.size()];

    // add the collected values to the result
    Iterator stringsIterator = strings.iterator();
    int resultIdx = 0;
    while (stringsIterator.hasNext()) {
      result[resultIdx++] = (String)stringsIterator.next();
    }

    return result;
  }

  public static String[] extractStringComponentsToArray(String s, int separator, boolean trim, int escapeCharacter) {

    return extractStringComponentsToArray(s, "" + (char)separator, trim, escapeCharacter);
  }

  /**
   * Replaces a String with String in a String
   *
   * @param text the text in which the replacement will be done
   * @param oldString the old String inside the text
   * @param newString the new String which will replace the oldstring inside the
   *      text
   * @param all <code>true
   *    </true>
   *    if all occurences of the oldString should be replaced
   * @return a String with the replaced text
   */
  public static String replace(String text,
    String oldString,
    String newString,
    boolean all) {

    if ((text == null)
       || (oldString == null)
       || (oldString.length() == 0)
       || (newString == null)) {
      return null;
    }

    StringBuffer result = null;
    int oldpos = 0;
    do {
      int pos = text.indexOf(oldString, oldpos);
      if (pos < 0) {
        break;
      }
      if (result == null) {
        result = new StringBuffer();
      }
      result.append(text.substring(oldpos, pos));
      result.append(newString);
      pos += oldString.length();
      oldpos = pos;
    } while (all);

    if (oldpos == 0) {
      return text;
    }
    else {
      result.append(text.substring(oldpos));
      return new String(result);
    }

  }

  /**
   * Encode the given strings to on string (e.g. for putting multiple strings
   * into one URL parameter) May be called recursively to pack complex
   * structures
   *
   * @param strings TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  public static String encodeStrings(String[] strings) {
    if (strings == null) {
      return null;
    }
    StringBuffer result = new StringBuffer(100);
    for (int i = 0; i < strings.length; i++) {
      result.append(strings[i] == null ? "" : URICodec.EncodePath(strings[i]));
      if (i < strings.length - 1) {
        result.append(DELIMITER);
      }
    }
    return result.toString();
  }

  /**
   * Convert an encoded string array back to the individual components May be
   * called recursively to unpack deep structured subcomponents
   *
   * @param encodedString TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  public static String[] decodeStrings(String encodedString) {
    if (encodedString == null) {
      return null;
    }
    if ("".equals(encodedString)) {
      return new String[1];
    }

    ArrayList items = new ArrayList();

    // start retrieving the individual items. We cannot use the standard string
    // tokenizer since it does not allow empty items ("")
    int pos = 0;

    // start retrieving the individual items. We cannot use the standard string
    // tokenizer since it does not allow empty items ("")
    int nextPos = 0;
    while (nextPos >= 0) {
      nextPos = encodedString.indexOf(DELIMITER, nextPos);
      if (nextPos >= 0) {
        items.add(URICodec.Decode(encodedString.substring(pos, nextPos)));
        nextPos = nextPos + 1;
        pos = nextPos;
      }
      else {
        items.add(URICodec.Decode(encodedString.substring(pos)));
      }
    }

    return (String[])items.toArray(new String[1]);
  }


  /**
   * Matches a string against a pattern which may contain the wildcards * and ?
   *
   * @param pattern the pattern
   * @param string the string
   * @param ignoreCase true iff the case of the pattern and the string should be
   *      ignored
   * @return true iff the string matches the pattern
   */
  public static boolean match(String pattern, String string, boolean ignoreCase) {

    if (pattern == null && string == null) {
      return true;
    }
    if (pattern == null || string == null) {
      return false;
    }
    if (pattern.equals("*")) {
      return true;
    }

    if (string.indexOf('*') >= 0) {
      return true;
    }
    if (string.indexOf('?') >= 0) {
      return true;
    }

    if (ignoreCase) {
      pattern = pattern.toUpperCase();
      string = string.toUpperCase();
    }

    int psave = -1;

    int ssave = -1;
    int patternIdx = 0;
    int stringIdx = 0;
    int patternLength = pattern.length();
    int stringLength = string.length();

    while (true) {
      while (patternIdx < patternLength
         && stringIdx < stringLength
         && pattern.charAt(patternIdx) == string.charAt(stringIdx)) {
        patternIdx++;
        stringIdx++;
      }
      if (patternIdx >= patternLength && stringIdx >= stringLength) {
        return true;
      }

      if (stringIdx < stringLength && patternIdx < patternLength && pattern.charAt(patternIdx) == '?') {
        patternIdx++;
        stringIdx++;
      }
      else {
        if (patternIdx < patternLength && pattern.charAt(patternIdx) == '*') {
          psave = ++patternIdx;
          ssave = stringIdx;
        }
        else {
          if (ssave != -1 && ssave < stringLength) {
            stringIdx = ++ssave;
            patternIdx = psave;
          }
          else {
            return false;
          }
        }
      }
    }
  }
}
