/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.string;
import java.util.Enumeration;

import java.util.LinkedList;
import java.util.NoSuchElementException;

/**
 * A StringTokenizer that uses strings(tags), rather than characters, as tokens.
 *
 * @author adriana.mitu@sapportals.com
 */
public class StrTokenizer implements Enumeration {

  /**
   * Contains a list of the ranges in the string to return as tokens
   */
  private LinkedList ranges = new LinkedList();
  /**
   * The next range to return
   */
  private int index = 0;
  /**
   * Internal copy of the string
   */
  private String original = null;

  /**
   * Return empty values
   */
  private boolean emptyValues = false;

  /**
   * Creates new StrTokenizer
   *
   * @param str What to break up
   * @param delim The items to tokenize on.
   */
  public StrTokenizer(String str, String[] delim) {
    original = str;
    breakEmUp(str, delim, false, true, false);
  }

  /**
   * Creates a new StrTokenizer, optionally return the tags as tokens
   *
   * @param str What to break up.
   * @param delim The tags to tokenize on
   * @param returnTags Return the tage as strings.
   */
  public StrTokenizer(String str, String[] delim, boolean returnTags) {
    original = str;
    breakEmUp(str, delim, returnTags, true, false);
  }

  /**
   * creates a new StrTokenizer, optionally return the tags and letting you
   * specify case sensitivity
   *
   * @param str What to break up
   * @param delim The tags to tokenize on
   * @param returnTags Return the tags as tokens
   * @param caseSensitive Should the tokenization be case sensitive.
   */
  public StrTokenizer(String str, String[] delim, boolean returnTags, boolean caseSensitive) {
    original = str;
    breakEmUp(str, delim, returnTags, caseSensitive, false);
  }

  /**
   * creates a new StrTokenizer, optionally return the tags and letting you
   * specify case sensitivity
   *
   * @param str What to break up
   * @param delim The tags to tokenize on
   * @param returnTags Return the tags as tokens
   * @param caseSensitive if true then the tokenization wiil be case sensitive.
   * @param emptyValues if true then the empty values will be considerer.
   */
  public StrTokenizer(String str, String[] delim, boolean returnTags, boolean caseSensitive, boolean emptyValues) {
    original = str;
    breakEmUp(str, delim, returnTags, caseSensitive, emptyValues);
  }

  /**
   * Break up the string into its parts
   *
   * @param str
   * @param delim
   * @param returnTags
   * @param caseSensitive
   * @param emptyValues
   */
  private void breakEmUp(String str, String[] delim, boolean returnTags, boolean caseSensitive, boolean emptyValues) {
    if (str == null) {
      return;
    }
    int strIndex = 0;

    //account for case sensitivity
    String localStr = caseSensitive ? str : str.toUpperCase();
    String localDelim[] = new String[delim.length];
    for (int i = 0; i < delim.length; i++) {
      localDelim[i] = caseSensitive ? delim[i] : delim[i].toUpperCase();
    }

    //loop while we keep finding delimiters
    while (strIndex < str.length()) {
      int closest = str.length();
      int whichTag = -1;
      //figure out which is the first tag to appear
      for (int i = 0; i < delim.length; i++) {
        int loc = localStr.indexOf(localDelim[i], strIndex);
        if ((loc > -1) && (loc < closest)) {
          closest = loc;
          whichTag = i;
        }
      }
      if (strIndex != closest || (emptyValues && strIndex == closest)) {
        ranges.add(new Range(strIndex, closest));
      }
      strIndex = closest;
      if (whichTag != -1) {
        if (returnTags) {
          ranges.add(new Range(closest, closest + delim[whichTag].length()));
        }
        strIndex += delim[whichTag].length();
      }
    }

  }

  /**
   * How many tokens there are in the string.
   *
   * @return Description of the Return Value
   */
  public int countTokens() {
    return ranges.size();
  }

  /**
   * Whether there are more Elements to return.
   *
   * @return Description of the Return Value
   */
  public boolean hasMoreElements() {
    return hasMoreTokens();
  }

  /**
   * Whether there are more tokens to return.
   *
   * @return Description of the Return Value
   */
  public boolean hasMoreTokens() {
    return (index < ranges.size());
  }

  /**
   * Get the next token.
   *
   * @return Description of the Return Value
   */
  public String nextToken() {
    if (index >= ranges.size()) {
      throw new NoSuchElementException("No more tokens");
    }
    Range r = (Range)ranges.get(index);
    index++;
    return original.substring(r.start, r.end);
  }

  /**
   * Get the next Element
   *
   * @return Description of the Return Value
   */
  public Object nextElement() {
    return nextToken();
  }

  /**
   * TBD: Description of the class.
   */
  private static class Range {
    /**
     * @param start
     * @param end
     */
    public Range(int start, int end) {
      this.start = start;
      this.end = end;
    }

    /**
     */
    public int start;
    /**
     */
    public int end;
  }

  /**
   * Description of the Method
   *
   * @param argv Description of the param
   */
  /*
   * public static void main(String argv[]) {
   * String testStr =
   * "<tag1>a<tag1>this is first text<tag2><tag2>this is<tag1><tag1>second test<tag1>this is third test<tag2>";
   * StrTokenizer tt = new StrTokenizer(testStr, new String[]{"<tag1>", "<tag2>"}, false, true, true);
   * while (tt.hasMoreTokens()) {
   * System.out.println(tt.nextToken());
   * }
   * System.out.println("----------------------------");
   * String testStr1 =
   * "fdhg nvbnn y<tag2>nfgdhn<tag2><tag2>gdfhgf fdg";
   * StrTokenizer tt1 = new StrTokenizer(testStr1, new String[]{"<tag2>"}, false, true, true);
   * while (tt1.hasMoreTokens()) {
   * System.out.println(tt1.nextToken());
   * }
   * System.out.println("--------------------------");
   * StrTokenizer tt2 = new StrTokenizer(testStr1, new String[]{"<tag2>"});
   * while (tt2.hasMoreTokens()) {
   * System.out.println(tt2.nextToken());
   * }
   * System.exit(0);
   * }
   */
}
