/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.resource;
import java.util.HashSet;
import java.util.Iterator;

import java.util.Locale;
import java.util.Properties;

/**
 * Store several description strings in one object. <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author heix
 * @version $Id: //javabas/com.sapportals.wcm/dev/src/java/util/api/com/sapportals/wcm/util/resource/Descriptions.java#5
 *      $
 */
public class Descriptions {

  // ---------
  // Constants ----------------------------------------------------------------
  // ---------

  /**
   * The key for the default description.
   */
  public static String DEFAULT = "en";

  /**
   * The default locale in the system.
   */
  public static Locale DEFAULT_LOCALE = Locale.ENGLISH;

  private static String[] g_RequiredLocaleKeys = new String[]{"en", "de", "fr", "es", "it", "pt"};

  // ------------------
  // Instance Variables -------------------------------------------------------
  // ------------------

  /**
   * the set of properties for this description.
   */
  private Properties m_Descriptions;

  /**
   * the locales for this description.
   */
  private volatile Locale[] m_Locales;


  // ------------
  // Constructors -------------------------------------------------------------
  // ------------

  // ------------------------------------------------------------------------
  /**
   * Create an empty description set.
   */
  public Descriptions() {

    m_Descriptions = new Properties();
    m_Locales = null;

  }


  // ------------------------------------------------------------------------
  /**
   * Create a description set with the locale-id's (as from <code>
   * Locale.toString()</code> ) as keys.
   *
   * @param descriptions a bag of <code>Properties</code> with the locale id's
   *      as keys (e.g. de for german, en_us for us english).
   */
  public Descriptions(Properties descriptions) {

    m_Descriptions = descriptions;
    if (m_Descriptions == null) {
      m_Descriptions = new Properties();
    }
    m_Locales = null;

  }


  // ---------------
  // Private Methods ----------------------------------------------------------
  // ---------------

  // ------------------------------------------------------------------------
  private static String getDescriptionByKey(String key,
    Properties descriptions) {

    if (key == null) {
      return null;
    }

    return (String)descriptions.get(key);
  }


  // ------------------------------------------------------------------------
  private static void setDescriptionByKey(String key,
    String description,
    Properties descriptions) {

    if (key == null) {
      return;
    }

    if (description == null) {
      descriptions.remove(key);
    }
    else {
      descriptions.setProperty(key, description);
    }

  }


  // ------------------------------------------------------------------------
  private static String getDescriptionByLocale(Locale locale,
    Properties descriptions) {

    if (locale == null) {
      locale = DEFAULT_LOCALE;
    }

    String description = null;

    if (locale.getVariant().length() > 0) {
      description = getDescriptionByKey(locale.getLanguage() + "_" + locale.getCountry() + "_" + locale.getVariant(), descriptions);
      if (description != null) {
        return description;
      }
    }

    if (locale.getCountry().length() > 0) {
      description = getDescriptionByKey(locale.getLanguage() + "_" + locale.getCountry(), descriptions);
      if (description != null) {
        return description;
      }
    }

    if (locale.getLanguage().length() > 0) {
      description = getDescriptionByKey(locale.getLanguage(), descriptions);
    }
    else {
      description = getDescriptionByKey(DEFAULT, descriptions);
    }

    return description;
  }


  // ------------------------------------------------------------------------
  private static void setDescriptionByLocale(Locale locale,
    String description,
    Properties descriptions) {

    if (locale == null) {
      locale = DEFAULT_LOCALE;
    }

    if (locale.getVariant().length() > 0) {
      setDescriptionByKey(locale.getLanguage() + "_" + locale.getCountry() + "_" + locale.getVariant(), description, descriptions);
    }
    else if (locale.getCountry().length() > 0) {
      setDescriptionByKey(locale.getLanguage() + "_" + locale.getCountry(), description, descriptions);
    }
    else if (locale.getLanguage().length() > 0) {
      setDescriptionByKey(locale.getLanguage(), description, descriptions);
    }
    else {
      setDescriptionByKey(DEFAULT, description, descriptions);
    }

  }


  // --------------
  // Public Methods -----------------------------------------------------------
  // --------------

  // ------------------------------------------------------------------------
  public static String getDescription(Locale locale,
    Properties descriptions) {

    String description = null;

    if (locale != null) {
      description = getDescriptionByLocale(locale, descriptions);
    }
    if (description != null) {
      return description;
    }

    description = getDescriptionByLocale(Locale.getDefault(), descriptions);
    if (description != null) {
      return description;
    }

    description = getDescriptionByKey(DEFAULT, descriptions);
    if (description != null) {
      return description;
    }

    return getDescriptionByKey("", descriptions);
  }


  // ------------------------------------------------------------------------
  public static Locale getLocale(String localeID) {

    Locale locale = null;

    int first;
    int second;
    try {
      first = localeID.indexOf('_');
      if (first <= 0) {
        locale = new Locale(localeID, "");
      }
      else {
        second = localeID.indexOf('_', first + 1);
        if (second <= 0) {
          locale = new Locale(localeID.substring(0, first - 1), localeID.substring(first + 1));
        }
        else {
          locale = new Locale(localeID.substring(0, first - 1), localeID.substring(first + 1, second - 1), localeID.substring(second + 1));
        }
      }
    }
    catch (Exception e) {
            //$JL-EXC$
      com.sap.tc.logging.Location.getLocation(Descriptions.class.getName()).debugT(e.getMessage());      
    }

    return locale;
  }


  // ------------------------------------------------------------------------
  public static Locale[] getLocales(Properties descriptions) {

    HashSet locales = new HashSet();
    Iterator iterator = descriptions.keySet().iterator();
    while (iterator.hasNext()) {
      Locale locale = getLocale((String)iterator.next());
      if (locale != null) {
        locales.add(locale);
      }
    }

    Locale[] result = new Locale[locales.size()];
    int i = 0;
    iterator = locales.iterator();
    while (iterator.hasNext()) {
      result[i++] = (Locale)iterator.next();
    }

    return result;
  }


  // ------------------------------------------------------------------------
  public static String[] getRequiredLocaleKeys() {

    return g_RequiredLocaleKeys;// Locale.getISOLanguages();

  }


  // ------------------------------------------------------------------------
  public String getDescription(Locale locale) {

    return getDescription(locale, m_Descriptions);
  }


  // ------------------------------------------------------------------------
  public Locale[] getLocales() {

    if (m_Locales == null) {
      synchronized (this) {
        if (m_Locales == null) {
          m_Locales = getLocales(m_Descriptions);
        }
      }
    }

    return m_Locales;
  }


  // ------------------------------------------------------------------------
  public void setDescription(Locale locale,
    String description) {

    setDescriptionByLocale(locale, description, m_Descriptions);

  }


  // ------------------------------------------------------------------------
  public Properties toProperties() {

    return m_Descriptions;
  }

}
