/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.regex.re;
import java.io.IOException;

import java.io.InputStream;

/**
 * Encapsulates InputStream, ...
 */
public final class StreamCharacterIterator implements CharacterIterator {
  /**
   * Underlying is
   */
  private final InputStream is;

  /**
   * Buffer of read chars
   */
  private final StringBuffer buff;

  /**
   * read end?
   */
  private boolean closed;

  /**
   * @param is an InputStream, which is parsed
   */
  public StreamCharacterIterator(InputStream is) {
    this.is = is;
    this.buff = new StringBuffer(512);
    this.closed = false;
  }

  /**
   * @param offset TBD: Description of the incoming method parameter
   * @param length TBD: Description of the incoming method parameter
   * @return a substring
   */
  public String substring(int offset, int length) {
    try {
      ensure(offset + length);
      return buff.toString().substring(offset, length);
    }
    catch (IOException e) {
            //$JL-EXC$
      throw new StringIndexOutOfBoundsException(e.getMessage());
    }
  }

  /**
   * @param offset TBD: Description of the incoming method parameter
   * @return a substring
   */
  public String substring(int offset) {
    try {
      readAll();
      return buff.toString().substring(offset);
    }
    catch (IOException e) {
            //$JL-EXC$
      throw new StringIndexOutOfBoundsException(e.getMessage());
    }
  }


  /**
   * @param pos TBD: Description of the incoming method parameter
   * @return a character at the specified position.
   */
  public char charAt(int pos) {
    try {
      ensure(pos);
      return buff.charAt(pos);
    }
    catch (IOException e) {
            //$JL-EXC$
      throw new StringIndexOutOfBoundsException(e.getMessage());
    }
  }

  /**
   * @param pos TBD: Description of the incoming method parameter
   * @return <tt>true</tt> iff if the specified index is after the end of the
   *      character stream
   */
  public boolean isEnd(int pos) {
    if (buff.length() > pos) {
      return false;
    }
    else {
      try {
        ensure(pos);
        return (buff.length() <= pos);
      }
      catch (IOException e) {
            //$JL-EXC$
        throw new StringIndexOutOfBoundsException(e.getMessage());
      }
    }
  }

  /**
   * Reads n characters from the stream and appends them to the buffer
   *
   * @param n TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception IOException Exception raised in failure situation
   */
  private int read(int n)
    throws IOException {
    if (closed) {
      return 0;
    }

    int c;
    int i = n;
    while (--i >= 0) {
      c = is.read();
      if (c < 0) {// EOF

        closed = true;
        break;
      }
      buff.append((char)c);
    }
    return n - i;
  }

  /**
   * Reads rest of the stream.
   *
   * @exception IOException Exception raised in failure situation
   */
  private void readAll()
    throws IOException {
    while (!closed) {
      read(1000);
    }
  }

  /**
   * Reads chars up to the idx
   *
   * @param idx TBD: Description of the incoming method parameter
   * @exception IOException Exception raised in failure situation
   */
  private void ensure(int idx)
    throws IOException {
    if (closed) {
      return;
    }

    if (idx < buff.length()) {
      return;
    }

    read(idx + 1 - buff.length());
  }
}
