/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.regex.re;

import java.io.*;

/**
 * Data driven (and optionally interactive) testing harness to exercise regular
 * expression compiler and matching engine.
 */
public class RETest {
  // Construct a matcher and a debug compiler
  RE r = new RE();
  REDebugCompiler compiler = new REDebugCompiler();

  // True if we want to see output from success cases
  final static boolean showSuccesses = false;

  /**
   * Main program entrypoint. If an argument is given, it will be compiled and
   * interactive matching will ensue. If no argument is given, the file
   * RETest.txt will be used as automated testing input.
   *
   * @param arg Command line arguments (optional regular expression)
   */
  public static void main(String[] arg) {
    try {
      //new RETest(arg);
      test();
    }
    catch (Exception e) {
            //$JL-EXC$
      e.printStackTrace();
    }
  }

  /**
   * Testing entrypoint.
   *
   * @return TBD: Description of the outgoing return value
   * @exception Exception thrown in case of error
   */
  public static boolean test()
    throws Exception {
    RETest test = new RETest();
    test.runAutomatedTests("docs/RETest.txt");
    return test.failures == 0;
  }

  /**
   * Constructor
   */
  public RETest() { }

  /**
   * Constructor for test
   *
   * @param arg Command line arguments
   */
  public RETest(String[] arg) {
    try {
      // Run interactive tests against a single regexp
      if (arg.length == 2) {
        runInteractiveTests(arg[1]);
      }
      else if (arg.length == 1) {
        // Run automated tests
        runAutomatedTests(arg[0]);
      }
      else {
        System.out.println("Usage: RETest ([-i] [regex]) ([/path/to/testfile.txt])");
      }
    }
    catch (Exception e) {
            //$JL-EXC$
      e.printStackTrace();
    }
  }

  /**
   * Compile and test matching against a single expression
   *
   * @param expr Expression to compile and test
   */
  void runInteractiveTests(String expr) {
    try {
      // Compile expression
      r.setProgram(compiler.compile(expr));

      // Show expression
      say("\n" + expr + "\n");

      // Show program for compiled expression
      compiler.dumpProgram(new PrintWriter(System.out));

      // Test matching against compiled expression
      while (true) {
        // Read from keyboard
        BufferedReader br = new BufferedReader(new InputStreamReader(System.in));
        System.out.print("> ");
        System.out.flush();
        String match = br.readLine();

        // Try a match against the keyboard input
        if (r.match(match)) {
          say("Match successful.");
        }
        else {
          say("Match failed.");
        }

        // Show subparen registers
        showParens(r);
      }
    }
    catch (Exception e) {
            //$JL-EXC$      
      say("Error: " + e.toString());
      e.printStackTrace();
    }
  }

  /**
   * Exit with a fatal error.
   *
   * @param s Last famous words before exiting
   */
  void die(String s) {
    say("FATAL ERROR: " + s);
    System.exit(0);
  }

  /**
   * Fail with an error
   *
   * @param s Failure description
   */
  void fail(String s) {
    failures++;
    say("\n");
    say("*******************************************************");
    say("*********************  FAILURE!  **********************");
    say("*******************************************************");
    say("\n");
    say(s);
    say("");
    compiler.dumpProgram(new PrintWriter(System.out));
    say("\n");
  }

  /**
   * Show a success
   *
   * @param s Success story
   */
  void success(String s) {
    if (showSuccesses) {
      show();
      say("Success: " + s);
    }
  }

  /**
   * Say something to standard out
   *
   * @param s What to say
   */
  void say(String s) {
    System.out.println(s);
  }

  /**
   * Show an expression
   */
  void show() {
    say("\n-----------------------\n");
    say("Expression #" + (n) + " \"" + expr + "\" ");
  }

  /**
   * Dump parenthesized subexpressions found by a regular expression matcher
   * object
   *
   * @param r Matcher object with results to show
   */
  void showParens(RE r) {
    // Loop through each paren
    for (int i = 0; i < r.getParenCount(); i++) {
      // Show paren register
      say("$" + i + " = " + r.getParen(i));
    }
  }

  // Pre-compiled regular expression "a*b"
  char[] re1Instructions =
    {
    0x007c, 0x0000, 0x001a, 0x007c, 0x0000, 0x000d, 0x0041,
    0x0001, 0x0004, 0x0061, 0x007c, 0x0000, 0x0003, 0x0047,
    0x0000, 0xfff6, 0x007c, 0x0000, 0x0003, 0x004e, 0x0000,
    0x0003, 0x0041, 0x0001, 0x0004, 0x0062, 0x0045, 0x0000,
    0x0000,
    };

  REProgram re1 = new REProgram(re1Instructions);

  /*
   * Current expression and number in automated test
   */
  String expr;
  int n = 0;

  /*
   * Count of failures in automated test
   */
  int failures = 0;

  /**
   * Run automated tests in RETest.txt file (from Perl 4.0 test battery)
   *
   * @param testDocument TBD: Description of the incoming method parameter
   * @exception Exception thrown in case of error
   */
  void runAutomatedTests(String testDocument)
    throws Exception {
    long ms = System.currentTimeMillis();

    // Simple test of pre-compiled regular expressions
    RE r = new RE(re1);
    say("a*b");
    say("aaaab = " + r.match("aaab"));
    showParens(r);
    say("b = " + r.match("b"));
    showParens(r);
    say("c = " + r.match("c"));
    showParens(r);
    say("ccccaaaaab = " + r.match("ccccaaaaab"));
    showParens(r);

    r = new RE("a*b");
    String[] s = r.split("xxxxaabxxxxbyyyyaaabzzz");
    r = new RE("x+");
    s = r.grep(s);
    for (int i = 0; i < s.length; i++) {
      System.out.println("s[" + i + "] = " + s[i]);
    }

    r = new RE("a*b");
    String s1 = r.subst("aaaabfooaaabgarplyaaabwackyb", "-");
    System.out.println("s = " + s1);

    // Test from script file
    File testInput = new File(testDocument);
    if (!testInput.exists()) {
      throw new Exception("Could not find: " + testDocument);
    }
    BufferedReader br = new BufferedReader(new FileReader(testInput));
    try {
      // While input is available, parse lines
      while (br.ready()) {
        // Find next re test case
        String number = "";
        String yesno;
        while (br.ready()) {
          number = br.readLine();
          if (number == null) {
            break;
          }
          number = number.trim();
          if (number.startsWith("#")) {
            break;
          }
          if (!number.equals("")) {
            System.out.println("Script error.  Line = " + number);
            System.exit(0);
          }
        }

        // Are we done?
        if (!br.ready()) {
          break;
        }

        // Get expression
        expr = br.readLine();
        n++;
        say("");
        say(n + ". " + expr);
        say("");

        // Compile it
        try {
          r.setProgram(compiler.compile(expr));
        }

        // Some expressions *should* cause exceptions to be thrown
        catch (Exception e) {
            //$JL-EXC$
          // Get expected result
          yesno = br.readLine().trim();

          // If it was supposed to be an error, report success and continue
          if (yesno.equals("ERR")) {
            say("   Match: ERR");
            success("Produces an error (" + e.toString() + "), as expected.");
            continue;
          }

          // Wasn't supposed to be an error
          fail("Produces the unexpected error \"" + e.getMessage() + "\"");
        }
        catch (Error e) {
            //$JL-EXC$
          // Internal error happened
          fail("Compiler threw fatal error \"" + e.getMessage() + "\"");
          e.printStackTrace();
        }

        // Get string to match against
        String matchAgainst = br.readLine().trim();
        say("   Match against: '" + matchAgainst + "'");

        // Expression didn't cause an expected error
        if (matchAgainst.equals("ERR")) {
          fail("Was expected to be an error, but wasn't.");
          continue;
        }

        // Try matching
        try {
          // Match against the string
          boolean b = r.match(matchAgainst);

          // Get expected result
          yesno = br.readLine().trim();

          // If match succeeded
          if (b) {
            // Status
            say("   Match: YES");

            // Match wasn't supposed to succeed
            if (yesno.equals("NO")) {
              fail("Matched \"" + matchAgainst + "\", when not expected to.");
            }
            else
              if (yesno.equals("YES")) {
              // Match succeeded as expected
              success("Matched \"" + matchAgainst + "\", as expected:");

              // Show subexpression registers
              if (showSuccesses) {
                showParens(r);
              }

              say("   Paren count: " + r.getParenCount());

              // Check registers against expected contents
              for (int p = 0; p < r.getParenCount(); p++) {
                // Get next register
                String register = br.readLine().trim();
                say("   Paren " + p + " : " + r.getParen(p));

                // Compare expected result with actual
                if (!register.equals(r.getParen(p))) {
                  // Register isn't what it was supposed to be
                  fail("Register " + p + " should be = \"" + register + "\", but is \"" + r.getParen(p) + "\" instead.");
                }
              }
            }
            else {
              // Bad test script
              die("Test script error!");
            }
          }
          else {
            // Status
            say("   Match: NO");

            // Match failed
            if (yesno.equals("YES")) {
              // Should have failed
              fail("Did not match \"" + matchAgainst + "\", when expected to.");
            }
            else
              if (yesno.equals("NO")) {
              // Should have failed
              success("Did not match \"" + matchAgainst + "\", as expected.");
            }
            else {
              // Bad test script
              die("Test script error!");
            }
          }
        }

        // Matcher blew it
        catch (Exception e) {
            //$JL-EXC$
          fail("Matcher threw exception: " + e.toString());
          e.printStackTrace();
        }

        // Internal error
        catch (Error e) {
            //$JL-EXC$
          fail("Matcher threw fatal error \"" + e.getMessage() + "\"");
          e.printStackTrace();
        }
      }
    }
    finally {
      br.close();
    }

    // Show match time
    System.out.println("\n\nMatch time = " + (System.currentTimeMillis() - ms) + " ms.");

    // Print final results
    System.out.println("\nTests complete.  " + n + " tests, " + failures + " failure(s).");
  }
}
