/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.regex.re;
import java.io.*;

import java.util.*;

/**
 * A subclass of RECompiler which can dump a regular expression program for
 * debugging purposes.
 */
public class REDebugCompiler extends RECompiler {
  /**
   * Mapping from opcodes to descriptive strings
   */
  static Hashtable hashOpcode = new Hashtable();
  static {
    hashOpcode.put(new Integer(RE.OP_RELUCTANTSTAR), "OP_RELUCTANTSTAR");
    hashOpcode.put(new Integer(RE.OP_RELUCTANTPLUS), "OP_RELUCTANTPLUS");
    hashOpcode.put(new Integer(RE.OP_RELUCTANTMAYBE), "OP_RELUCTANTMAYBE");
    hashOpcode.put(new Integer(RE.OP_END), "OP_END");
    hashOpcode.put(new Integer(RE.OP_BOL), "OP_BOL");
    hashOpcode.put(new Integer(RE.OP_EOL), "OP_EOL");
    hashOpcode.put(new Integer(RE.OP_ANY), "OP_ANY");
    hashOpcode.put(new Integer(RE.OP_ANYOF), "OP_ANYOF");
    hashOpcode.put(new Integer(RE.OP_BRANCH), "OP_BRANCH");
    hashOpcode.put(new Integer(RE.OP_ATOM), "OP_ATOM");
    hashOpcode.put(new Integer(RE.OP_STAR), "OP_STAR");
    hashOpcode.put(new Integer(RE.OP_PLUS), "OP_PLUS");
    hashOpcode.put(new Integer(RE.OP_MAYBE), "OP_MAYBE");
    hashOpcode.put(new Integer(RE.OP_NOTHING), "OP_NOTHING");
    hashOpcode.put(new Integer(RE.OP_GOTO), "OP_GOTO");
    hashOpcode.put(new Integer(RE.OP_ESCAPE), "OP_ESCAPE");
    hashOpcode.put(new Integer(RE.OP_OPEN), "OP_OPEN");
    hashOpcode.put(new Integer(RE.OP_CLOSE), "OP_CLOSE");
    hashOpcode.put(new Integer(RE.OP_BACKREF), "OP_BACKREF");
    hashOpcode.put(new Integer(RE.OP_POSIXCLASS), "OP_POSIXCLASS");
  }

  /**
   * Returns a descriptive string for an opcode.
   *
   * @param opcode Opcode to convert to a string
   * @return Description of opcode
   */
  String opcodeToString(char opcode) {
    // Get string for opcode
    String ret = (String)hashOpcode.get(new Integer(opcode));

    // Just in case we have a corrupt program
    if (ret == null) {
      ret = "OP_????";
    }
    return ret;
  }

  /**
   * Return a string describing a (possibly unprintable) character.
   *
   * @param c Character to convert to a printable representation
   * @return String representation of character
   */
  String charToString(char c) {
    // If it's unprintable, convert to '\###'
    if (c < ' ' || c > 127) {
      return "\\" + (int)c;
    }

    // Return the character as a string
    return String.valueOf(c);
  }

  /**
   * Returns a descriptive string for a node in a regular expression program.
   *
   * @param node Node to describe
   * @return Description of node
   */
  String nodeToString(int node) {
    // Get opcode and opdata for node
    char opcode = instruction[node + RE.offsetOpcode];
    int opdata = (int)instruction[node + RE.offsetOpdata];

    // Return opcode as a string and opdata value
    return opcodeToString(opcode) + ", opdata = " + opdata;
  }

  /**
   * Dumps the current program to a PrintWriter
   *
   * @param p PrintWriter for program dump output
   */
  public void dumpProgram(PrintWriter p) {
    // Loop through the whole program
    for (int i = 0; i < lenInstruction; ) {
      // Get opcode, opdata and next fields of current program node
      char opcode = instruction[i + RE.offsetOpcode];
      char opdata = instruction[i + RE.offsetOpdata];
      short next = (short)instruction[i + RE.offsetNext];

      // Display the current program node
      p.print(i + ". " + nodeToString(i) + ", next = ");

      // If there's no next, say 'none', otherwise give absolute index of next node
      if (next == 0) {
        p.print("none");
      }
      else {
        p.print(i + next);
      }

      // Move past node
      i += RE.nodeSize;

      // If character class
      if (opcode == RE.OP_ANYOF) {
        // Opening bracket for start of char class
        p.print(", [");

        // Show each range in the char class
        int rangeCount = opdata;
        for (int r = 0; r < rangeCount; r++) {
          // Get first and last chars in range
          char charFirst = instruction[i++];
          char charLast = instruction[i++];

          // Print range as X-Y, unless range encompasses only one char
          if (charFirst == charLast) {
            p.print(charToString(charFirst));
          }
          else {
            p.print(charToString(charFirst) + "-" + charToString(charLast));
          }
        }

        // Annotate the end of the char class
        p.print("]");
      }

      // If atom
      if (opcode == RE.OP_ATOM) {
        // Open quote
        p.print(", \"");

        // Print each character in the atom
        for (int len = opdata; len-- != 0; ) {
          p.print(charToString(instruction[i++]));
        }

        // Close quote
        p.print("\"");
      }

      // Print a newline
      p.println("");
    }
  }
}
