/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.regex;

import com.sapportals.wcm.util.regex.re.RE;
import com.sapportals.wcm.util.regex.re.RESyntaxException;

/**
 * Provides a pattern object for matching strings.
 */
public class Pattern {
  /**
   * Enables canonical equivalence.
   */
  public static int CANON_EQ = 1;

  /**
   * Enables case-insensitive matching.
   */
  public static int CASE_INSENSITIVE = 2;

  /**
   * Permits whitespace and comments in pattern.
   */
  public static int COMMENTS = 4;

  /**
   * Enables dotall mode.
   */
  public static int DOTALL = 8;

  /**
   * Enables multiline mode.
   */
  public static int MULTILINE = 16;

  /**
   * Enables Unicode-aware case folding.
   */
  public static int UNICODE_CASE = 32;

  /**
   * Enables Unix lines mode.
   */
  public static int UNIX_LINES = 64;

  private RE m_expression = null;
  private String m_pattern = null;
  private int m_flags = 0;


  public Pattern(String pattern)
    throws PatternSyntaxException {
    this(pattern, 0);
  }

  public Pattern(String pattern, int flags)
    throws PatternSyntaxException {
    m_pattern = pattern;
    m_flags = flags;

    int reFlags = 0;
    if ((flags & CASE_INSENSITIVE) != 0) {
      reFlags |= RE.MATCH_CASEINDEPENDENT;
    }
    else {
      reFlags |= RE.MATCH_NORMAL;
    }
    if ((flags & MULTILINE) != 0) {
      reFlags |= RE.MATCH_MULTILINE;
    }
    else {
      reFlags |= RE.MATCH_SINGLELINE;
    }

    try {
      m_expression = new RE(pattern, reFlags);
    }
    catch (RESyntaxException e) {
            //$JL-EXC$
      throw new PatternSyntaxException(e.toString());
    }
  }

  /**
   * Compiles the given regular expression into a pattern.
   *
   * @param regex TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception PatternSyntaxException Exception raised in failure situation
   */
  public static Pattern compile(String regex)
    throws PatternSyntaxException {
    return new Pattern(regex);
  }

  /**
   * Compiles the given regular expression into a pattern with the given flags.
   *
   * @param regex TBD: Description of the incoming method parameter
   * @param flags TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception PatternSyntaxException Exception raised in failure situation
   */
  public static Pattern compile(String regex, int flags)
    throws PatternSyntaxException {
    return new Pattern(regex, flags);
  }

  /**
   * Returns this pattern's match flags.
   *
   * @return TBD: Description of the outgoing return value
   */
  public int flags() {
    return m_flags;
  }

  /**
   * Creates a matcher that will match the given input against this pattern.
   *
   * @param input TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  public Matcher matcher(CharSequence input) {
    return new Matcher(this, input.toString());
  }

  /**
   * Creates a matcher that will match the given input against this pattern.
   *
   * @param input TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  public Matcher matcher(String input) {
    return new Matcher(this, input);
  }

  /**
   * Compiles the given regular expression and attempts to match the given input
   * against it.
   *
   * @param regex TBD: Description of the incoming method parameter
   * @param input TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception PatternSyntaxException Exception raised in failure situation
   */
  public static boolean matches(String regex, CharSequence input)
    throws PatternSyntaxException {
    return new Matcher(new Pattern(regex), input.toString()).matches();
  }

  /**
   * Returns the regular expression from which this pattern was compiled.
   *
   * @return TBD: Description of the outgoing return value
   */
  public String pattern() {
    return m_pattern;
  }

  /**
   * Splits the given input sequence around matches of this pattern.
   *
   * @param input TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  public String[] split(CharSequence input) {
    return m_expression.split(input.toString());
  }

  /**
   * Splits the given input sequence around matches of this pattern.
   *
   * @param input TBD: Description of the incoming method parameter
   * @param limit TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  public String[] split(CharSequence input, int limit) {
    throw new UnsupportedOperationException();
  }

  protected RE getExpression() {
    return m_expression;
  }
}
