package com.sapportals.wcm.util.name;

import org.w3c.dom.Node;


/**
 * Generic class for the representation of namespace-qualified names.
 * <p>
 * Copyright (c) SAP AG 2001-2004
 * @author julian.reschke@greenbytes.de
 * @version $Id: Name.java,v 1.2 2004/10/29 14:22:00 sei Exp $
 */

public class Name implements IName, java.io.Serializable {

  private final String namespace;
  private final String name;
  private String string = null;
  private final int hash;

  /**
   * @param namespace namespaceUri. Note that both "" (the empty string) and
   * <code>null</code> are internally treated identically (as a non-namespace-qualified
   * name).
   * @param name <a href="http://www.w3.org/TR/1999/REC-xml-names-19990114/#NT-NCName">NCName</a>
   */
  public Name(String namespace, String name) {
    // expand null into empty string for consistency
    this.namespace = (namespace == null) ? "" : namespace;
    this.name = name;
    this.hash = this.name.hashCode() ^ this.namespace.hashCode();
  }

  /**
   * @param prefixedName namespace & name in {namespace}name format
   */
  public Name(String prefixedName) throws Exception {

    if (prefixedName.charAt(0) != '{') {
      throw new Exception("illegal format: " + prefixedName);
    }

    int rightBracket = prefixedName.lastIndexOf('}');
    if (rightBracket < 0) {
      throw new Exception("illegal format: " + prefixedName);
    }

    this.namespace = prefixedName.substring(1, rightBracket);
    this.name = prefixedName.substring(rightBracket + 1);
    this.string = prefixedName;
    this.hash = name.hashCode() ^ namespace.hashCode();
  }


  /**
   * @return the namespace URI (or the empty string when no namespace was given)
   */
  public final String getNamespace() {
    return this.namespace;
  }

  /**
   * @return the namespace URI (or <code>null</code> when no namespace was given)
   */
  public final String getNamespaceOrNull() {
    return namespace.length() == 0 ? null : this.namespace;
  }

  /**
   * return get the local name
   */

  public final String getName () {
    return this.name;
  }


  public final boolean equals(Object other) {

    if (other == null) {
      return false;
    }
    else {
      if (!(other instanceof Name)) {
        return false;
      }
      else {
        Name otherName = (Name)other;
        return this.hash == otherName.hash && this.name.equals(otherName.name) && this.namespace.equals(otherName.namespace);
      }
    }
  }
  
  /**
   * Compare namespace name and local name with namespace and name of a DOM
   * node.
   * 
   * @param node
   *          DOM node to compare with
   * @return whether both match
   */
  public boolean equalsNode(Node node) {
    if (node == null) {
      return false;
    }
    else {
      String ns2 = node.getNamespaceURI();
      if (ns2 == null) {
        ns2 = "";
      }
      return this.namespace.equals(ns2) && this.name.equals(node.getLocalName());
    }
  }
  
  public final int hashCode() {
    return this.hash;
  }


  public final String toString() {
    if (this.string == null) {
      StringBuffer b = new StringBuffer(this.namespace.length() + this.name.length() + 2);
      b.append("{");
      b.append(this.namespace);
      b.append("}");
      b.append(this.name);
      this.string = b.toString();
    }
    return this.string;
  }
}
