/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.mmparser;
import java.io.FilterInputStream;

import java.io.IOException;

/**
 * A <code>PartInputStream</code> filters a <code>MMInputStream</code> ,
 * providing access to a single mime part contained in it. <p>
 *
 * Note the <code>readLine</code> method of <code>MMInputStream</code> adds the
 * \r\n also to the end of the last line. We have to cut those chars. We must
 * always maintain at least 2 characters in the buffer to allow us to trim when
 * necessary.
 *
 * @author SAP AG
 * @version $Id: //javabas/com.sapportals.wcm/dev/src/java/util/api/com/sapportals/wcm/util/mmparser/PartInputStream.java#5
 *      $ Copyright (c) SAP AG 2001-2002
 */
public class PartInputStream extends FilterInputStream {

  /**
   * boundary which "ends" the stream
   */
  private String m_boundary;

  /**
   * our buffer
   */
  private byte[] m_buf = new byte[64 * 1024];// 64k

  /**
   * number of bytes we've read into the buffer
   */
  private int m_count;

  /**
   * current position in the buffer
   */
  private int m_pos;

  /**
   * flag that indicates if we have encountered the boundary
   */
  private boolean m_eof;

  MMInputStream m_in = null;

  /**
   * Creates a <code>PartInputStream</code> which stops at the specified
   * boundary from a <code>ServletInputStream<code>.
   *
   * @param in a servlet input stream.
   * @param boundary the MIME boundary to stop at.
   * @exception IOException Exception raised in failure situation
   */
  PartInputStream(MMInputStream in,
    String boundary)
    throws IOException {
    super(in);
    m_in = in;
    m_boundary = boundary;
  }

  /**
   * Fill up our buffer from the underlying input stream, and check for the
   * boundary that signifies end-of-file. Users of this method must ensure that
   * they leave exactly 2 characters in the buffer before calling this method
   * (except the first time), so that we may only use these characters if a
   * boundary is not found in the first line read.
   *
   * @exception IOException if an I/O error occurs.
   */
  private void fill()
    throws IOException {
    if (m_eof) {
      return;
    }

    // as long as we are not just starting up
    if (m_count > 0) {
      // if the caller left the requisite amount spare in the buffer
      if (m_count - m_pos == 2) {
        // copy it back to the start of the buffer
        System.arraycopy(m_buf, m_pos, m_buf, 0, m_count - m_pos);
        m_count -= m_pos;
        m_pos = 0;
      }
      else {
        // should never happen, but just in case
        throw new IllegalStateException("fill() detected illegal buffer state");
      }
    }

    // try and fill the entire buffer, starting at count, line by line
    // but never read so close to the end that we might split a boundary
    int read = 0;
    int maxRead = m_buf.length - m_boundary.length();
    while (m_count < maxRead) {
      // read a line
      read = m_in.readLine(m_buf, m_count, m_buf.length - m_count);
      // check for eof and boundary
      if (read == -1) {
        throw new IOException("unexpected end of part");
      }
      else {
        if (read >= m_boundary.length()) {
          m_eof = true;
          for (int i = 0; i < m_boundary.length(); i++) {
            if (m_boundary.charAt(i) != m_buf[m_count + i]) {
              // Not the boundary!
              m_eof = false;
              break;
            }
          }
          if (m_eof) {
            break;
          }
        }
      }
      // success
      m_count += read;
    }
  }

  /**
   * See the general contract of the <code>read</code> method of <code>
   * InputStream</code> . <p>
   *
   * Returns <code>-1</code> (end of file) when the MIME boundary of this part
   * is encountered.
   *
   * @return the next byte of data, or <code>-1</code> if the end of the stream
   *      is reached.
   * @exception IOException if an I/O error occurs.
   */
  public int read()
    throws IOException {
    if (m_count - m_pos <= 2) {
      fill();
      if (m_count - m_pos <= 2) {
        return -1;
      }
    }
    return m_buf[m_pos++] & 0xff;
  }

  /**
   * See the general contract of the <code>read</code> method of <code>
   * InputStream</code> . <p>
   *
   * Returns <code>-1</code> (end of file) when the MIME boundary of this part
   * is encountered.
   *
   * @param b the buffer into which the data is read.
   * @return the total number of bytes read into the buffer, or <code>-1</code>
   *      if there is no more data because the end of the stream has been
   *      reached.
   * @exception IOException if an I/O error occurs.
   */
  public int read(byte b[])
    throws IOException {
    return read(b, 0, b.length);
  }

  /**
   * See the general contract of the <code>read</code> method of <code>
   * InputStream</code> . <p>
   *
   * Returns <code>-1</code> (end of file) when the MIME boundary of this part
   * is encountered.
   *
   * @param b the buffer into which the data is read.
   * @param off the start offset of the data.
   * @param len the maximum number of bytes read.
   * @return the total number of bytes read into the buffer, or <code>-1</code>
   *      if there is no more data because the end of the stream has been
   *      reached.
   * @exception IOException if an I/O error occurs.
   */
  public int read(byte b[], int off, int len)
    throws IOException {
    int total = 0;
    if (len == 0) {
      return 0;
    }

    int avail = m_count - m_pos - 2;
    if (avail <= 0) {
      fill();
      avail = m_count - m_pos - 2;
      if (avail <= 0) {
        return -1;
      }
    }
    int copy = Math.min(len, avail);
    System.arraycopy(m_buf, m_pos, b, off, copy);
    m_pos += copy;
    total += copy;

    while (total < len) {
      fill();
      avail = m_count - m_pos - 2;
      if (avail <= 0) {
        return total;
      }
      copy = Math.min(len - total, avail);
      System.arraycopy(m_buf, m_pos, b, off + total, copy);
      m_pos += copy;
      total += copy;
    }
    return total;
  }

  /**
   * Returns the number of bytes that can be read from this input stream without
   * blocking. This is a standard <code>InputStream</code> idiom to deal with
   * buffering gracefully, and is not same as the length of the part arriving in
   * this stream.
   *
   * @return the number of bytes that can be read from the input stream without
   *      blocking.
   * @exception IOException if an I/O error occurs.
   */
  public int available()
    throws IOException {
    int avail = (m_count - m_pos - 2) + m_in.available();
    // Never return a negative value
    return (avail < 0 ? 0 : avail);
  }

  /**
   * Closes this input stream and releases any system resources associated with
   * the stream. <p>
   *
   * This method will read any unread data in the MIME part so that the next
   * part starts an an expected place in the parent <code>InputStream</code> .
   * Note that if the client code forgets to call this method on error, <code>
   * MultipartParser</code> will call it automatically if you call <code>
   * readNextPart()</code> .
   *
   * @exception IOException if an I/O error occurs.
   */
  public void close()
    throws IOException {
    if (!m_eof) {
      while (read(m_buf, 0, m_buf.length) != -1) {
        ;
      }
    }
  }
}

