/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.mmparser;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;

/**
 * A <code>ParamPart</code> is an upload part which represents a normal input
 * (i.e. non <code>TYPE="file"</code> ) form parameter.
 *
 * @author SAP AG
 * @version $Id: //javabas/com.sapportals.wcm/dev/src/java/util/api/com/sapportals/wcm/util/mmparser/ParamPart.java#5
 *      $ Copyright (c) SAP AG 2001-2002
 */
public class ParamPart extends AbstractPart {

  /**
   * contents of the parameter
   */
  private byte[] value;

  /**
   * Constructs a parameter part; this is called by the parser.
   *
   * @param name the name of the parameter.
   * @param in the servlet input stream to read the parameter value from.
   * @param boundary the MIME boundary that delimits the end of parameter value.
   * @exception IOException <Please write something about why this exception
   *      would be thrown>
   *
   *
   */
  ParamPart(String name, MMInputStream in, String boundary)
    throws IOException {
    super(name);

    // Copy the part's contents into a byte array
    PartInputStream pis = new PartInputStream(in, boundary);
    ByteArrayOutputStream baos = new ByteArrayOutputStream(512);
    byte[] buf = new byte[128];
    int read;
    while ((read = pis.read(buf)) != -1) {
      baos.write(buf, 0, read);
    }
    pis.close();
    baos.close();

    // save it for later
    value = baos.toByteArray();
  }

  /**
   * Returns the value of the parameter as an array of bytes or a zero length
   * array if the user entered no value for this parameter.
   *
   * @return value of parameter as a ISO-8559-1 string.
   */
  public byte[] getValue() {
    return value;
  }

  /**
   * Returns the value of the parameter in the default ISO-8859-1 encoding or
   * empty string if the user entered no value for this parameter.
   *
   * @return value of parameter as a ISO-8559-1 encoded string.
   * @exception UnsupportedEncodingException <Please write something about why
   *      this exception would be thrown>
   *
   *
   */
  public String getStringValue()
    throws UnsupportedEncodingException {
    return getStringValue("ISO-8859-1");
  }

  /**
   * Returns the value of the parameter in the supplied encoding or empty string
   * if the user entered no value for this parameter.
   *
   * @param encoding <Please write something about this parameter>
   *
   *
   * @return value of parameter as a string.
   * @exception UnsupportedEncodingException <Please write something about why
   *      this exception would be thrown>
   *
   *
   */
  public String getStringValue(String encoding)
    throws UnsupportedEncodingException {
    return new String(value, encoding);
  }

  /**
   * Returns <code>true</code> to indicate this part is a file.
   *
   * @return true.
   */
  public boolean isFile() {
    return false;
  }
}

