/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.mmparser;

import java.io.*;

import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;

/**
 * This class implements a readLine Method for a byte stream.
 *
 * @author SAP AG
 * @version $Id: //javabas/com.sapportals.wcm/dev/src/java/util/api/com/sapportals/wcm/util/mmparser/MMInputStream.java#5
 *      $ Copyright (c) SAP AG 2001-2002
 */
public class MMInputStream extends FilterInputStream {

  /**
   * input stream we are filtering
   */
  private InputStream m_in;

  /**
   * our buffer
   */
  private byte[] m_buf = new byte[64 * 1024];// 64k

  /**
   * number of bytes we've read into the buffer
   */
  private int m_count;

  /**
   * current position in the buffer
   */
  private int m_pos;

  /**
   * Creates a <code>BufferedServletInputStream</code> that wraps the provided
   * <code>ServletInputStream</code> .
   *
   * @param in a servlet input stream.
   */
  public MMInputStream(InputStream in) {
    super(in);
    m_in = in;
  }

  /**
   * Fill up our buffer from the underlying input stream. Users of this method
   * must ensure that they use all characters in the buffer before calling this
   * method.
   *
   * @exception IOException if an I/O error occurs.
   */
  private void fill()
    throws IOException {
    int i = m_in.read(m_buf, 0, m_buf.length);
    if (i > 0) {
      m_pos = 0;
      m_count = i;
    }
  }

  /**
   * Implement buffering on top of the <code>readLine</code> method of the
   * wrapped <code>ServletInputStream</code> .
   *
   * @param b an array of bytes into which data is read.
   * @param off an integer specifying the character at which this method begins
   *      reading.
   * @param len an integer specifying the maximum number of bytes to read.
   * @return an integer specifying the actual number of bytes read, or -1 if the
   *      end of the stream is reached.
   * @exception IOException if an I/O error occurs.
   */
  public int readLine(byte b[], int off, int len)
    throws IOException {
    int total = 0;
    if (len == 0) {
      return 0;
    }

    int avail = m_count - m_pos;
    if (avail <= 0) {
      fill();
      avail = m_count - m_pos;
      if (avail <= 0) {
        return -1;
      }
    }
    int copy = Math.min(len, avail);
    int eol = findeol(m_buf, m_pos, copy);
    if (eol != -1) {
      copy = eol;
    }
    System.arraycopy(m_buf, m_pos, b, off, copy);
    m_pos += copy;
    total += copy;

    while (total < len && eol == -1) {
      fill();
      avail = m_count - m_pos;
      if (avail <= 0) {
        return total;
      }
      copy = Math.min(len - total, avail);
      eol = findeol(m_buf, m_pos, copy);
      if (eol != -1) {
        copy = eol;
      }
      System.arraycopy(m_buf, m_pos, b, off + total, copy);
      m_pos += copy;
      total += copy;
    }
    return total;
  }

  /**
   * Attempt to find the '\n' end of line marker as defined in the comment of
   * the <code>readLine</code> method of <code>ServletInputStream</code> .
   *
   * @param b byte array to search.
   * @param pos position in byte array to search from.
   * @param len maximum number of bytes to search.
   * @return the number of bytes including the \n, or -1 if none found.
   */
  private static int findeol(byte b[], int pos, int len) {
    int end = pos + len;
    int i = pos;
    while (i < end) {
      if (b[i++] == '\n') {
        return i - pos;
      }
    }
    return -1;
  }

  /**
   * Implement buffering on top of the <code>read</code> method of the wrapped
   * <code>ServletInputStream</code> .
   *
   * @return the next byte of data, or <code>-1</code> if the end of the stream
   *      is reached.
   * @exception IOException if an I/O error occurs.
   */
  public int read()
    throws IOException {
    if (m_count <= m_pos) {
      fill();
      if (m_count <= m_pos) {
        return -1;
      }
    }
    return m_buf[m_pos++] & 0xff;
  }

  /**
   * Implement buffering on top of the <code>read</code> method of the wrapped
   * <code>ServletInputStream</code> .
   *
   * @param b the buffer into which the data is read.
   * @param off the start offset of the data.
   * @param len the maximum number of bytes read.
   * @return the total number of bytes read into the buffer, or <code>-1</code>
   *      if there is no more data because the end of the stream has been
   *      reached.
   * @exception IOException if an I/O error occurs.
   */
  public int read(byte b[], int off, int len)
    throws IOException {
    int total = 0;
    while (total < len) {
      int avail = m_count - m_pos;
      if (avail <= 0) {
        fill();
        avail = m_count - m_pos;
        if (avail <= 0) {
          if (total > 0) {
            return total;
          }
          else {
            return -1;
          }
        }
      }
      int copy = Math.min(len - total, avail);
      System.arraycopy(m_buf, m_pos, b, off + total, copy);
      m_pos += copy;
      total += copy;
    }
    return total;
  }
}

