/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.log;

import java.io.*;

/**
 * Channel for writing log output to a file
 */
public class FileChannel extends AbstractChannel implements IChannel {
  private String m_fileName = null;
  private FileOutputStream m_output = null;
  private long m_bytesWritten = 0;

  /**
   * Construct
   *
   * @param fileName TBD: Description of the incoming method parameter
   * @param maxSize TBD: Description of the incoming method parameter
   * @param maxBacklog TBD: Description of the incoming method parameter
   * @param showStackTraces TBD: Description of the incoming method parameter
   * @param showThreads TBD: Description of the incoming method parameter
   * @exception Exception Exception raised in failure situation
   */
  public FileChannel(String fileName, Long maxSize, Integer maxBacklog, Boolean showStackTraces, Boolean showThreads)
    throws Exception {

    super(maxSize, maxBacklog, showStackTraces, showThreads);
    setFileName(fileName);
  }

  /**
   * Set the name of the file the output is written to
   *
   * @param fileName fileName to be set
   * @exception FileNotFoundException Exception raised in failure situation
   */
  public void setFileName(String fileName)
    throws FileNotFoundException {

    if (fileName == null) {
      System.out.println("com.sapportals.wcm.util.log.FileChannel: no filename is specified!");
    }

    m_fileName = fileName;
    File file = new File(fileName);
    if (file.exists()) {
      m_bytesWritten = file.length();
    }
    else {
      m_bytesWritten = 0;
    }
    m_fileName = fileName;
    m_output = new FileOutputStream(fileName, true);
  }

  /**
   * Close the output stream
   */
  public void finalize() {
    try {
      if (m_output != null) {
        m_output.close();
      }
    }
    catch (Exception e) {
            //$JL-EXC$
      com.sap.tc.logging.Location.getLocation(this.getClass()).debugT(e.getMessage());
    }
  }

  /**
   * Log
   *
   * @param method TBD: Description of the incoming method parameter
   * @param level TBD: Description of the incoming method parameter
   * @param message TBD: Description of the incoming method parameter
   * @param throwable TBD: Description of the incoming method parameter
   */
  public synchronized void log(String method, int level, Object message, Throwable throwable) {

    if (m_output == null) {
      return;
    }

    try {
      String logText = getLogText(method, level, message, throwable);
      byte[] bytes = logText.getBytes();
      m_output.write(bytes);
      m_output.flush();

      m_bytesWritten += bytes.length;
      if (m_maxSize > 0 && m_bytesWritten >= m_maxSize) {
        backlog();
      }
    }
    catch (Exception e) {
            //$JL-EXC$      
      e.printStackTrace();
    }
  }

  /**
   * Rename the current log file; remove the oldest log file if the maximum
   * number of stored log files is reached
   *
   * @exception Exception Exception raised in failure situation
   */
  private void backlog()
    throws Exception {

    final int retries = 400;// 4 seconds

    m_output.close();
    m_output = null;// prevent further logging in case an exception is thrown

    if (m_maxBacklog > 0) {
      // clear oldest file
      File oldest = new File(m_fileName + m_maxBacklog);
      if (oldest.exists()) {
        oldest.delete();
      }

      // rename backlog files
      for (int i = m_maxBacklog - 1; i > 0; i--) {
        File logged = new File(m_fileName + i);
        if (logged.exists()) {
          File nextOlder = new File(m_fileName + (i + 1));
          logged.renameTo(nextOlder);
        }
      }

      // rename
      int i = 0;
      while (true) {
        File current = new File(m_fileName);
        File first = new File(m_fileName + 1);

        if (current.renameTo(first)) {
          break;
        }

        if (i++ >= retries) {
          break;
        }

        Thread.sleep(10);// 1/100 second
      }
    }
    else {
      // no backlog: clear output file
      File current = new File(m_fileName);
      current.delete();
    }

    m_output = new FileOutputStream(m_fileName, false);
    m_bytesWritten = 0;
  }
}
