/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.log;

import java.io.*;
import java.text.*;
import java.util.*;

/**
 * Copyright (c) SAP AG 2001-2002
 *
 * @author Sven Walter
 */

public abstract class AbstractChannel {

  private final static char SPACE = ' ';
  private final static char COLON = ':';
  private final static char SEPARATOR = '-';
  private final static char BRACKET_OPEN = '[';
  private final static char BRACKET_CLOSE = ']';
  private final static String LINEBREAK = "\r\n";
  private final static String FATAL = "FATAL";
  private final static String ERROR = "ERROR";
  private final static String WARN = "WARN";
  private final static String INFO = "INFO";
  private final static String DEBUG = "DEBUG";

  public long m_maxSize = 0;// unlimited
  public int m_maxBacklog = 0;// no backlog
  public boolean m_showStackTraces = true;
  public boolean m_showThreads = true;

  protected DateFormat formatter = DateFormat.getDateTimeInstance(DateFormat.MEDIUM, DateFormat.MEDIUM, Locale.getDefault());

  public AbstractChannel(Long maxSize, Integer maxBacklog, Boolean showStackTraces, Boolean showThreads)
    throws Exception {
    if (showStackTraces != null) {
      m_showStackTraces = showStackTraces.booleanValue();
    }
    if (showThreads != null) {
      m_showThreads = showThreads.booleanValue();
    }
    if (maxSize != null) {
      m_maxSize = maxSize.longValue();
    }
    if (maxBacklog != null) {
      m_maxBacklog = maxBacklog.intValue();
    }
  }

  public String getDateTimeString() {
    return formatter.format(new Date());
  }

  public String getLogLevelText(int level) {
    switch (level) {
      case ICategory.LEVEL_FATAL:
        return FATAL;
      case ICategory.LEVEL_ERROR:
        return ERROR;
      case ICategory.LEVEL_WARN:
        return WARN;
      case ICategory.LEVEL_INFO:
        return INFO;
      case ICategory.LEVEL_DEBUG:
        return DEBUG;
    }

    return "";
  }

  public String getLogText(String method, int level, Object message, Throwable throwable) {
    final int approxLength = 255;

    StringBuffer result = new StringBuffer(approxLength);

    result.append(getLogLevelText(level));
    result.append(COLON);
    result.append(SPACE);
    result.append(getDateTimeString());
    result.append(SPACE);
    if (method != null) {
      result.append(method);
      result.append(SPACE);
    }
    result.append(SEPARATOR);
    result.append(SPACE);
    if (message != null) {
      result.append(message.toString());
      result.append(SPACE);
    }
    if (m_showThreads) {
      result.append(BRACKET_OPEN);
      result.append(Thread.currentThread().getName());
      result.append(BRACKET_CLOSE);
      result.append(SPACE);
    }

    if (throwable != null) {
      if (m_showStackTraces) {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        throwable.printStackTrace(new PrintStream(out));
        result.append(LINEBREAK);
        result.append(out.toString());
      }
      else {
        result.append(throwable.toString());
        result.append(LINEBREAK);
      }
    }
    else {
      result.append(LINEBREAK);
    }

    return result.toString();
  }
}
