/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.http.slim;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.util.http.*;

import java.security.MessageDigest;

/**
 * WDDigestAuthentication is a {@link IWDCredentials} which implements HTTP 1.1
 * digest authentication as defined in RFC 2617 with the following exceptions:
 * </p>
 * <ul>
 *   <li> Authentication-Info headers in responses are not evalualted</li>
 *   <li> domain parameters are ignored</li>
 *   <li> auth-int is not supported</li>
 *   <li> proxy authorization is not supported</li>
 *   <li> MD5-sess is not tested</li>
 * </ul>
 * <p>
 *
 * The class was tested against an MS ISS 5.0. <p>
 *
 * Copyright (c) SAP AG 2001-2003
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Id: WDDigestAuthentication.java,v 1.7 2003/02/18 12:35:49 sei Exp $
 */
final class WDDigestAuthentication implements ICredentials {

  private final static com.sap.tc.logging.Location log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.http.slim.WDDigestAuthentication.class);

  private final static char[] nibbleChar = new char[]{
    '0', '1', '2', '3', '4', '5', '6', '7',
    '8', '9', 'a', 'b', 'c', 'd', 'e', 'f',
    };

  /**
   * the context the credentials belongs to
   */
  private final UserInfo m_info;

  /**
   * the MD5 implementation
   */
  private final MessageDigest m_digest;
  /**
   * scratch string buffer
   */
  private final StringBuffer m_sb;

  /**
   * Did we have valid setup information?
   */
  private boolean m_valid;
  /**
   * The realm setup parameter
   */
  private String m_realm;
  /**
   * The nonce setup parameter
   */
  private String m_nonce;
  /**
   * The nonce domain parameter or <code>null</code>
   */
  private String m_domain;
  /**
   * The nonce opaque parameter or <code>null</code>
   */
  private String m_opaque;
  /**
   * The nonce stale parameter or <code>null</code>
   */
  private String m_stale;
  /**
   * The nonce algorithm parameter or <code>null</code>
   */
  private String m_algorithm;
  /**
   * The nonce qop parameter or <code>null</code>
   */
  private String m_qop;

  /**
   * the nc credentials value which is incremented for each request
   */
  private int m_nc;


  public WDDigestAuthentication(UserInfo ui)
    throws WcmException {
    m_info = ui;
    m_sb = new StringBuffer();
    try {
      m_digest = MessageDigest.getInstance("MD5");
    }
    catch (Exception ex) {
      throw new WcmException(ex, false);
    }
  }

  public String getName() {
    return "Digest";
  }

  /**
   * Clears all parameters and sets object to invalid state
   */
  public synchronized void reset() {
    m_valid = false;
    m_realm = null;
    m_nonce = null;
    m_nc = 0;
    m_domain = null;
    m_opaque = null;
    m_stale = null;
    m_algorithm = null;
    m_qop = null;
  }

  public synchronized boolean apply(IRequester requester, String uri, IRequest request, String headerName)
    throws WcmException {
    // No valid setup informatio supplied, empty credentials
    if (!m_valid) {
      return false;
    }

    try {
      // encoding of hashed characters is not really specified in RFC 2617
      // (mayby 8859_1 might be a better choice?
      //
      String encoding = "utf-8";
      // the real uri as seen by the server
      String nc = null;
      String cnonce = null;

      m_sb.setLength(0);
      m_sb.append(m_info.getUser()).append(':');
      m_sb.append(m_realm).append(':');
      m_sb.append(m_info.getPassword());

      // Hash A1
      //
      m_digest.reset();
      byte[] hashA1 = m_digest.digest(m_sb.toString().getBytes(encoding));

      // Hash A1 when MD5-sess is requested
      //
      if (m_algorithm != null && m_algorithm.equalsIgnoreCase("MD5-sess")) {
        m_sb.setLength(0);
        cnonce = generateNonce();
        m_sb.append(':').append(m_nonce).append(':').append(cnonce);
        m_digest.reset();
        m_digest.update(hashA1);
        if (log.beDebug()) {
          log.debugT("apply(163)", "Hashing A1(2): " + m_sb.toString());
        }
        hashA1 = m_digest.digest(m_sb.toString().getBytes(encoding));
      }
      String sessionKey = convert(hashA1);
      if (log.beDebug()) {
        log.debugT("apply(169)", "        is: " + sessionKey);
      }

      // Hash A2
      //
      m_sb.setLength(0);
      m_sb.append(request.getMethod()).append(':').append(uri);
      m_digest.reset();
      if (log.beDebug()) {
        log.debugT("apply(178)", "Hashing A2: " + m_sb.toString());
      }
      byte[] hashA2 = m_digest.digest(m_sb.toString().getBytes(encoding));
      String hashA2Hex = convert(hashA2);
      if (log.beDebug()) {
        log.debugT("apply(183)", "        is: " + hashA2Hex);
      }

      // Hash Response
      //
      m_sb.setLength(0);
      m_sb.append(sessionKey).append(':');
      m_sb.append(m_nonce).append(':');
      if (m_qop != null) {
        nc = incrementNC();
        if (cnonce == null) {
          cnonce = generateNonce();
        }
        m_sb.append(nc).append(':');
        m_sb.append(cnonce).append(':');
        m_sb.append("auth").append(':');
      }
      m_sb.append(hashA2Hex);
      m_digest.reset();
      if (log.beDebug()) {
        log.debugT("apply(203)", "Hashing R : " + m_sb.toString());
      }
      byte[] hashR = m_digest.digest(m_sb.toString().getBytes(encoding));
      String hashRHex = convert(hashR);
      if (log.beDebug()) {
        log.debugT("apply(208)", "        is: " + hashRHex);
      }

      // Generate the credentials
      //
      m_sb.setLength(0);
      m_sb.append("Digest username=\"").append(m_info.getUser());
      m_sb.append("\", realm=\"").append(m_realm);
      m_sb.append("\", nonce=\"").append(m_nonce);
      m_sb.append("\", uri=\"").append(uri).append('\"');
      if (m_qop != null) {
        m_sb.append(", qop=\"auth\", nc=").append(nc);
        m_sb.append(", cnonce=\"").append(cnonce);
      }
      m_sb.append("\", response=\"").append(hashRHex).append("\"");
      if (m_opaque != null) {
        m_sb.append(", opaque=\"").append(m_opaque).append("\"");
      }

      String result = m_sb.toString();
      if (log.beDebug()) {
        log.debugT("apply(229)", "Digest result: " + result);
      }
      request.setHeader(headerName, result);
      return true;
    }
    catch (Exception ex) {
      if (log.beDebug()) {
        log.debugT("apply(236)", "while calculating credentials" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(ex));
      }
      throw new WcmException(ex, false);
    }
  }

  /**
   * Set the credentials for this context depending on the header information
   * (received from a 401).
   *
   * @param requester TBD: Description of the incoming method parameter
   * @param header TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception WcmException Exception raised in failure situation
   */
  public synchronized boolean setup(IRequester requester, String header)
    throws WcmException {
    return setup(requester, header, false);
  }

  /**
   * Set the credentials for this context depending on the header information
   * (received from a 401).
   *
   * @param requester TBD: Description of the incoming method parameter
   * @param header TBD: Description of the incoming method parameter
   * @param retry TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception WcmException Exception raised in failure situation
   */
  public synchronized boolean setup(IRequester requester, String header, boolean retry)
    throws WcmException {
    if (log.beDebug()) {
      log.debugT("setup(269)", "SETUP: " + header);
    }

    // clear and invalidate this
    reset();

    // Header values we are interested in
    //
    String[] keys = new String[]{
      "realm=", "nonce=", "opaque=", "domain=", "stale=", "qop=", "algorithm="
      };
    String[] values = new String[keys.length];

    for (int i = 0; i < keys.length; ++i) {
      int index = header.indexOf(keys[i]);
      if (index >= 0) {
        values[i] = getUnquoted(header, index + (keys[i].length()));
      }
    }

    m_realm = values[0];
    m_nonce = values[1];
    m_opaque = values[2];
    m_domain = values[3];
    m_stale = values[4];
    m_qop = values[5];
    m_algorithm = values[6];

    if (m_realm == null || m_nonce == null) {
      // required values missing
      if (log.beDebug()) {
        log.debugT("setup(300)", "realm or nonce null, realm=" + m_realm + ", nonce=" + m_nonce);
      }
      return false;
    }
    if (m_algorithm != null
       && !m_algorithm.equalsIgnoreCase("MD5")
       && !m_algorithm.equalsIgnoreCase("MD5-SESS")) {
      // Unsupported algorithm
      if (log.beDebug()) {
        log.debugT("setup(309)", "unsupported algorithm, " + m_algorithm);
      }
      return false;
    }
    if (m_stale != null && retry && m_stale.equalsIgnoreCase("FALSE")) {
      // This indicates that user/passwd values originally
      // sent are not valid
      if (log.beDebug()) {
        log.debugT("setup(317)", "on retry got non-stale answer, meaning user/pass pair is not accepted");
      }
      return false;
    }

    // only here is the setup valid and we will generate credentials
    //
    m_valid = true;
    return m_valid;
  }

  /**
   * Process authenticate information in the response message, like
   * Authenticate-Info
   *
   * @param requester TBD: Description of the incoming method parameter
   * @param response TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   */
  public synchronized void got(IRequester requester, IResponse response)
    throws WcmException {
    String value = response.getHeader("Authentication-Info");
    if (value == null) {
      return;
    }

    // This does nothing yet, but cry out loud if we ever
    // see such a thing
    //
  }

  public void startUse(IRequester requester) { }

  public void endUse(IRequester requester) { }

  public boolean canTriggerAuthentication(IRequester requester) {
    return false;
  }

  // ------------------ private / protected ---------------------------

  /**
   * Extract the contents of a quoted string from line, starting at position
   * start.
   *
   * @param line TBD: Description of the incoming method parameter
   * @param start TBD: Description of the incoming method parameter
   * @return unquoted
   */
  private static String getUnquoted(String line, int start) {
    if (line.charAt(start) != '\"') {
      return null;
    }
    StringBuffer sb = new StringBuffer();
    int len = line.length();
    for (int i = start + 1; i < len; ++i) {
      char c = line.charAt(i);
      switch (c) {
        case '\"':
          return sb.toString();
        default:
          sb.append(c);
      }
    }
    // unterminated quoted string
    return null;
  }

  private static String convert(byte[] bytes) {
    StringBuffer sb = new StringBuffer(bytes.length * 2);
    for (int i = 0; i < bytes.length; ++i) {
      sb.append(nibbleChar[(((int)bytes[i]) & 0xf0) >> 4]);
      sb.append(nibbleChar[((int)bytes[i]) & 0x0f]);
    }

    return sb.toString();
  }

  /**
   * Increment our nc value for the next request
   *
   * @return TBD: Description of the outgoing return value
   */
  private synchronized String incrementNC() {
    int count = ++m_nc;
    char[] chars = new char[8];
    for (int i = 7; i >= 0; --i) {
      chars[i] = nibbleChar[count & 0x0f];
      count >>>= 4;
    }
    return new String(chars);
  }

  /**
   * Generate a client nonce value for the server
   *
   * @return TBD: Description of the outgoing return value
   */
  private String generateNonce() {
    return String.valueOf(System.currentTimeMillis());
  }

//  private void teststring(String s, String real) {
//    System.out.print("MD5(\"" + s + "\") = ");
//
//    m_digest.reset();
//    m_digest.update(s.getBytes());
//    String hex = convert(m_digest.digest());
//
//    if (real == null || hex.equals(real))
//      System.out.println("ok");
//    else
//      System.out.println(hex + " should be " + real);
//  }
//
//  public void testsuite() {
//    teststring("", "d41d8cd98f00b204e9800998ecf8427e");
//    teststring("a", "0cc175b9c0f1b6a831c399e269772661");
//    teststring("abc", "900150983cd24fb0d6963f7d28e17f72");
//    teststring("message digest", "f96b697d7cb7938d525a2f31aaf161d0");
//    teststring("abcdefghijklmnopqrstuvwxyz", "c3fcd3d76192e4007dfb496cca67e13b");
//    teststring("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789", "d174ab98d277d9f5a5611c2c9f419d9f");
//    teststring("12345678901234567890123456789012345678901234567890123456789012345678901234567890", "57edf4a22be3c955ac49da2e2107b67a");
//  }
//
//  public static void main(String[] args) throws Exception {
//    IWDContext context = new com.sapportals.wcm.util.http.slim.SlimContext("Mufasa", "Circle of Life");
//    WDDigestAuthentication credentials = new WDDigestAuthentication(context.getUserInfo());
//
//    // Check MD5 encodings and our hex converter
//    //
//    credentials.testsuite();
//
//    // Check example from RFC 2617 Ch. 3.5
//    //
//    String expected = "6629fae49393a05397450978507c4ef1";
//    credentials.setup("Digest realm=\"testrealm@host.com\", "
//                      +"qop=\"auth,auth-int\", "
//                      +"nonce=\"dcd98b7102dd2f0e8b11d0f600bfb0c093\", "
//                      +"opaque=\"5ccc069c403ebaf9f0171e9517f40e41\"");
//    WDRequest request = new WDRequest();
//    request.setMethod("GET");
//    request.setURI("/dir/index.html");
//    String response = credentials.calculate("", request);
//    int index = response.indexOf("response=");
//    if (index >= 0) {
//      String hash = getUnquoted(response, index+"response=".length());
//      if (hash.equals(expected)) {
//        System.out.println("Digest test ok!");
//      }
//      else {
//        System.out.println("Digest got \""+hash+"\"\n should be \""+expected+"\"");
//      }
//    }
//    else {
//      System.out.println("Digest did not find response in: "+response);
//    }
//  }
}

