/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.http.slim;

import java.io.UnsupportedEncodingException;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.util.http.*;

/**
 * WDBasicAuthentication is a {@link IWDCredentials} which implements HTTP 1.1
 * basic authentication as defined in RFC 2617. <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Id: WDBasicAuthentication.java,v 1.5 2003/11/27 12:52:07 sei Exp $
 */
final class WDBasicAuthentication implements ICredentials {
  
  private static final String[] Encodings = {
    "UTF-8",
    "ISO-8859-1",
    null,
  };

  private final UserInfo info;
  private String credentials;
  private int enc;

  public WDBasicAuthentication(UserInfo ui) {
    this.info = ui;
  }

  public String getName() {
    return "Basic";
  }

  /**
   * Calculate the credentials for this HTTP Request
   *
   * @param requester TBD: Description of the incoming method parameter
   * @param uri TBD: Description of the incoming method parameter
   * @param request TBD: Description of the incoming method parameter
   * @param headerName TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception WcmException Exception raised in failure situation
   */
  public boolean apply(IRequester requester, String uri, IRequest request, String headerName)
    throws WcmException {
    if (this.credentials == null) {
      try {
        String message = this.info.getUser() + ":" + this.info.getPassword();
        String encoding = getEncoding();
        if (encoding != null) {
          this.credentials = "Basic " + Base64.encode(message, encoding);
        }
        else {
          this.credentials = "Basic " + Base64.encode(message);
        }
      }
      catch (UnsupportedEncodingException ex) {
        throw new WcmException(ex);
      }
    }
    request.setHeader(headerName, this.credentials);
    return true;
  }

  /**
   * Set the credentials for this context depending on the header information
   * (received from a 401 response).
   *
   * @param requester TBD: Description of the incoming method parameter
   * @param header TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception WcmException Exception raised in failure situation
   */
  public boolean setup(IRequester requester, String header)
    throws WcmException {
    return setup(requester, header, false);
  }

  /**
   * Set the credentials for this context depending on the header information
   * (received from a 401 response).
   *
   * @param requester TBD: Description of the incoming method parameter
   * @param header TBD: Description of the incoming method parameter
   * @param retry TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception WcmException Exception raised in failure situation
   */
  public boolean setup(IRequester requester, String header, boolean retry)
    throws WcmException {
    // basic authentication is independant on any parameters
    this.credentials = null;
    if (retry) {
      this.nextEncoding();
    }
    return true;
  }

  /**
   * Process authenticate information in the response message, like
   * Authenticate-Info
   *
   * @param requester TBD: Description of the incoming method parameter
   * @param response TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   */
  public void got(IRequester requester, IResponse response)
    throws WcmException { }

  public void startUse(IRequester requester) { }

  public void endUse(IRequester requester) { }

  public boolean canTriggerAuthentication(IRequester requester) {
    return false;
  }
  
  private void nextEncoding() {
    ++this.enc;
  }
  
  private String getEncoding() {
    if (this.enc >= 0 && this.enc < Encodings.length) {
      return Encodings[this.enc];
    }
    return null;
  }
}

