/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.http.slim;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.util.http.IContext;
import com.sapportals.wcm.util.uri.HttpUrl;

import iaik.security.ssl.SSLClientContext;

import java.io.IOException;
import java.net.Socket;
import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.Map;

/**
 * Pool of requesters for a particular server. <p>
 *
 * Copyright (c) SAP AG 2001-2003
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Id: SlimServerPool.java,v 1.6 2003/02/17 14:24:04 jre Exp $
 */
final class SlimServerPool {

  private final static com.sap.tc.logging.Location log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.http.slim.SlimServerPool.class);

  private final static long DEFAULT_TIMEOUT = 1 * 5 * 1000;

  private final static int SINGLE_POOL_MAX = 4;
  // private static final int USER_POOL_MAX = 1;

  private final static Map SERVERS = new HashMap(1024);

  public static SlimServerPool getInstance(HttpUrl server) {
    SlimServerPool pool = null;
    synchronized (SERVERS) {
      pool = (SlimServerPool)SERVERS.get(server);
      if (pool == null) {
        pool = new SlimServerPool(server);
        SERVERS.put(server, pool);
      }
    }

    return pool;
  }

  private final HttpUrl server;
  private final SlimRequesterPool requesterPool;
  private final long timeout;
  private int maxRequester;
  private int activeRequester;
  private int maxWaitMS;

  private SlimServerPool(HttpUrl server) {
    this.server = server;
    this.timeout = DEFAULT_TIMEOUT;
    this.requesterPool = new SlimRequesterPool(SINGLE_POOL_MAX, this.timeout);
    this.maxRequester = Integer.MAX_VALUE;
    this.activeRequester = 0;
    this.maxWaitMS = 10000;
  }

  public int hashCode() {
    return this.server.hashCode();
  }

  public boolean equals(Object other) {
    if (other instanceof SlimServerPool) {
      SlimServerPool o = (SlimServerPool)other;
      return this.server.equals(o.server);
    }
    return false;
  }

  public String toString() {
    return "SlimServerPool[" + this.server + ", " + this.requesterPool + "]";
  }

  public HttpUrl getServer() {
    return this.server;
  }

  public long getTimeout() {
   return this.timeout; 
  }
  
  public synchronized SlimRequester getRequester(IContext context)
    throws WcmException {
    long endOfWait = System.currentTimeMillis() + this.maxWaitMS;
    while (this.activeRequester >= this.maxRequester) {
      int remaining = (int)(endOfWait - System.currentTimeMillis());

      if (log.beDebug()) {
        log.debugT("waiting for requester to become available: active "
          + this.activeRequester+", max "+this.maxRequester+" -- "+this.server);
      }
      if (remaining <= 0) {
        // We have been throwing exceptions before, but that makes the system
        // unstable and non-deterministic. Reason is that we have to rely on the
        // garbage collector to finalize requester and that might take minutes.
        // So, new strategy: we shape traffic by waiting for requester to become
        // available, but when the time is up, we pretend that a requester was lost
        // and hand out a new one.
        // This gives us deterministic behaviour while sacrificing optimal connection
        // traffic shaping which is a subgoal anyway.
        //throw new WcmException("(" + this.server + ")timeout waiting for connection to become available");
        decrActive();
        break;
      }

      try {
        this.wait(remaining);
      }
      catch (InterruptedException e) {
        // ignore
      }
    }

    SlimRequesterPool pool = getRequesterPool(context);
    SlimRequester requester = pool.get(context);
    if (requester != null) {
      requester.setContext(context);
      incrActive();
      if (log.beDebug()) {
        log.debugT("getRequester(114)", "" + this + " handing out pooled requester " + requester + ", active: " + this.activeRequester);
      }
    }
    else {
      requester = new SlimRequester(this, context);
      incrActive();
      if (log.beDebug()) {
        log.debugT("getRequester(121)", "" + this + " handing out new requester " + requester + ", active: " + this.activeRequester);
      }
    }
    return requester;
  }

  public Socket createSocket(String protocol, String host, int port)
    throws IOException, UnknownHostException {
    if (protocol.equals("https")) {
      return SecureSockets.S.createSocket(host, port);
    }
    else {
      log.infoT("createSocket(133)", "creating plain socket");
      return new Socket(host, port);
    }
  }

  public Socket upgradeTLS(Socket socket, HttpUrl url)
    throws IOException, UnknownHostException {
    return SecureSockets.S.upgradeTLS(socket, url);
  }

  public synchronized void reuse(SlimRequester requester) {
    SlimRequesterPool pool = getRequesterPool(requester.getContext());
    pool.put(requester);
    decrActive();
    this.notifyAll();
    if (log.beDebug()) {
      log.debugT("reuse(exit)", "" + this + "returned requester into pool, active: " + this.activeRequester);
    }
  }

  public synchronized void requesterDestroyed() {
    decrActive();
    this.notifyAll();
    if (log.beDebug()) {
      log.debugT("requesterDestroyed(157)", "requester finalized, active(" +
        this.server + "): " + this.activeRequester);
    }
  }

  public synchronized void setMaxConnections(int max) {
    if (max > 0 && max < 65000) {
      this.maxRequester = max;
    }
    else {
      this.maxRequester = Integer.MAX_VALUE;
    }
  }

  //------------------------------- private -------------------------------------
  
  private void decrActive() {
    if (this.activeRequester > 0) {
      --this.activeRequester;
    }
  }

  private void incrActive() {
    if (this.activeRequester < this.maxRequester) {
      ++this.activeRequester;
    }
  }
  
  private SlimRequesterPool getRequesterPool(IContext context) {
    return this.requesterPool;
  }

// This version uses javax.net.ssl, which is standard for SSL in JDK 1.4
// Due to legalese, we are not allowed to use JSSE from sun.
//
//  private static class SecureSockets {
//
//    public static final SecureSockets S = new SecureSockets();
//
//    private final SocketFactory sslFactory;
//    private final Method sslUpgrade;
//
//    private SecureSockets() {
//      log.infoT("initializing SSL Factory and methods");
//
//      SocketFactory l_sslFactory = null;
//      Method l_sslUpgrade = null;
//      String classname = "javax.net.ssl.SSLSocketFactory";
//      try {
//        String provider = System.getProperty("ssl.Provider");
//        if (provider == null || provider.length() == 0) {
//          provider = "com.sun.net.ssl.internal.ssl.Provider";
//        }
//        Class providerClass = Class.forName(provider);
//        java.security.Security.addProvider((java.security.Provider)providerClass.newInstance());
//
//        Class fac = Class.forName(classname);
//        Method m = fac.getMethod("getDefault", null);
//        l_sslFactory = (SocketFactory)m.invoke(fac, null);
//        fac = l_sslFactory.getClass();
//
//        try {
//          Class[] args = new Class[] { Socket.class, String.class,
//                                       int.class, boolean.class };
//          l_sslUpgrade = fac.getMethod("createSocket", args);
//        }
//        catch (Exception ex) {
//          log.fatalT("retrieving method createSocket" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(ex));
//          l_sslUpgrade = null;
//        }
//      }
//      catch (Exception ex) {
//        log.fatalT("Initialising SSL Socket Factory" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(ex));
//        l_sslFactory = null;
//      }
//
//      sslFactory = l_sslFactory;
//      sslUpgrade = l_sslUpgrade;
//    }
//
//    public Socket upgradeTLS(Socket socket, HttpUrl url)
//    throws IOException, UnknownHostException {
//      log.infoT("upgrading existing socket to TLS");
//      if (sslUpgrade == null) {
//        throw new IOException("upgrade to TLS not available");
//      }
//
//      try {
//        Object[] args = new Object[] { socket, url.getHost(), new Integer(url.getPort()), new Boolean(true) };
//        return (Socket)sslUpgrade.invoke(sslFactory, args);
//      }
//      catch (Exception ex) {
//        log.errorT("cannot upgrade socket to TLS" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(ex));
//        throw new IOException("cannot upgrade socket to TLS");
//      }
//    }
//
//    public Socket createSocket(String host, int port)
//    throws IOException {
//      log.infoT("creating SSL socket");
//      if (sslFactory == null) {
//        throw new IOException("SocketFactory for SSL/TLS not available");
//      }
//
//      return sslFactory.createSocket(host, port);
//    }
//
//  }

  /**
   * TBD: Description of the class.
   */
  private static class SecureSockets {

    public final static SecureSockets S = new SecureSockets();

    private final SSLClientContext sslContext;

    private SecureSockets() {
      log.infoT("SecureSockets(264)", "initializing SSL Factory and methods");
      this.sslContext = new SSLClientContext();
    }

    public Socket upgradeTLS(Socket socket, HttpUrl url)
      throws IOException, UnknownHostException {
      log.infoT("upgradeTLS(270)", "upgrading existing socket to TLS");

      return new iaik.security.ssl.SSLSocket(socket, this.sslContext, url.getHost(), url.getPort());
    }

    public Socket createSocket(String host, int port)
      throws IOException {
      log.infoT("createSocket(277)", "creating SSL socket");

      return new iaik.security.ssl.SSLSocket(host, port, this.sslContext);
    }

  }

}
