/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.http.slim;

import com.sapportals.wcm.util.http.IContext;

import java.util.ArrayList;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;

/**
 * SlimRequesterPool can keep a maximum number of IWDRequester objects for a
 * certain amount of time. <p>
 *
 * The pool is initialized with a maximum number of objects to keep and the
 * amount of milliseconds, the objects can linger in the cache before being
 * discarded. <p>
 *
 * Copyright (c) SAP AG 2001-2003
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Id: SlimRequesterPool.java,v 1.5 2003/02/17 14:24:04 jre Exp $
 */
final class SlimRequesterPool {

  private final static com.sap.tc.logging.Location log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.http.slim.SlimRequesterPool.class);

  /**
   * TBD: Description of the class.
   */
  private static class T {
    final static Timer timer = new Timer(true);
  }

  /**
   * TBD: Description of the class.
   */
  private static class RequesterEntry {
    public final SlimRequester requester;
    public final long returned;

    public RequesterEntry(SlimRequester requester) {
      this.requester = requester;
      this.returned = System.currentTimeMillis();
    }

    public String toString() {
      return "SRENtry[" + this.requester + "]";
    }
    
    public boolean equals(Object o) {
      if (o instanceof RequesterEntry) {
        return this.requester.equals(((RequesterEntry)o).requester);
      }
      return false;
    }
    
    public int hashCode() {
      return this.requester.hashCode();
    }
  }

  private final List pool;

  private int maxPoolSize;
  private final long timeout;

  // private boolean registered;
  private TimerTask client;

  /**
   * Construct a new SlimRequesterPool
   *
   * @param maxSize maximum number of requesters to keep
   * @param timeout milliseconds that a requester can hand around
   */
  public SlimRequesterPool(int maxSize, long timeout) {
    this.maxPoolSize = maxSize;
    this.timeout = timeout;
    this.pool = new ArrayList(this.maxPoolSize);
  }

  /**
   * @param context TBD: Description of the incoming method parameter
   * @return a pooled SlimRequester if <code>null</code> if the pool is empty
   */
  public synchronized SlimRequester get(IContext context) {
    return getLatest(context);
  }

  /**
   * Put a SlimRequester back into the pool
   *
   * @param requester the requester which is put into the pool
   */
  public synchronized void put(SlimRequester requester) {
    log.debugT("put(92)", "POOL: put requester");
    RequesterEntry entry = new RequesterEntry(requester);
    if (!contains(entry)) {
      int oldSize = this.pool.size();
      if (oldSize >= this.maxPoolSize) {
        SlimRequester oldest = getOldest();
        oldest.discard();
      }

      this.pool.add(entry);
      boolean schedule = false;
      if (oldSize == 0 && this.client == null) {
        this.client =
          new TimerTask() {
            public void run() {
              if (!tick()) {
                SlimRequesterPool.this.client = null;
                cancel();
              }
            }
          };
        schedule = true;
      }

      if (schedule) {
        T.timer.schedule(this.client, this.timeout, this.timeout);
      }
    }
    else {
      log.infoT("put(121)", "putting requester which is already in pool");
    }
  }

  /**
   * Get the maximum pool size
   *
   * @return the maximum size of the pool
   */
  public synchronized int getMaxSize() {
    return this.maxPoolSize;
  }

  /**
   * Sets the maximum pool size. Will shrink the pool if necessary.
   *
   * @param max
   * @throws IllegalArgumentException if max is not greater than 0
   */
  public synchronized void setMaxSize(int max) {
    if (max < 1) {
      throw new IllegalArgumentException("max must be > 0");
    }
    while (max < this.pool.size()) {
      RequesterEntry entry = (RequesterEntry)this.pool.remove(0);
      entry.requester.discard();
    }
    this.maxPoolSize = max;
  }

  public synchronized int size() {
    return this.pool.size();
  }

  public synchronized void discard() {
    SlimRequester old;
    while ((old = getOldest()) != null) {
      old.discard();
    }
  }

  public synchronized String toString() {
    return "SlimRequesterPool[" + this.pool + "]";
  }

  //------------------------------ private ----------------------------------

  /**
   * Tells if a requester is already in the pool
   *
   * @param entry TBD: Description of the incoming method parameter
   * @return if requester is alread in pool
   */
  private boolean contains(RequesterEntry entry) {
    return this.pool.contains(entry);
  }

  /**
   * @param context TBD: Description of the incoming method parameter
   * @return the latest pooled SlimRequester if <code>null</code> if the pool is
   *      empty
   */
  private SlimRequester getLatest(IContext context) {
    int size = this.pool.size();
    if (size > 0) {
      if (log.beDebug()) {
        log.debugT("getLatest(187)", "" + this + " getLatest: start");
      }
      RequesterEntry entry = null;
      if (context != null) {
        for (int i = size - 1; i >= 0; --i) {
          RequesterEntry e = (RequesterEntry)this.pool.get(i);
          if (context.equals(e.requester.getContext())) {
            entry = (RequesterEntry)this.pool.remove(i);
            break;
          }
        }
      }

      if (entry == null) {
        entry = (RequesterEntry)this.pool.remove(size - 1);
      }
      if (log.beDebug()) {
        log.debugT("getLatest(204)", "" + this + " getLatest: got " + entry);
      }
      return entry.requester;
    }
    else {
      if (log.beDebug()) {
        log.debugT("getLatest(210)", "" + this + " getLatest: pool is empty");
      }
      return null;
    }
  }

  /**
   * @return the oldest pooled SlimRequester if <code>null</code> if the pool is
   *      empty
   */
  private SlimRequester getOldest() {
    if (this.pool.size() > 0) {
      log.debugT("getOldest(222)", "POOL: get requester");
      RequesterEntry entry = (RequesterEntry)this.pool.remove(0);
      return entry.requester;
    }
    else {
      log.debugT("getOldest(227)", "POOL: get requester == null");
      return null;
    }
  }

  /**
   * Perform cleanup tasks such as removing old requester
   *
   * @return TBD: Description of the outgoing return value
   */
  private synchronized boolean tick() {
    if (this.pool.isEmpty()) {
      return false;
    }
    else {
      if (log.beDebug()) {
        log.debugT("tick(243)", "checking for pool cleanup of: " + this.pool);
      }
      long now = System.currentTimeMillis();
      while (!this.pool.isEmpty()) {
        RequesterEntry entry = (RequesterEntry)this.pool.get(0);
        if (now - entry.returned > this.timeout) {
          this.pool.remove(0);
          entry.requester.discard();
        }
        else {
          break;
        }
      }
      return true;
    }
  }

}

