/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.http.slim;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.util.http.IContext;
import com.sapportals.wcm.util.http.IProxyInfo;
import com.sapportals.wcm.util.http.IRequester;
import com.sapportals.wcm.util.http.IRequesterFactory;
import com.sapportals.wcm.util.uri.HttpUrl;
import com.sapportals.wcm.util.uri.IHierarchicalUri;
import com.sapportals.wcm.util.uri.UriFactory;

import java.net.MalformedURLException;
import java.util.HashMap;
import java.util.Map;

/**
 * Implementation of IWDRequesterFactory for the slim package. <p>
 *
 * SlimRequesterFactory is caching IWDRequester objects per IWDContext. There is
 * a maximum number of cached (not constructed) objects per context. Requester
 * objects time out after a while and are removed. There is not limit on the
 * number of IWDRequester per context, however when requester are released, only
 * a certain number is kept for further use. <p>
 *
 * See {@link SlimRequester} for information about open connections. <p>
 *
 * Copyright (c) SAP AG 2001-2003
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Id: SlimRequesterFactory.java,v 1.3 2003/02/17 14:24:04 jre Exp $
 */
public final class SlimRequesterFactory implements IRequesterFactory {

  // private static final com.sap.tc.logging.Location log = com.sap.tc.logging.Location.getLocation(SlimRequesterFactory.class.getName());

  private final static IContext ANONYMOUS = new SlimContext();

  private HttpUrl m_url;
  private IProxyInfo pi;

  private final Map contexts;
  private SlimServerPool serverPool;

  private int m_soTimeout;

  public SlimRequesterFactory() {
    this.contexts = new HashMap(1025);
  }

  /**
   * @param uri TBD: Description of the incoming method parameter
   * @return the full URL for the given relative URI
   */
  public String getAbsoluteURL(String uri) {
    return m_url.appendPath(uri).toString();
  }

  public synchronized HttpUrl getProxy() {
    return (this.pi != null)? this.pi.getProxyFor(this.m_url) : null;
  }

  public IRequester getRequester(IContext context)
    throws WcmException {
    if (context == null) {
      context = SlimRequesterFactory.ANONYMOUS;
    }

    SlimRequester requester = this.serverPool.getRequester(context);

    requester.setProxy(getProxy());
    requester.setSoTimeout(m_soTimeout);

    return requester;
  }

  public IContext getContext(Object key) {
    SlimContext context = null;
    synchronized (this.contexts) {
      context = (SlimContext)this.contexts.get(key);
      if (context == null) {
        context = new SlimContext();
        this.contexts.put(key, context);
      }
    }
    return context;
  }

  /**
   * @return the port number
   */
  public int getPort() {
    return m_url.getPort();
  }

  public void setBaseURL(String url)
    throws MalformedURLException {

    IHierarchicalUri uri = UriFactory.parseHierarchicalUri(url);
    if (!(uri instanceof HttpUrl)) {
      throw new MalformedURLException("no http url " + uri);
    }

    m_url = (HttpUrl)uri;

    // boolean useTLS = m_url.getScheme().equals("https");
    this.serverPool = SlimServerPool.getInstance(m_url);
  }

  public HttpUrl getBase() {
    return m_url;
  }

  /**
   * Sets the proxy information to use by this factory.
   *
   * @param info proxyInfo to be set
   */
  public synchronized void setProxyInfo(IProxyInfo info) {
    this.pi = info;
  }

  public void setSoTimeout(int ms) {
    if (ms < 0) {
      throw new IllegalArgumentException("timeout cannot be < 0");
    }
    m_soTimeout = ms;
  }

  public int getSoTimeout() {
    return m_soTimeout;
  }

  public void setMaxConnections(int max) {
    this.serverPool.setMaxConnections(max);
  }

  // --------------------------- protected / private ----------------------------

}

