/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.http.slim;

import java.io.*;
import java.net.*;
import java.util.*;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.util.http.*;
import com.sapportals.wcm.util.uri.HttpUrl;
import com.sapportals.wcm.util.uri.IUri;
import com.sapportals.wcm.util.uri.IUriReference;
import com.sapportals.wcm.util.uri.UriFactory;
import com.sapportals.wcm.util.xml.DomBuilder;
import com.sapportals.wcm.util.xml.SimpleSerializer;

/**
 * SlimRequester is an implementation of the IWDRequester interface. <p>
 *
 * SlimRequester handles HTTP/1.0 and 1.1 connections. It also supports HTTP
 * proxies. <p>
 *
 * HTTP/1.1 connections are kept open either until an error is encountered or
 * until the requester is explicitly closed. <p>
 *
 * SlimRequester will not follow redirects but deliver 302 responses to the
 * client. <p>
 *
 * SlimRequester is <b>not</b> multithread safe. The client is expected to make
 * all calls to the requester in defined order (e.g. not interleaved).
 * Multithread clients should use the IWDRequesterFactory interface to receive
 * one SlimRequester per thread. On the other hand, SlimRequester is also not
 * thread-bound. <p>
 *
 * Copyright (c) SAP AG 2001-2004
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Id: SlimRequester.java,v 1.7 2004/07/20 10:16:35 sei Exp $
 */
final class SlimRequester implements IRequester {

  private final static com.sap.tc.logging.Location log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.http.slim.SlimRequester.class);

  private final static String HTTP11_ID = "HTTP/1.1";
  private final static String HTTP10_ID = "HTTP/1.0";
  private final static String HTTP0x_ID = "HTTP/0.";

  private final static int HTTP_UNKOWN = -1;
  private final static int HTTP_10 = 0;
  private final static int HTTP_11 = 1;

  private final static String DEFAULT_ENCODING = "UTF-8";

  public final static String NEWLINE = "\r\n";

  private final static byte[] NEWLINE_BYTES = new byte[]{0x0d, 0x0a};
  private final static byte[] ZERO_BYTES = new byte[]{0x30, 0x0d, 0x0a};

  private final static int COPY_BUFFER_SIZE = 64 * 1024;
  private final static int OUTPUT_BUFFER_SIZE = 4 * 1024;

  private final static int CONNECT_TIMEOUT_DEFAULT = 10 * 1000;
  private final static int CONNECT_FAILURES_MAX = 4;
  private final static int MAX_TRIES = 4;

  private final static String USER_AGENT = "java:com.sapportals.wcm.protocol.webdav.http.slim $Revision: 1.1 $";
  private final static boolean USE_USER_AGENT = false;

  private static int instances;

  private static String getNextId() {
    if (log.beInfo()) {
      synchronized (SlimRequester.class) {
        return String.valueOf(++instances);
      }
    }
    return "";
  }

  private final static Set SAFE_METHODS;
  static {
    SAFE_METHODS = new HashSet();
    SAFE_METHODS.add("GET");
    SAFE_METHODS.add("HEAD");
    SAFE_METHODS.add("OPTIONS");
    SAFE_METHODS.add("PROPFIND");
    SAFE_METHODS.add("REPORT");
    SAFE_METHODS.add("SEARCH");
  }

  private static boolean isSafeMethod(String method) {
    return SAFE_METHODS.contains(method);
  }

  private final String id;
  private final SlimServerPool pool;
  private final HttpUrl base;
  private HttpUrl proxyDefault;

  private IContext context;

  /**
   * reads header lines
   */
  private final HeaderReader headerReader;

  /**
   * remote server talks http/1.1
   */
  private int serverHTTPVersion = HTTP_UNKOWN;

  /**
   * our connection to the outside world
   */
  private Socket socket;
  private InputStream input;
  private OutputStream output;
  private OutputStream socketOutput;
  private boolean isFreshConnection = true;

  /**
   * timeout for socket io operations (0 = infinite)
   */
  private int soTimeoutMS;

  /**
   * timeout for connect operations
   */
  private int connectTimeoutMS;

  /**
   * indicates that connection needs to be closed before next use
   */
  private boolean closeOnNextUse;

  /**
   * indicates that someone called release, but there was a response still
   * dangling around.
   */
  private boolean releasePending;

  /**
   * reponse on which the client may read input, needs ot be discarded on next
   * use of requester
   */
  private IResponse danglingResponse;

  private boolean discarded;

  /**
   * Internal helper to keep track of the various properties of the current
   * request and its connection
   */
  private static class RequestStatus {
    public boolean definedLength;// request has known content length
    public boolean willClose;// connection will be closed afterwards
    public boolean needsTLSUpgrade;// connection needs upgrade to TLS
    public boolean talkingToProxy;// we are talking to a proxy (not tunneled or direct connection)
    public boolean clientWillRead;// our client/caller will read the body himself
    public boolean isSSL;// is SSL connection

    public int connectFailures;// how often we tried to establish a connection
    public int IOErrors;// number of connection erros encountered
    public int tries;// number of tries made for request

    public HttpUrl proxy;// proxy to use for request

    public RequestStatus(HttpUrl proxy) {
      this.proxy = proxy;
      this.talkingToProxy = (this.proxy != null);
    }

    /**
     * Get a new status usable for direct proxy communication
     */
    public RequestStatus forProxy() {
      RequestStatus rproxy = new RequestStatus(this.proxy);
      return rproxy;
    }
  }

  /**
   * Constructor for new requster
   *
   * @param pool of requesters this one belongs to
   * @param context user information for non-anonymous requests
   */
  protected SlimRequester(SlimServerPool pool, IContext context) {
    this.id = getNextId();
    this.pool = pool;
    this.context = context;
    this.base = this.pool.getServer();
    this.proxyDefault = null;
    this.headerReader = new HeaderReader();
    this.connectTimeoutMS = CONNECT_TIMEOUT_DEFAULT;

    this.context.startUse(this);
  }

  protected synchronized void setProxy(HttpUrl proxy) {
    if (this.proxyDefault != proxy) {
      // If proxy changes, make sure open connections are closed
      //
      if (this.proxyDefault == null || !this.proxyDefault.equals(proxy)) {
        try {
          closeConnection();
        }
        catch (WcmException e) {
          log.infoT("setProxy(218)", "closing connection" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
        }
      }
      this.proxyDefault = proxy;
    }
  }

  /**
   * Performs a HTTP request with information from request object
   *
   * @return the HTTP response received from the host
   * @throws WcmException on misformed requests or connection failures
   */
  public synchronized IResponse perform(IRequest request)
    throws WcmException {
    RequestStatus rstat = new RequestStatus(this.proxyDefault);
    // request is repeatable if without body or body is given as string.
    // otherwise we are reading from an inputstream...
    //
    boolean repeatable = (!request.hasBody() || (request.getBody() != null));

    // if not repeatable and context might require HTTP authentication
    // send dummy request beforehand to trigger authentication
    if (!repeatable && this.isFreshConnection && this.context.canTriggerAuthentication(this)) {
      HttpRequest dummy = new HttpRequest(request.getURI());
      dummy.setMethod("OPTIONS");
      doPerform(dummy, rstat, true);
      // we are not interested in the result of this. It's now
      // time to try the real thing an return that response
    }

    requesting :
    while (true) {
      IResponse response = doPerform(request, rstat, repeatable);
      switch (response.getStatus()) {
        case 300:
          // 300 Multiple Choices - no generic handling known, return to caller
          break;
        case 301:
        // 301 Moved Permanently, report to caller
        case 302:
          // 302 Move Temporarily, report to caller
          break;
        case 303:
          // 303 See other, response can be GET und location header URI, Should we follow automatically?
          break;
        case 304:
          // 304 Not Modified, must be handled by caller
          break;
        case 305:
          if (rstat.proxy == null && repeatable) {
            // 305 Use Proxy, should we repeat request against proxy? Should we remember the proxy setting?
            IUri location = getLocation(response, request);
            if (location instanceof HttpUrl) {
              rstat.proxy = (HttpUrl)location;
              continue requesting;
            }
            else {
              log.infoT("perform(278)", "location header not a HTTP URI: " + location);
              // return response to caller
            }
          }
          else {
            // already talking to proxy or not repeatable, report back to caller
          }
          break;
        case 306:
          // Unused, reserved. Return to caller
          break;
        case 307:
          // 307 Temporary Redirect, should be reported to caller
          break;
        default:
          break;
      }

      return response;
    }
  }

  public synchronized IContext getContext() {
    return this.context;
  }

  protected synchronized void setContext(IContext context) {
    this.context.endUse(this);
    this.context = context;
    this.context.startUse(this);
  }

  public synchronized void release() {
    if (this.danglingResponse == null) {

      if (this.closeOnNextUse) {
        // close early, save resources
        close();
      }

      this.releasePending = false;
      if (log.beInfo()) {
        log.infoT("release(320)", "" + this + " releasing requester to pool");
      }

      this.pool.reuse(this);
    }
    else {
      if (log.beInfo()) {
        log.infoT("release(327)", "" + this + " pending release, waiting for response close, pool is " + this.pool);
      }

      this.releasePending = true;
    }
  }

  public synchronized void responseClosed() {
    if (this.danglingResponse != null) {
      this.danglingResponse = null;

      if (this.closeOnNextUse) {
        try {
          if (log.beDebug()) {
            log.debugT("responseClosed(341)", "" + this + " closing connection before next use");
          }
          closeConnection();
        }
        catch (WcmException ex) {
          log.warningT("responseClosed(346)", "" + this + " closing connection" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(ex));
        }
        this.closeOnNextUse = false;
      }

      if (this.releasePending) {
        if (log.beDebug()) {
          log.debugT("responseClosed(353)", "" + this + " response closed, releasing requester");
        }
        release();
      }
      else {
        if (log.beDebug()) {
          log.debugT("responseClosed(359)", "" + this + " response closed, no release pending");
        }
      }
    }
  }

  public synchronized void close() {
    if (log.beInfo()) {
      log.infoT("close(367)", "" + this + " shutting down requester to " + this.base);
    }
    try {
      closeConnection();
    }
    catch (WcmException ex) {
      // ignore
      log.infoT("close(374)", "while closing connection" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(ex));
    }
  }

  public synchronized int getSoTimeout() {
    return this.soTimeoutMS;
  }

  public synchronized void setSoTimeout(int ms) {
    if (ms != 0) {
      if (ms < 0) {
        throw new IllegalArgumentException("timeout cannot be < 0");
      }

      this.connectTimeoutMS = ms;
    }

    this.soTimeoutMS = ms;
    if (this.socket != null) {
      try {
        this.socket.setSoTimeout(this.soTimeoutMS);
      }
      catch (SocketException e) {
        // ignore
        if (log.beDebug()) {
          log.debugT(e.getMessage());
        }        
      }
    }

    if (log.beDebug()) {
      log.debugT("setSoTimeout(402)", "timeout for socket operations set to: " + this.soTimeoutMS);
    }
  }

  public void discard() {
    if (log.beInfo()) {
      log.infoT("discard(408)", "discarding requester " + this);
    }
    close();
    this.discarded = true;
  }

  public void finalize() {
    this.context.endUse(this);
    if (!this.discarded) {
      this.pool.requesterDestroyed();
    }
  }

  public String toString() {
    return "SlimRequester[" + this.id + ", " + this.base + "]";
  }

  // --------- protected / private methods ------------------------------------------

  private IResponse doPerform(IRequest request, RequestStatus rstat, boolean repeatable)
    throws WcmException {
    try {
      this.releasePending = false;
      HttpStatus status = new HttpStatus();
      Headers headers = new Headers();

      long duration = 0L;
      if (log.beInfo()) {
        duration = System.currentTimeMillis();
      }
      boolean acceptResponse = false;
      sending :
      while (!acceptResponse) {
        // assure open connection to host
        //
        int state = 0;
        long start = System.currentTimeMillis();
        try {
          checkConnection(request, rstat);

          state = 1;
          HttpUrl url = send(request, rstat);
          ++rstat.tries;

          state = 2;
          checkEarlyClose(rstat);
          readResponse(url, status, headers, rstat, true);

          state = 3;
          acceptResponse = (rstat.tries >= MAX_TRIES
             || !shouldRetryRequest(status, headers, repeatable));

          if (!acceptResponse) {
            try {
              cleanUpConnection(request, status, headers, rstat);
            }
            catch (Exception e) {
              closeConnection();
            }
          }
        }
        catch (ExpectationFailedException ex) {
          status = ex.status;
          headers = ex.headers;
          acceptResponse = true;
        }
        catch (InterruptedIOException ex) {
          if (log.beInfo()) {
            log.infoT("doPerform(471)", "" + this + " operation timed out" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(ex));
          }
          throw ex;
        }
        catch (IOException ex) {
          switch (state) {
            case 0:
              // Connect failed
              //
              if (log.beInfo()) {
                log.infoT("doPerform(481)", "" + this + " while opening connection" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(ex));
              }
              closeConnection();
              ++rstat.connectFailures;
              long timeSpent = System.currentTimeMillis() - start;
              if (rstat.connectFailures >= CONNECT_FAILURES_MAX
                 || timeSpent > this.connectTimeoutMS) {
                throw ex;
              }

              if (rstat.connectFailures > 1) {
                // linear backoff from connecting, give remote server some time
                // to recover.
                //
                long waiting = (timeSpent) * rstat.connectFailures;
                if (waiting > this.connectTimeoutMS) {
                  waiting = this.connectTimeoutMS;
                }
                if (log.beInfo()) {
                  log.infoT("doPerform(500)", "" + this + " retrying connect in " + waiting + " ms");
                }

                try {
                  Thread.sleep(waiting);
                }
                catch (InterruptedException e) {
                  /*
                   *  ignore
                   */
                }
              }
              continue sending;
            case 1:
              // Send failed
              //
              if (log.beInfo()) {
                log.infoT("doPerform(517)", "" + this + "while sending request" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(ex));
              }
              closeConnection();
              ++rstat.IOErrors;
              if (rstat.IOErrors >= MAX_TRIES || !repeatable) {
                throw ex;
              }
              continue sending;
            case 2:
              // receive failed
              //
              log.infoT("doPerform(528)", "" + this + "while reading response" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(ex));
              acceptResponse = false;
              ++rstat.IOErrors;
              // On receive errors, we cannot retry methods which are not safe.
              if (rstat.IOErrors >= MAX_TRIES || !repeatable || !isSafeMethod(request.getMethod())) {
                this.closeOnNextUse = true;
                throw ex;
              }
              try {
                cleanUpConnection(request, status, headers, rstat);
              }
              catch (Exception e) {
                closeConnection();
              }
              break;
            default:
              // response received, but something else is wrong. Fail immediately
              throw ex;
          }
        }
      }

      // OK, we accept the response as it is.
      //
      IResponse response = createResponse(request, status, headers, rstat, false);

      // Give the context a chance to process any repsonse header
      // if might want to look at
      this.context.responseCredentials(this, response);

      this.danglingResponse = rstat.clientWillRead ? response : null;

      // Do we need to close this connection? If yes and we hand
      // out a response stream to the client, we have to delay the
      // actual closing of the connection until the client has closed
      // the stream we have given it.
      //
      if (rstat.willClose) {
        if (rstat.clientWillRead) {
          // client might still read from it, close later
          if (log.beInfo()) {
            log.infoT("doPerform(569)", "" + this + " remembering to close connection on next use");
          }
          this.closeOnNextUse = true;
        }
        else {
          closeConnection();
        }
      }

      if (log.beInfo()) {
        duration = System.currentTimeMillis() - duration;
        log.infoT("doPerform(580)", "" + this + " request took " + duration + " ms");
      }
      return response;
    }
    catch (IOException ex) {
      log.infoT("doPerform(585)", "" + this + " sending request" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(ex));
      throw new WcmException("sending request to: " + this.base
         + " request uri: " + request.getURI() + " " + ex.getMessage(), ex, false);
    }
  }

  /**
   * Determine the HTTP Version we will use in next request
   *
   * @param rstat status flags for this request
   * @return http version to use
   */
  private int getClientHTTPVersion(RequestStatus rstat) {
    // When we do not know yet what protocol version the
    // other side is talking, we announce
    // 1) for non-proxy connections: HTTP/1.1
    //    This is safe since HTTP/1.0 servers will not have
    //    to act on any headers and close the connection after
    //    sending the response. HTTP/1.1 servers will however
    //    (if willing) persist the connection
    // 2) for proxy connections: HTTP/1.0
    //    This is safe since HTTP/1.1 servers and proxies
    //    know how to treat a 1.0 client.
    //    Announcing HTTP/1.1 is dangerous when we have mixed
    //    protocol versions in the proxy-server connection chain.
    //    Proxies and servers might get confused about the
    //    peristent state of connections and hang.
    //
    switch (this.serverHTTPVersion) {
      case HTTP_11:
        return HTTP_11;
      case HTTP_10:
        return HTTP_10;
      default:
        return rstat.talkingToProxy ? HTTP_10 : HTTP_11;
    }
  }

  /**
   * Detect what protocol the other side is using
   *
   * @param status of response
   */
  private void detectProtocolUsed(HttpStatus status) {
    String protocol = status.getProtocol();
    if (HTTP11_ID.equalsIgnoreCase(protocol)) {
      this.serverHTTPVersion = HTTP_11;
    }
    else if (HTTP10_ID.equalsIgnoreCase(protocol)
       || protocol.startsWith(HTTP0x_ID)) {
      this.serverHTTPVersion = HTTP_10;
    }
    else {
      this.serverHTTPVersion = HTTP_11;
    }
  }

  /**
   * Set appropriate headers to achieve persistent connections
   *
   * @param sb where to append the headers to
   * @param rstat status flags for this request
   * @param seenHeaders add generated headers to this set
   */
  private void appendConnectionHeaders(StringBuffer sb, RequestStatus rstat, Set seenHeaders) {
    // We like to have persistent connections whenever possible. The mechanisms to
    // do so are very much different depending on the protocol version:
    // HTTP/1.1: connections are persistent unless a "Connection: close" header
    //           is sent.
    // HTTP/1.0: unless we plan to close the connection anyway, we send
    //           a "Connection: Keep-Alive". Otherwise the other end will
    //           always close the connection.
    //           In case we are talking to a proxy, the header is named
    //           "Proxy-Connection", but the behaviour is the same.
    // UNKNOWN:  We try to do the best of both worlds until we know the
    //           protocol version of the other end. This is only the
    //           case in the first request done by this object.
    //
    switch (this.serverHTTPVersion) {
      case HTTP_11:
        if (rstat.willClose) {
          sb.append("Connection: close").append(NEWLINE);
          seenHeaders.add("connection");
        }
        sb.append("TE: gzip, deflate").append(NEWLINE);
        break;
      case HTTP_10:
        if (!rstat.willClose) {
          if (rstat.talkingToProxy) {
            sb.append("Proxy-Connection: keep-alive").append(NEWLINE);
            seenHeaders.add("proxy-connection");
          }
          else {
            sb.append("Connection: keep-alive").append(NEWLINE);
            seenHeaders.add("connection");
          }
          sb.append("Keep-Alive: 30").append(NEWLINE);
        }
        break;
      case HTTP_UNKOWN:
        if (rstat.willClose) {
          sb.append("Connection: close").append(NEWLINE);
          seenHeaders.add("connection");
        }
        else {
          if (rstat.talkingToProxy) {
            sb.append("Proxy-Connection: keep-alive").append(NEWLINE);
            seenHeaders.add("proxy-connection");
          }
          else {
            sb.append("Connection: keep-alive").append(NEWLINE);
            seenHeaders.add("connection");
          }
          sb.append("Keep-Alive: 30").append(NEWLINE);
        }
        break;
    }
  }

  /**
   * Determine if we need to close the connection after this request
   *
   * @param request we are about to make.
   * @return if connection needs to be closed afterwards
   */
  private boolean needToCloseConnectionFor(IRequest request) {
    //
    switch (this.serverHTTPVersion) {
      case HTTP_11:
        // no need in HTTP/1.1 to ever close connections
        //
        return false;
      case HTTP_10:
        // in HTTP/1.0 we need to close connections when we
        // send data with indeterminate length
        //
        return (request.hasBody()
           && request.getBody() == null
           && request.getHeader("Content-Length") == null);
      case HTTP_UNKOWN:
        // We do not know yet which version the other side is talking
        //
        return (request.hasBody()
           && request.getBody() == null
           && request.getHeader("Content-Length") == null);
      default:
        return false;
    }
  }

  /**
   * Determine if we need to close the connection upon receiving the headers
   *
   * @param headers we received in response
   * @param rstat status flags for this request
   * @return if connection needs to be closed
   */
  private boolean needToCloseConnectionFor(Headers headers, RequestStatus rstat) {
    if (rstat.willClose) {
      return true;
    }
    else {
      String tmp = headers.get("Connection");
      switch (this.serverHTTPVersion) {
        case HTTP_11:
          if (tmp != null && tmp.toLowerCase().indexOf("close") >= 0) {
            if (log.beInfo()) {
              log.infoT("needToCloseConnectionFor(752)", "" + this + " remote server asks for closing connection");
            }
            return true;
          }
          break;
        case HTTP_10:
          if (rstat.talkingToProxy) {
            tmp = headers.get("Proxy-Connection");
          }
          if (tmp == null || tmp.toLowerCase().indexOf("keep-alive") < 0) {
            return true;
          }
          break;
        default:
          log.debugT("needToCloseConnectionFor(766)", "unknown http protocol versionm, should not happen");
          return true;
      }
      return false;
    }
  }

  private void cleanUpConnection(IRequest request, HttpStatus status, Headers headers, RequestStatus rstat)
    throws IOException, WcmException {
    // The response is not acceptable. Probably the server requests
    // authorization and we have to retry. But before we do that, we
    // have either to close the connection and open a new one, or,
    // we have to read pending data from the connection as if we
    // had a valid response.
    //
    IResponse intermediate = null;

    if (!rstat.willClose && rstat.IOErrors == 0) {
      intermediate = createResponse(request, status, headers, rstat, true);
    }

    // During construction of the response, we may find out that
    // we have to close the connection anyway (indeterminate length
    // of message body).
    //
    if (rstat.willClose || rstat.IOErrors > 0) {
      closeConnection();
    }
    else if (intermediate != null) {
      InputStream is = intermediate.getStream();
      if (is != null) {
        readEmpty(is);
        is.close();
      }
    }
  }

  private IUri getLocation(IResponse response, IRequest request) {
    String location = response.getHeader("Location");
    if (location != null) {
      try {
        IUriReference ref = UriFactory.parseUriReference(location);
        if (ref.isAbsolute()) {
          return ref.getUri();
        }
        else {
          HttpUrl base = getRequestUrl(request);
          return base.resolve(ref);
        }
      }
      catch (MalformedURLException ex) {
        log.infoT("getLocation(817)", "malformed uri in location header: " + location + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(ex));
      }
    }
    return null;
  }

  private HttpUrl getRequestUrl(IRequest request) {
    IUriReference ref = request.getReference();
    if (ref != null) {
      return (HttpUrl)this.base.resolve(ref);
    }
    else {
      return (HttpUrl)this.base.appendPath(request.getURI());
    }
  }

  private HttpUrl appendRequestLine(StringBuffer sb, IRequest request, RequestStatus rstat) {
    // If the request carries an uri reference, use that. Otherwise
    // fall back to using the old URI string.
    // When we are not talking to proxies, we might no need to build
    // the complete Url if the uri reference starts with a slash.
    //
    sb.setLength(0);
    HttpUrl url = getRequestUrl(request);
    if (rstat.talkingToProxy) {
      sb.append(url);
    }
    else {
      sb.append(url.getPath());
      if (url.getQuery() != null) {
        sb.append('?').append(url.getQuery());
      }
    }

    String requestUri = sb.toString();
    sb.setLength(0);
    sb.append(request.getMethod()).append(' ');
    sb.append(requestUri).append(' ');
    switch (getClientHTTPVersion(rstat)) {
      case HTTP_11:
        sb.append(SlimRequester.HTTP11_ID);
        break;
      case HTTP_10:
        sb.append(SlimRequester.HTTP10_ID);
        break;
      default:
        sb.append(SlimRequester.HTTP11_ID);
        break;
    }
    sb.append(SlimRequester.NEWLINE);

    return url;
  }

  /**
   * Send the request on the output stream.
   *
   * @return if this request can be send again
   */
  private HttpUrl send(IRequest request, RequestStatus rstat)
    throws IOException, WcmException {
    StringBuffer sb = new StringBuffer();

    // Send the request line
    //
    HttpUrl requestURI = appendRequestLine(sb, request, rstat);

    // Send general headers
    //
    long contentLength = -1;
    Set seenHeaders = new HashSet();

    sb.append("Host: ").append(this.base.getAuthority()).append(NEWLINE);
    seenHeaders.add("host");

    if (this.context.applyCredentials(this, requestURI.getPath(), request)) {
      if (log.beInfo()) {
        StringBuffer sb2 = new StringBuffer(128);
        sb2.append(this).append(" using authentication: ");
        UserInfo ui = this.context.getUserInfo();
        if (!UserInfo.ANONYMOUS.equals(ui)) {
          sb2.append(" server(").append(ui).append(")");
        }
        if (rstat.talkingToProxy) {
          ui = this.context.getProxyUserInfo();
          if (!UserInfo.ANONYMOUS.equals(ui)) {
            sb2.append(" proxy(").append(ui).append(")");
          }
        }
        log.infoT("send(911)", sb2.toString());
      }
    }

    appendConnectionHeaders(sb, rstat, seenHeaders);

    String cookies = this.context.getCookieHeaderValue(requestURI);
    if (cookies != null) {
      sb.append("Cookie: ").append(cookies).append(NEWLINE);
    }

    // Send request headers. Do not generate headers, which we
    // have already sent. Also look for content-length header,
    // in case the client supplied one.
    //
    contentLength = applyHeaders(request, seenHeaders, sb);

    if (USE_USER_AGENT && !seenHeaders.contains("user-agent")) {
      sb.append("User-Agent: ").append(USER_AGENT).append(NEWLINE);
    }

    // Determine, if we have to send additional headers.
    // This is the case when we have a request body, but
    // a) have not send a Content-Length header and can
    //    find out the length of the body. Then we generate
    //    a Content-Length header.
    // b) have not sent a Content-Length header and cannot
    //    determine the length. Then we generate a Transfer-Encoding
    //    header and have to chunk the body out.
    // c) we have no request body and have not seen a Content-Length
    //    header. In this case we have to set the content length to 0.
    //
    byte[] body = null;
    boolean chunked = false;
    File tmpFile = null;
    InputStream bodyStream = request.getBodyStream();
    try {
      if (request.hasBody()) {
        String tmp = request.getBody();
        if (tmp != null) {
          String encoding = getEncoding(request);
          if (encoding != null) {
            body = tmp.getBytes(encoding);
          }
          else {
            addEncoding(request, DEFAULT_ENCODING);
            try {
              body = tmp.getBytes(DEFAULT_ENCODING);
            }
            catch (UnsupportedEncodingException e) {
              log.warningT("send(961)", "encoding not supported by runtime, continuing with default encoding" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
              body = tmp.getBytes();
            }
          }
          contentLength = body.length;
          sb.append("Content-Length: ").append(contentLength).append(NEWLINE);
        }
        else {
          if (contentLength >= 0) {
            sb.append("Content-Length: ").append(contentLength).append(NEWLINE);
          }
          else {
            switch (this.serverHTTPVersion) {
              case HTTP_11:
                sb.append("Transfer-Encoding: chunked").append(NEWLINE);
                chunked = true;
                break;
              default:
                // No HTTP/1.1 server, no content-length known, but we have
                // an input stream. We have to close the connection at the
                // end of the request.
                // 2003-02-03: Alas, our squid proxy requires a Content-Length
                // header as well. Sigh.
                //
                if (rstat.isSSL || rstat.talkingToProxy) {
                  // Well, well. Seems like our SSL implementation does not
                  // like Socket.shutdownOutput() so much. Try it with
                  // chunked encoding.
                  tmpFile = File.createTempFile("slim", "data");
                  FileOutputStream out = new FileOutputStream(tmpFile);
                  byte[] buffer = new byte[COPY_BUFFER_SIZE];
                  copy(out, bodyStream, buffer);
                  out.close();
                  bodyStream = new FileInputStream(tmpFile);
                  contentLength = tmpFile.length();
                  sb.append("Content-Length: ").append(contentLength).append(NEWLINE);
                }
                else {
                  if (log.beInfo()) {
                    log.infoT("send(1000)", "" + this + "send body to HTTP/1.0 without content-length");
                  }
                }
                break;
            }
          }
        }
      }
      else {
        // no body
        contentLength = 0;
        if (!"GET".equals(request.getMethod())) {
          sb.append("Content-Length: 0").append(NEWLINE);
        }
      }

      rstat.definedLength = (contentLength >= 0);

      // Headers are separated from the body by an additional newline
      //
      sb.append(NEWLINE);
      String header = sb.toString();

      // in theory, the encoding shouldn't matter as the header should be
      // strictly ASCII. In practice, we need this to able to pass "badly"
      // encoded ISO through (Web folder client tests). Maybe this should
      // be configurable.
      this.output.write(header.getBytes("ISO-8859-1"));

      if (log.beInfo()) {
        if (log.beDebug()) {
          log.debugT("send(1031)", "" + this + " Request: \n" + header);
        }
        else {
          log.infoT("send(1034)", "" + this + " Request: " + request.getMethod() + " " + requestURI);
        }
      }

      // Headers written, now let's take care of the body
      //
      if (request.hasBody()) {
        if (seenHeaders.contains("expect")) {
          String value = request.getHeader("Expect");
          if (value != null && (value.indexOf("100-continue") >= 0)) {
            this.output.flush();

            HttpStatus status = new HttpStatus();
            Headers headers = new Headers();
            this.readResponse(requestURI, status, headers, rstat, false);
            if (status.getCode() >= 200) {
              throw new ExpectationFailedException(requestURI, status, headers);
            }
          }
        }
        
        if (!chunked && body != null) {
          if (log.beDebug()) {
            log.debugT("send(1043)", "" + this + "request body: " + request.getBody());
          }
          this.output.write(body);
        }
        else {
          long written = 0L;
          try {
            byte[] buffer = new byte[COPY_BUFFER_SIZE];
            if (!chunked) {
              // copy through direct
              written = copy(this.output, bodyStream, buffer);
            }
            else {
              // copy through chunked
              written = copyChunked(this.output, bodyStream, buffer);
            }
          }
          catch (IOException ex) {
            // we might have an error reading the input. Better make sure
            // that the connection is closed so that the remote server can
            // detect that we stopped writing.
            try {
              closeConnection();
            }
            catch (WcmException ex2) {
              if (log.beDebug()) {
                log.debugT(ex2.getMessage());
              }
            }
            throw ex;
          }

          if (contentLength >= 0 && contentLength != written) {
            // Uuups, we did not write what we promised to write. Close
            // the connection so that the remote server has the chance to
            // detect the EOF. If we would keep the connection open, the
            // server might hang on the socket timeout
            closeConnection();
            throw new IOException("set ContentLength and stream length differ. Expected "
               + contentLength + ", but got " + written + " bytes from stream");
          }
        }
      }
      if (this.output != null) {
        this.output.flush();
      }
    }
    finally {
      if (tmpFile != null) {
        try {
          bodyStream.close();
        }
        catch (IOException e) {
          // ignore
          if (log.beDebug()) {
            log.debugT(e.getMessage());
          }          
        }
        tmpFile.delete();
      }
    }

    return requestURI;
  }

  /**
   * Plain copy of stream to stream
   */
  private long copy(OutputStream output, InputStream input, byte[] buffer)
    throws IOException {
    long written = 0L;
    int len;
    while ((len = input.read(buffer)) != -1 && this.output != null) {
      output.write(buffer, 0, len);
      written += len;
    }
    return written;
  }

  /**
   * Copy from input to output stream with use of HTTP chunked transfer encoding
   */
  private long copyChunked(OutputStream output, InputStream input, byte[] buffer)
    throws IOException {
    long written = 0L;
    int len;
    while ((len = input.read(buffer)) != -1) {
      if (log.beDebug()) {
        log.debugT("copyChunked(1140)", "" + this + "writing chunk length: " + len);
      }
      output.write(Integer.toHexString(len).getBytes());
      output.write(NEWLINE_BYTES);
      output.write(buffer, 0, len);
      output.write(NEWLINE_BYTES);
      written += len;
    }
    // end chunk
    if (log.beDebug()) {
      log.debugT("copyChunked(1150)", "" + this + "writing end chunk");
    }
    output.write(ZERO_BYTES);
    output.write(NEWLINE_BYTES);
    return written;
  }

  private void readHeaders(Headers headers, StringBuffer sb)
    throws IOException {
    headers.clear();
    String name = null;
    String value = null;
    while (this.headerReader.hasNext()) {
      String line = this.headerReader.next();
      int index = line.indexOf(':');
      if (index >= 0) {
        name = line.substring(0, index).trim();
        value = line.substring(index + 1).trim();
        headers.add(name, value);
        if (sb != null) {
          sb.append(line).append('\n');
        }
      }
      else if (name != null &&
        (line.charAt(0) == ' ' || line.charAt(0) == '\t')) {
        value += line.trim();
        headers.add(name, value);
      }
      else {
        if (sb != null) {
          sb.append("illegal header: ").append(line).append('\n');
        }
      }
    }
  }

  private void readResponse(HttpUrl url, HttpStatus status, Headers headers, RequestStatus rstat,
      boolean skip100s) throws IOException {

    StringBuffer sb = null;
    String line;
    do {
      line = this.headerReader.first(this.input);

      // OK, we got a line back from the server and try
      // to parse the HTTP status. If we get an error
      // on that, we rethrow an IOException so that the
      // retry mechanism will be triggered. It may be
      // that we have garbage on the connection...
      // (ISSUE: some server answer HEAD with sending
      // a request body (illegal) and our persistent
      // connection will be screwed up)
      //
      try {
        status.parse(line);
      }
      catch (WcmException e) {
        throw new IOException(e.getMessage());
      }

      if (log.beDebug()) {
        sb = new StringBuffer();
        sb.append(this).append(" response: \n");
        sb.append(line).append('\n');
      }
      else if (log.beInfo()) {
        log.infoT("readResponse(1216)", "" + this + " Response: " + line);
      }

      detectProtocolUsed(status);

      // read headers
      //
      readHeaders(headers, sb);
    } while (skip100s && status.getCode() >= 100 && status.getCode() < 200);

    if (log.beDebug()) {
      log.debugT("readResponse(1227)", sb.toString());
    }

    rstat.willClose = needToCloseConnectionFor(headers, rstat);
    extractCookies(url, headers, rstat);
  }

  private IResponse createResponse(IRequest request,
    HttpStatus status,
    Headers headers,
    RequestStatus rstat,
    boolean ignoreClient)
    throws WcmException, IOException {

    SlimResponse response = new SlimResponse(status, headers);

    // Do we know that there is no body?
    //
    String type = response.getContentType();
    String tencoding = headers.get("Transfer-Encoding");
    long resultLen = response.getContentLength();

    boolean isHEADRequest = request.getMethod().equalsIgnoreCase("HEAD");
    int statusCode = status.getCode();
    if (isHEADRequest || statusCode == 204 || statusCode == 304) {
      // some methods are more special than others.
      // RFC 2616 says that all responses to HEAD and all 204 repsonse MUST NOT
      // have a response body!
      //
      resultLen = 0;
      if (isHEADRequest && !rstat.talkingToProxy) {
        switch (status.getCode()) {
          case 200:
            // we trust a 200 on HEAD to not carry any content
            break;
          case 401:
            // we have trust these answers not to carry any content either. Thing
            // is that for beasts like NTLM authentication, we need the connection
            // to remain open
            break;
            
          default:
            // In general, we do not really trust servers which send error codes on HEAD
            // to keep the contract and better close the connection before next use.
            // But we do trust proxy system...
            rstat.willClose = true;
            break;
        }
      }
    }
    else if (resultLen == -1L) {
      if (type == null) {
        if (tencoding != null && tencoding.indexOf("chunked") >= 0) {
          // We have a body, but no Content-Type, however if the status code
          // is 207 which indicates a text/xml multistatus response.
          // If we have other status, we nevertheless simulate a type, so
          // that the response body is read
          //
          type = (status.getCode() == 207) ? "text/xml" : "application/octet-stream";
        }
        else {
          resultLen = 0;
        }
      }
      else {
        // No Content-Length, but Content-Type
        // This can have content only, if
        // 1. Transfer-Encoding is chunked
        // 2. Connection header is close
        // 3. HTTP/1.0 and Connection header is not Keep-Alive
        //
        if (tencoding == null) {
          String connection = headers.get("Connection");
          switch (this.serverHTTPVersion) {
            case HTTP_11:
              if (connection == null) {
                // Wow, a HTTP/1.1 server with content-type without length, without transfer-encoding,
                // and it does not announce that it will close the connection. This is no compliant server
                // IIS/4.0 is known to behave like this.
                // We definitely want to close our connection after having seen this response!
                //
                rstat.willClose = true;
              }
              else if (connection.toLowerCase().indexOf("close") < 0) {
                resultLen = 0;
              }
              break;
            default:
              // Leave this test for http/1.0 for the moment since we have
              // servers on SAP side which send such responses
              //
              //else if (!this.m_ishttp11 && connection != null && connection.indexOf("Keep-Alive") >= 0) {
              //  resultLen = 0;
              //}
              break;
          }
        }
      }
    }

    // read possible body
    //
    rstat.clientWillRead = false;
    if (resultLen == 0 && tencoding == null) {
      // We know that there is no body, but if the client
      // expects a response stream, we give him an empty one
      // to suck on
      // 2003-02-19 stefan.eissing@greenbytes.de: it is legal to send
      // a Content-Length: 0 with transfer encoding. In this case there
      // will be a body nevertheless. Transfer-Encoding has precedence over
      // any Content-Length information. See RFC 2616, ch. 4.4
      //
      if (!isHEADRequest && status.getCode() != 204
         && !ignoreClient && request.expectsResponseStream()) {
        response.setStream(new ClientInputStream());
      }
    }
    else {
      if (type != null) {

        InputStream input = this.input;
        if (tencoding != null) {
          // Apply transfer encoding headers
          List encodings = new ArrayList(5);
          for (StringTokenizer tokens = new StringTokenizer(tencoding, ", "); tokens.hasMoreTokens(); ) {
            String tenc = tokens.nextToken();
            if (tenc.indexOf(';') > 0) {
              // cut off any parameters
              tenc = tenc.substring(0, tenc.indexOf(';'));
            }
            encodings.add(tenc);
          }
          // We have to handle encodings in reverse order for decoding
          //
          for (int i = encodings.size() - 1; i >= 0; --i) {
            String tenc = (String)encodings.get(i);
            if ("chunked".equals(tenc)) {
              // In the case of chunked encoding, any Content-Length
              // is not telling the message body length (see RFC 2616, Ch. 4.4)
              //
              input = new ChunkedInputStream(input);
              resultLen = -1;
            }
            else if ("gzip".equals(tenc)) {
              input = new java.util.zip.GZIPInputStream(input);
              resultLen = -1;
            }
            else if ("deflate".equals(tenc)) {
              input = new java.util.zip.InflaterInputStream(input);
              resultLen = -1;
            }
            else {
              log.warningT("createResponse(1370)", "unsupported transfer-encoding used by server: " + tenc);
            }
          }
        }

        if (!ignoreClient && request.expectsResponseStream()) {
          // Prevent closing of connection at the end of this method
          //
          rstat.clientWillRead = true;
          response.setStream(new ClientInputStream(input, resultLen, this, rstat.isSSL));
        }
        else if (!ignoreClient
           && (request.expectsResponseDocument()
           || response.getStatus() == 207
           || response.getStatus() >= 300)
           && (type.indexOf("text/xml") != -1) || type.indexOf("application/xml") != -1) {
          InputStream is = null;
          is = new ClientInputStream(input, resultLen, null, rstat.isSSL);

          try {
            response.setDocument(DomBuilder.parse(is, false));
            if (log.beDebug()) {
              log.debugT("createResponse(1393)", "" + this + " response document: " + SimpleSerializer.toString(response.getDocument(), true));
            }
          }
          catch (org.xml.sax.SAXException ex) {
            log.debugT("createResponse(1397)", "" + this + " error building document" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(ex));
            throw new WcmException("cannot parse response document", ex, false);
          }
          finally {
            is.close();
          }
        }
        else {
          // Our caller does not want to see the message body,
          // but there may be bytes waiting on the socket for us to read
          //
          if (input != null) {
            if (resultLen > 0) {
              if (log.beInfo()) {
                log.infoT("createResponse(1411)", "" + this + " skipping " + resultLen + " bytes on input");
              }
              if (!log.beDebug()) {
                input.skip(resultLen);
              }
              else {
                dumpAndSkip(input, resultLen);
              }
            }
            else {
              // undetermined result len (== -1)
              //
              if (this.input != input) {// chunking
                if (log.beDebug()) {
//                  log.debugT(""+this+" reading chunked stream empty");
                  dumpAndSkip(input, -1);
                }
                input.close();
              }
              else {
                // No chunking and undetermined length of response:
                // we need to close the connection
                //
                if (log.beInfo()) {
                  log.infoT("createResponse(1435)", "" + this + " input pending, don't know how much, closing connection");
                }
                rstat.willClose = true;
              }
            }
          }
        }
      }
    }

    return response;
  }

  private long lastCheck;
  
  /**
   * Assert that the connection to the remote system is open
   *
   * @param request we are about to make
   * @param rstat status flags for request
   * @throws WcmException when connection could not be established
   */
  private void checkConnection(IRequest request, RequestStatus rstat)
    throws IOException, WcmException {
    // If there is still an open connection which we need to close, do so.
    // If we have a dangling response with a stream, read it empty.
    // Determine host:port and protocol we need to talk to/with.
    // For an open conneciton try to get the streams again. This will fail
    // when the socket is no longer OK. Ignore the failures and open
    // a new socket.
    // In case of https through a proxy, prepare to upgrade the connection
    // _if_ we have to establish a new TCP connection.
    //
    long now = System.currentTimeMillis();
    if (this.closeOnNextUse) {
      closeConnection();
    }
    else if (this.danglingResponse != null) {
      InputStream is = this.danglingResponse.getStream();
      if (is != null) {
        try {
          readEmpty(is);
        }
        catch (IOException ex) {
          closeConnection();
        }
      }
      this.danglingResponse = null;
    }
    else {
      if (this.lastCheck != 0 && (now - this.lastCheck) > this.pool.getTimeout()) {
        closeConnection();
      }
    }
    
    String protocol = this.base.getScheme();
    String host = this.base.getHost();
    int port = this.base.getPort();
    boolean mayNeedUpgrade = false;
    rstat.needsTLSUpgrade = false;
    rstat.talkingToProxy = (rstat.proxy != null);

    rstat.willClose = needToCloseConnectionFor(request);

    if (rstat.proxy != null) {
      host = rstat.proxy.getHost();
      port = rstat.proxy.getPort();
      if (log.beDebug()) {
        log.debugT("checkConnection(1496)", "" + this + " using proxy " + host + ":" + port);
      }

      if (protocol.equals("https")) {
        mayNeedUpgrade = true;
        protocol = "http";
      }
    }

    try {
      rstat.isSSL = "https".equals(protocol);
      if (this.socket == null) {
        if (log.beInfo()) {
          log.infoT("checkConnection(1509)", "" + this + " new " + protocol + " connection to " + host + ":" + port);
        }
        this.socket = this.pool.createSocket(protocol, host, port);
        if (this.soTimeoutMS > 0) {
          this.socket.setSoTimeout(this.soTimeoutMS);
        }
        rstat.needsTLSUpgrade = mayNeedUpgrade;

        try {
          // Try to get the streams anew. This will let the
          // socket check if it still open.
          //
          initInputOutputStreams();
        }
        catch (IOException ex) {
          // Hmm, socket was no longer properly open,
          // let's try again
          //
          if (log.beInfo()) {
            log.infoT("checkConnection(1528)", "" + this + " reopening the socket and streams" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(ex));
          }

          if (this.socket != null) {
            this.socket.close();
          }

          this.socket = this.pool.createSocket(protocol, host, port);
          if (this.soTimeoutMS > 0) {
            this.socket.setSoTimeout(this.soTimeoutMS);
          }

          initInputOutputStreams();
          rstat.needsTLSUpgrade = mayNeedUpgrade;
        }

        this.isFreshConnection = true;
        this.context.openedConnection(this);
      }
      else {
        this.isFreshConnection = false;
      }
    }
    catch (UnknownHostException ex) {
      String msg = "" + this + " unknown host " + host;
      log.debugT("checkConnection(1553)", msg + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(ex));
      throw new com.sapportals.wcm.util.http.ConnectException(host, "unknown host: "+ex.getMessage(), 30, ex);
    }
    catch (java.net.ConnectException ex) {
      String msg = "" + this + " could not connect to " + host;
      log.debugT("checkConnection(1553)", msg + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(ex));
      // Rethrow since we get this exception when the remote server is busy. 
      // Rethrowing will trigger the retry mechanism higher up the call stack
      throw ex;
      //throw new com.sapportals.wcm.util.http.ConnectException(host, "unable to connect: "+ex.getMessage(), 10, ex);
    }
    catch (NoRouteToHostException ex) {
      String msg = "" + this + " no route to host " + host;
      log.debugT("checkConnection(1558)", msg + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(ex));
      throw new com.sapportals.wcm.util.http.ConnectException(host, "no route to host: "+ex.getMessage(), 60, ex);
    }

    if (rstat.needsTLSUpgrade) {
      if (rstat.talkingToProxy) {
        upgradeProxyConnection(rstat);
      }
      else {
        upgradeConnection(rstat);
      }
    }
    
    this.lastCheck = now;
  }

  private void initInputOutputStreams()
    throws IOException {
    this.input = this.socket.getInputStream();
    this.socketOutput = this.socket.getOutputStream();
    this.output = new BufferedOutputStream(this.socketOutput, OUTPUT_BUFFER_SIZE);
  }

  private void closeConnection()
    throws WcmException {
    Exception iex = null;
    this.closeOnNextUse = false;

    if (this.input != null) {
      try {
        this.input.close();
      }
      catch (IOException ex) {
        iex = ex;
      }
      this.input = null;
    }
    if (this.output != null) {
      try {
        this.output.close();
      }
      catch (IOException ex) {
        iex = ex;
      }
      this.output = null;
    }
    if (this.socketOutput != null) {
      try {
        this.socketOutput.close();
      }
      catch (IOException ex) {
        iex = ex;
      }
      this.socketOutput = null;
    }
    if (this.socket != null) {
      try {
        this.socket.close();
      }
      catch (IOException ex) {
        iex = ex;
      }
      this.socket = null;
      if (log.beInfo()) {
        log.infoT("closeConnection(1620)", "" + this + " closed connection to " + this.base);
      }
    }

    this.isFreshConnection = true;
    if (iex != null) {
      log.warningT("closeConnection(1626)", "" + this + " error closing connection" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(iex));
      throw new WcmException(iex, false);
    }
  }

  private void checkEarlyClose(RequestStatus rstat)
    throws IOException {
    // In http 1.0 requests, we have no chunking and thus
    // need to close the output on requests without content length.
    // Only then can the client know, e.g. on a PUT request,
    // that the end of input has been reached
    //
    if (rstat.willClose && !rstat.definedLength) {
      if (log.beInfo()) {
        log.infoT("checkEarlyClose(1640)", "" + this + " shutting down output to remote server (non HTTP/1.1)");
      }
      this.socketOutput = null;
      this.output = null;
      this.socket.shutdownOutput();
    }
  }

  private boolean shouldRetryRequest(HttpStatus status, Headers headers, boolean repeatable)
    throws WcmException {
    boolean gotCredentials = false;
    if (repeatable) {
      switch (status.getCode()) {
        case 401:
          gotCredentials = this.context.setupCredentials(this, headers);
          if (log.beInfo()) {
            log.infoT("shouldRetryRequest(1656)", "" + this + (gotCredentials ?
              " retrying request with credentials for " + this.context.getUserInfo()
               : " no credentials, accepting unauthorized response"));
          }
          break;
        case 407:
          gotCredentials = this.context.setupProxyCredentials(this, headers);
          if (log.beInfo()) {
            log.infoT("shouldRetryRequest(1664)", "" + this + (gotCredentials ?
              " retrying request with proxy credentials for " + this.context.getProxyUserInfo()
               : " no proxy credentials, accepting unauthorized response"));
          }
          break;
        default:
          break;
      }
    }

    // We should retry the request, if it is repeatable and we have
    // new credentials to try out.
    //
    return repeatable && gotCredentials;
  }

  /**
   * Upgrade the existing proxy connection to a direct TLS connection
   *
   * @param rstat status flags for request
   */
  private void upgradeProxyConnection(RequestStatus rstat)
    throws IOException, WcmException {

    HttpStatus status = new HttpStatus();

    boolean done = false;
    for (int i = 0; i < 4 && !done; ++i) {
      HttpRequest request = new HttpRequest("/");
      request.setMethod("CONNECT");

      StringBuffer sb = new StringBuffer();
      sb.append("CONNECT ").append(this.base.getHost()).append(":").append(this.base.getPort());
      sb.append(" ").append(HTTP11_ID).append(NEWLINE).append("Host: ");
      sb.append(this.base.getAuthority()).append(NEWLINE);

      if (this.context.applyCredentials(this, "/", request)) {
        applyHeaders(request, null, sb);
      }

      sb.append(NEWLINE);

      if (log.beDebug()) {
        log.debugT("upgradeProxyConnection(1709)", "" + this + " upgradeProxyConnection to SSL:\n" + sb.toString());
      }

      this.output.write(sb.toString().getBytes(DEFAULT_ENCODING));
      this.output.flush();

      String line = this.headerReader.first(this.input);
      if (log.beInfo()) {
        log.infoT("upgradeProxyConnection(1717)", "" + this + " connect response: " + line);
      }

      status.parse(line);
      int code = status.getCode();

      Headers headers = new Headers();
      StringBuffer sb2 = null;
      if (log.beDebug()) {
        sb2 = new StringBuffer(128);
        sb2.append(line).append('\n');
      }

      detectProtocolUsed(status);
      
      readHeaders(headers, sb2);

      rstat.willClose = needToCloseConnectionFor(headers, rstat);

      // IResponse response =
      createResponse(request, status, headers, rstat, false);

      if (log.beDebug()) {
        log.debugT("upgradeProxyConnection(1738)", "" + this + " CONNECT response: " + sb2.toString());
      }

      if (code >= 200 && code < 300) {
        done = true;
      }
      else {
        if (!shouldRetryRequest(status, headers, true)) {
          throw new WcmException("cannot establish SSL through proxy: " + code, false);
        }

        if (rstat.willClose) {
          // Damn, proxy wants to close the connection. We have to trigger our
          // retry in perform() by throwing a Exception
          throw new IOException("CONNECT must retry on new connection");
        }
      }
    }

    rstat.talkingToProxy = false;
    upgradeConnection(rstat);
  }

  private void upgradeConnection(RequestStatus rstat)
    throws IOException {
    this.socket = this.pool.upgradeTLS(this.socket, this.base);
    initInputOutputStreams();
    rstat.needsTLSUpgrade = false;
    rstat.isSSL = true;
  }

  private final static int HEADER_SPLIT_LENGTH = 80;

  private long applyHeaders(IRequest request, Set seenHeaders, StringBuffer sb) {
    // Send request headers. Do not generate headers, which we
    // have already sent. Also look for content-length header,
    // in case the client supplied one.
    //
    long contentLength = -1L;
    Iterator iter = request.getHeaderNames();
    while (iter.hasNext()) {
      String key = (String)iter.next();
      String lower = key.toLowerCase();
      if (seenHeaders != null) {
        if (seenHeaders.contains(lower)) {
          continue;
        }
        seenHeaders.add(lower);
      }
      String value = request.getHeader(key);
      if (lower.equals("content-length")) {
        try {
          contentLength = Long.parseLong(value);
        }
        catch (NumberFormatException ex) {
          log.warningT("applyHeaders(1793)", "could not parse client supplied content-length: " + value);
        }
      }
      else {
        if (value != null) {
          // Split "too long" headers into multiple lines if a separator is present
          //
          value = value.trim();
          if (value.length() > HEADER_SPLIT_LENGTH && request.wasAddedHeader(key)) {
            // header  is quite long and constructed with add methods. We can
            // safely split the comma separated value among multiple header lines
            //
            for (int index = value.indexOf(", "); index > 0
               && (index < value.length() - 2)
               && value.length() > HEADER_SPLIT_LENGTH; ) {
              String value1 = value.substring(0, index);
              value = value.substring(index + 2);
              sb.append(key).append(": ").append(value1).append(NEWLINE);
            }
          }
        }
        sb.append(key).append(": ").append(value).append(NEWLINE);
      }
    }

    return contentLength;
  }

  private void extractCookies(HttpUrl url, Headers headers, RequestStatus rstat) {
    String headerName = "Set-Cookie2";
    String cookies = headers.get(headerName);
    if (cookies == null) {
      headerName = "Set-Cookie";
      cookies = headers.get(headerName);
      if (cookies == null) {
        return;
      }
    }

    SlimCookie.set(url, this.context, headerName, cookies);
  }

  private void readEmpty(InputStream is)
    throws IOException {
    byte[] buffer = new byte[16 * 1024];
    while (is.read(buffer, 0, buffer.length) != -1) {
      //
    }
  }

  private void dumpAndSkip(InputStream is, long offset)
    throws IOException {
    byte[] buffer = new byte[16 * 1024];

    if (offset < 0) {
      int read = 0;
      while ((read = is.read(buffer)) >= 0) {
        log.debugT("dumpAndSkip(1854)", "body: " + new String(buffer, 0, read));
      }
    }
    else {
      long toread = offset;
      do {
        int bytes = toread > buffer.length ? buffer.length : (int)toread;
        int r = is.read(buffer, 0, bytes);
        toread = (r < 0) ? 0 : toread - r;
        log.debugT("dumpAndSkip(1863)", "body: " + new String(buffer, 0, r));
      } while (toread > 0);
    }
  }

  private String getEncoding(IRequest request) {
    String ct = request.getHeader("content-type");
    if (ct != null) {
      int index = ct.indexOf("charset=");
      if (index >= 0) {
        ct = ct.substring(index + "charset=".length());
        int len = ct.length();
        int i = 0;
        for (i = 0; i < len; ++i) {
          char c = ct.charAt(i);
          if (c == ' ' || c == ';') {
            break;
          }
        }

        if (i < len) {
          return ct.substring(0, i).toUpperCase();
        }
        return ct.toUpperCase();
      }
    }
    return null;
  }

  private void addEncoding(IRequest request, String encoding) {
    String ct = request.getHeader("content-type");
    if (ct != null) {
      ct = ct + "; charset=" + encoding;
      request.setHeader("Content-Type", ct);
    }
  }

  private static class HeaderReader {

    private final ByteArrayOutputStream bos;
    private InputStream is;
    private String last;

    public HeaderReader() {
      bos = new ByteArrayOutputStream(256);
    }

    public String first(InputStream is)
      throws IOException {
      this.is = is;
      while ((last = readLine(true)) == null) {
        // read first, non-emtpy, required line
      }
      return last;
    }

    public boolean hasNext()
      throws IOException {
      // headers end in first empty line or eof
      last = readLine(false);
      return last != null;
    }

    public String next() {
      return last;
    }

    /**
     * Read a line from the input stream, assuming that bytes are utf-8 encoded.
     */
    private String readLine(boolean required) throws IOException {
      this.bos.reset();
      try {
        while (true) {
          int n = this.is.read();
          switch (n) {
            case '\r':
              break;
            case '\n':
              if (this.bos.size() > 0) {
                return this.bos.toString(DEFAULT_ENCODING);
              }
              return null;
            case -1:
              // server closed the connection
              if (this.bos.size() > 0) {
                return this.bos.toString(DEFAULT_ENCODING);
              }
              else if (required) {
                throw new IOException("unexpected EOF on reading line");
              }
              return null;
            default:
              this.bos.write(n);
              break;
          }
        }
      }
      catch (IOException ex) {
        if (log.beDebug()) {
          log.debugT("readLine(1970)", "reading response, read \"" + this.bos.toString(DEFAULT_ENCODING) + "\"" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(ex));
        }
        throw ex;
      }
    }
  }

  private static class ExpectationFailedException extends WcmException {
    
    public final HttpUrl url;
    public final HttpStatus status;
    public final Headers headers;
    
    public ExpectationFailedException(HttpUrl url, HttpStatus status, Headers headers) {
      this.url = url;
      this.status = status;
      this.headers = headers;
    }
  }
}

