/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.http.slim;

import com.sapportals.wcm.util.http.DateFormat;
import com.sapportals.wcm.util.http.IContext;
import com.sapportals.wcm.util.http.ICookie;
import com.sapportals.wcm.util.http.IExtendedCookie;
import com.sapportals.wcm.util.uri.HttpUrl;
import com.sapportals.wcm.util.uri.IHierarchicalUri;
import com.sapportals.wcm.util.uri.IUri;
import com.sapportals.wcm.util.uri.IUriReference;
import com.sapportals.wcm.util.uri.UriFactory;

import java.net.MalformedURLException;
import java.text.ParseException;
import java.util.*;

/**
 * Implementation of ICookie for the slim package. <p>
 *
 * Follows RFC 2965 and <a href="http://developer.netscape.com/docs/manuals/js/client/jsref/cookies.htm">
 * Netscape's cookie specification</a> . <p>
 *
 * Copyright (c) SAP AG 2001-2003
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Id: SlimCookie.java,v 1.5 2004/01/28 11:37:00 sei Exp $
 */
final class SlimCookie implements ICookie, IExtendedCookie {

  private final static com.sap.tc.logging.Location log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.http.slim.SlimCookie.class);

  private final static Set ATTRIBUTES;

  static {
    ATTRIBUTES = new HashSet(17);
    ATTRIBUTES.add("comment");
    ATTRIBUTES.add("commentURL");
    ATTRIBUTES.add("discard");
    ATTRIBUTES.add("domain");
    ATTRIBUTES.add("max-age");
    ATTRIBUTES.add("path");
    ATTRIBUTES.add("port");
    ATTRIBUTES.add("secure");
    ATTRIBUTES.add("version");
    ATTRIBUTES.add("expires");
  }

  public static void set(HttpUrl url, IContext context, String headerName, String header) {
    if (log.beDebug()) {
      log.debugT("set(52)", "parsing set-cookie header: " + header + " in context: " + context);
    }

    try {
      Input input = new Input(header);
      StringBuffer sb = new StringBuffer();
      long now = System.currentTimeMillis();
      while (input.attr(sb)) {
        String name = sb.toString();
        if (input.peek() == '=') {
          input.next();
          if (input.value(sb)) {
            String value = sb.toString();
            if (log.beDebug()) {
              log.debugT("set(65)", "parsing cookie " + name + "=" + value);
            }
            Map avs = getValues(input, sb);

            String domain = getDomain(url, avs);
            String path = getPath(url, avs);
            int port = getPort(url, avs);
            long expires = getExpires(avs);
            int version = getVersion(avs);
            SlimCookie cookie = new SlimCookie(name, value, domain, 
              path, port, now, expires, version, avs, null);
            context.setCookie(cookie);
          }
        }
        input.skipComma();
      }
    }
    catch (Exception ex) {
      log.infoT("set(81)", "reading cookie from (" + header + ")" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(ex));
    }
  }

  private final String name;
  private final String value;
  private final String domain;
  private final int domainLen;
  private final String path;
  private final int port;
  private final int version;
  private final long created;
  private final long expires;

  private final String key;
  private final String headerValue;

  private SlimCookie(String name, String value,
    String domain, String path,
    int port, long created, long expires,
    int version, Map avs, String headerValue) {
    this.name = name;
    this.value = value;
    this.domain = domain;
    this.domainLen = this.domain.length();
    this.path = path;
    this.port = port;
    this.created = created;
    this.expires = expires;
    this.version = version;

    StringBuffer sb = new StringBuffer(128);
    sb.append(this.domain);
    sb.append(":").append(this.port);
    sb.append(this.path);
    sb.append(" - ").append(this.name);
    this.key = sb.toString();

    if (headerValue == null) {
      sb.setLength(0);
      sb.append(this.name).append('=').append(this.value);
      if (this.version >= 1 && avs != null) {
        if (avs.containsKey("path")) {
          sb.append("; $Path=").append(this.path);
        }
        if (avs.containsKey("domain")) {
          sb.append("; $Domain=").append(this.domain);
        }
        if (avs.containsKey("port")) {
          sb.append("; $Port=\"").append(this.port).append('\"');
        }
      }
      headerValue = sb.toString();
    }
    this.headerValue = headerValue;

    if (log.beDebug()) {
      log.debugT("SlimCookie(133)", "new cookie: " + this);
    }
  }

  //-------- ICookie -----------------------------------------------------------
  
  public boolean isExpired() {
    return System.currentTimeMillis() >= this.expires;
  }

  public boolean appliesTo(IUri uri) {
    if (uri instanceof HttpUrl) {
      HttpUrl url = (HttpUrl)uri;
      if (this.port == url.getPort()
         && url.getPath().startsWith(this.path)
         && matchesDomain(url.getHost())) {
        return true;
      }
    }
    return false;
  }

  public String getKey() {
    return this.key;
  }

  public int getVersion() {
    return this.version;
  }

  public String getHeaderValue() {
    return this.headerValue;
  }

  //-------- IExtendedCookie ---------------------------------------------------
  
  public long getCreationTime() {
    return this.created;
  }
  
  public long getExpiryTime() {
    return this.expires;
  }
  
  public IExtendedCookie setExpiryTime(long expiry) {
    return new SlimCookie(this.name, this.value, this.domain, this.path, this.port,
      this.created, expiry, this.version, null, this.headerValue);
  }
  

  //-------- general -----------------------------------------------------------
  public String toString() {
    StringBuffer sb = new StringBuffer(128);
    sb.append("SlimCookie[domain=").append(this.domain).append(":");
    sb.append(this.port).append(this.path).append(", name=");
    sb.append(this.name).append(", value=").append(this.value).append("]");
    return sb.toString();
  }

  // ------------------------------------- private ------------------------------------

  private boolean matchesDomain(String host) {
    // See RFC 2965 for the definition of domain matching
    //
    int hlen = host.length();
    if (hlen == this.domainLen) {
      return this.domain.equalsIgnoreCase(host);
    }
    else if (this.domainLen > 0 && this.domain.charAt(0) == '.') {
      if (this.domainLen == 1) {
        //hmm, let domain=. match everything?
        return true;
      }
      else {
        if (hlen > this.domainLen) {
          // host=a.b.c.com domain=.c.com must match
          int prefix = hlen - this.domainLen;
          host = host.substring(prefix);
          return this.domain.equalsIgnoreCase(host);
        }
        else {
          // host is shorter than domain, can happen with
          // host=slashdot.org domain=.slashdot.org
          // and must match
          if (hlen + 1 == this.domainLen) {
            String domainHost = this.domain.substring(1);
            return domainHost.equalsIgnoreCase(host);
          }
        }
      }
    }

    return false;
  }

  private static boolean isCookieAttribute(String name) {
    return ATTRIBUTES.contains(name);
  }

  private static Map getValues(Input input, StringBuffer sb) {
    HashMap map = null;
    while (input.attr(sb)) {
      String attr = sb.toString().toLowerCase();
      if (!isCookieAttribute(attr)) {
        input.pushback(attr.length());
        return map;
      }
      String value = null;
      if (input.peek() == '=') {
        input.next();
        boolean allowsWhiteSpace = attr.equals("expires");
        if (input.value(sb, allowsWhiteSpace)) {
          value = sb.toString();
        }
      }

      if (map == null) {
        map = new HashMap(11);
      }
      map.put(attr, value);
    }

    return map;
  }

  private static String getDomain(HttpUrl url, Map avs) {
    String domain = url.getHost();
    if (avs != null) {
      String tmp = (String)avs.get("domain");
      if (tmp != null && tmp.length() > 0) {
        if (tmp.charAt(0) != '.') {
          tmp = "." + tmp;
        }
        domain = tmp;
      }
    }

    return domain;
  }

  private static String getPath(HttpUrl url, Map avs) {
    String path = url.getPath();
    if (avs != null) {
      String tmp = (String)avs.get("path");
      if (tmp != null && tmp.length() > 0) {
        // OK, server supplied path value. We have to check if path is a prefix
        // of the request url. For security reasons, we cannot accept cookies
        // for paths outside the url.
        // Normally path is defined as absolute path, but there are servers (yuk!)
        // which supply relative values.
        if (tmp.charAt(0) != '/') {
          try {
            IUriReference ref = UriFactory.parseUriReference(tmp);
            ref = ref.resolveWith(url);
            if (ref.isAbsolute()) {
              IUri uri = ref.getUri();
              if (uri instanceof IHierarchicalUri) {
                IHierarchicalUri huri = (IHierarchicalUri)uri;
                if (huri.equals(url) || huri.isAncestorOf(url)) {
                  return huri.getPath();
                }
              }
            }
            // Not resolved or not hierarchical or not ancestor, fall
            // back to prefix test
          }
          catch (MalformedURLException ex) {
            // Well, path does not have to be a valid uri ref
            // Fall through to prefix test, it will probably fail anyway
            if (log.beDebug()) {
              log.debugT(ex.getMessage());
            }
          }
        }
        if (!path.startsWith(tmp)) {
          // Not a valid cookie according to RFC 2965 Ch. 3.3.2
          throw new IllegalArgumentException("cookie path outside request uri");
        }
        return tmp;
      }
    }

    int index = path.lastIndexOf('/');
    if (index >= 0 && index < path.length() - 1) {
      path = path.substring(0, index + 1);
    }
    return path;
  }

  private static long getExpires(Map avs) {
    if (avs != null) {
      String ma = (String)avs.get("max-age");
      if (ma != null && ma.length() > 0) {
        try {
          int minutes = Integer.parseInt(ma);
          return (((long)minutes) * 1000L) + System.currentTimeMillis();
        }
        catch (NumberFormatException e) {
          // ignore
          if (log.beDebug()) {
            log.debugT(e.getMessage());
          }          
        }
      }
      else {
        // old netscape spec
        String exp = (String)avs.get("expires");
        if (exp != null && exp.length() > 0) {
          try {
            DateFormat df = new DateFormat();
            Date d = df.parseHTTP(exp);
            return d.getTime();
          }
          catch (ParseException e) {
            // ignore
            if (log.beDebug()) {
              log.debugT(e.getMessage());
            }            
          }
        }

      }
    }
    return Long.MAX_VALUE;
  }

  private static int getPort(HttpUrl url, Map avs) {
    if (avs != null) {
      String port = (String)avs.get("port");
      if (port != null && port.length() > 0) {
        if (port.charAt(0) == '\"' && port.length() > 2) {
          port = port.substring(1, port.length() - 1);
        }
        try {
          return Integer.parseInt(port);
        }
        catch (NumberFormatException e) {
          // ignore
          if (log.beDebug()) {
            log.debugT(e.getMessage());
          }          
        }
      }
    }

    return url.getPort();
  }

  private static int getVersion(Map avs) {
    if (avs != null) {
      String version = (String)avs.get("version");
      if (version != null && version.length() > 0) {
        if (version.charAt(0) == '\"' && version.length() > 2) {
          version = version.substring(1, version.length() - 1);
        }
        try {
          return Integer.parseInt(version);
        }
        catch (NumberFormatException e) {
          // ignore
          if (log.beDebug()) {
            log.debugT(e.getMessage());
          }          
        }
      }
    }

    return 0;
  }

  /**
   * TBD: Description of the class.
   */
  private static class Input {
    private final String line;
    private final int len;
    private int offset;

    public Input(String line) {
      this.line = line;
      this.len = line.length();
      this.offset = 0;
    }

    public int peek() {
      if (this.offset < this.len) {
        return this.line.charAt(this.offset);
      }
      return -1;
    }

    public int next() {
      if (this.offset < this.len) {
        char c = this.line.charAt(this.offset);
        ++this.offset;
        return c;
      }
      return -1;
    }

    public void pushback(int count) {
      if (this.offset > count) {
        this.offset -= count;
        return;
      }
      this.offset = 0;
    }

    public void skipComma() {
      loop :
      for (; this.offset < this.len; ++this.offset) {
        int c = peek();
        switch (c) {
          case ',':
          case ' ':
          case '\r':
          case '\n':
          case '\t':
            break;
          default:
            break loop;
        }
      }
    }
    
    public boolean attr(StringBuffer sb) {
      sb.setLength(0);
      boolean gotChars = false;
      loop :
      for (; this.offset < this.len; ++this.offset) {
        int c = peek();
        switch (c) {
          case -1:
          case ',':
            break loop;
          case '=':
            break loop;
          case ';':
          case ' ':
          case '\r':
          case '\n':
          case '\t':
            if (gotChars) {
              break loop;
            }
            break;
          default:
            gotChars = true;
            sb.append((char)c);
            break;
        }
      }
      return gotChars;
    }

    public boolean value(StringBuffer sb) {
      return value(sb, false);
    }

    public boolean value(StringBuffer sb, boolean allowsWS) {
      sb.setLength(0);
      boolean gotChars = false;
      if (peek() == '\"') {
        loop :
        for (; this.offset < this.len; ++this.offset) {
          int c = peek();
          switch (c) {
            case -1:
            case '\"':
              break loop;
            default:
              gotChars = true;
              sb.append((char)c);
              break;
          }
        }
      }
      else {
        int chars = 0;
        boolean seenSeparator = false;
        int lastChar = 0;
        loop :
        for (; this.offset < this.len; ++this.offset) {
          ++chars;
          int c = peek();
          switch (c) {
            case -1:
              break loop;
            case ';':
              break loop;
            case '=':
              // whoops we have most likely run into the next cookie value,
              // skipping a little bit too much over whitespace and comma chars
              // lets get back to the last comma seen
              if (seenSeparator) {
                this.pushback(chars);
                sb.setLength(sb.length() - chars);
                gotChars = (sb.length() > 0);
                break loop;
                
              }
              gotChars = true;
              sb.append((char)c);
              break;
              
            case ' ':
              if (lastChar ==',') {
                chars = 2;
                seenSeparator = true;
              }
              // fall through
            case ',':
            case '\n':
            case '\t':
              if (!allowsWS) {
                break loop;
              }
            // fall through, whitespace allowed

            default:
              gotChars = true;
              sb.append((char)c);
              break;
          }
          lastChar = c;
        }
      }
      return gotChars;
    }
  }
}
