/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.http.slim;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.util.http.*;
import com.sapportals.wcm.util.uri.HttpUrl;

import java.util.*;

/**
 * Implementation of IWDContext for the slim package. <p>
 *
 * SlimContext currently supports Basic and Digest Authentication. <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Id: SlimContext.java,v 1.2 2002/08/26 11:49:38 sei Exp $
 */
final class SlimContext implements IContext {

  private final static com.sap.tc.logging.Location log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.http.slim.SlimContext.class);

  private final static Set KNOWN_AUTH_SCHEMES;

  static {
    KNOWN_AUTH_SCHEMES = new HashSet(17);
    KNOWN_AUTH_SCHEMES.add("basic");
    KNOWN_AUTH_SCHEMES.add("digest");
    KNOWN_AUTH_SCHEMES.add("ntlm");
  }

  private boolean isNew = true;

  private final Map cookies;

  private UserInfo userInfo;
  private ICredentials credentials;

  private UserInfo proxyUserInfo = UserInfo.ANONYMOUS;
  private ICredentials proxyCredentials;

  private ICredentials specialCredentials;
  private boolean triedSpecialCredentials;

  /**
   * Default Constructor, no userinfo assigned.
   */
  public SlimContext() {
    this(UserInfo.ANONYMOUS);
  }

  /**
   * Constructor with known UserInfo
   *
   * @param ui userinfo to use
   */
  public SlimContext(UserInfo ui) {
    setUserInfo(ui);
    this.cookies = new HashMap(5);
  }

  public synchronized boolean isNew() {
    if (this.isNew) {
      this.isNew = false;
      return true;
    }
    return false;
  }

  public synchronized UserInfo getUserInfo() {
    return this.userInfo;
  }

  public synchronized UserInfo getProxyUserInfo() {
    return this.proxyUserInfo;
  }

  public synchronized void setUserInfo(UserInfo ui) {
    if (ui == null || ( ! ui.equals(this.userInfo))) {
      this.userInfo = ui;
      this.credentials = null;
      this.triedSpecialCredentials = false;
    }
  }

  public synchronized void setProxyUserInfo(UserInfo ui) {
    if (ui == null || ( ! ui.equals(this.proxyUserInfo))) {
      this.proxyUserInfo = ui;
      this.proxyCredentials = null;
    }
  }

  /**
   * @return the map of valid cookies
   */
  public synchronized Map getCookies() {
    if (this.cookies.size() > 0) {
      Map result = new HashMap(this.cookies);
      return result;
    }
    return Collections.EMPTY_MAP;
  }

  public synchronized String getCookieHeaderValue(HttpUrl requestUri) {
    if (this.cookies.isEmpty()) {
      return null;
    }

    List expiredCookies = null;
    StringBuffer sb = new StringBuffer(64);
    for (Iterator iter = this.cookies.keySet().iterator(); iter.hasNext(); ) {
      String key = (String)iter.next();
      Object value = this.cookies.get(key);
      if (value instanceof ICookie) {
        ICookie cookie = (ICookie)value;
        if (cookie.isExpired()) {
          if (expiredCookies == null) {
            expiredCookies = new ArrayList(5);
          }
          expiredCookies.add(key);
        }
        else if (cookie.appliesTo(requestUri)) {
          if (log.beDebug()) {
            log.debugT("getCookieHeaderValue(124)", "" + cookie + " applies to " + requestUri);
          }
          if (sb.length() > 0) {
            sb.append("; ");
          }
          else {
            // see RFC 2965
            sb.append("$Version=").append(cookie.getVersion()).append("; ");
          }
          sb.append(cookie.getHeaderValue());
        }
        else {
          if (log.beDebug()) {
            log.debugT("getCookieHeaderValue(137)", "" + cookie + " does not apply to " + requestUri);
          }
        }
      }
      else {
        if (sb.length() > 0) {
          sb.append("; ");
        }
        sb.append(key).append("=").append(value);
      }
    }

    if (expiredCookies != null) {
      for (int i = 0, n = expiredCookies.size(); i < n; ++i) {
        this.cookies.remove((String)expiredCookies.get(i));
      }
    }

    return (sb.length() > 0) ? sb.toString() : null;
  }

  public synchronized void setCookie(ICookie cookie) {
    if (cookie.isExpired()) {
      this.cookies.remove(cookie.getKey());
    }
    else {
      this.cookies.put(cookie.getKey(), cookie);
    }
  }

  public synchronized boolean appliesCredentials() {
    // ignore authentication at proxies
    return (this.credentials != null);
  }

  public synchronized boolean applyCredentials(IRequester requester, String uri, IRequest request)
    throws WcmException {
    boolean applied = false;
    if (this.credentials != null) {
      applied |= this.credentials.apply(requester, uri, request, "Authorization");
    }
    if (this.proxyCredentials != null) {
      applied |= this.proxyCredentials.apply(requester, uri, request, "Proxy-Authorization");
    }
    return applied;
  }

  public String toString() {
    return this.userInfo.toString();
  }

  public synchronized boolean setupCredentials(IRequester requester, Headers headers)
    throws WcmException {
    // We need to set up credentials in this context (probably got a 401 response)
    // If we have a special credentials handler, let this run first.
    // If we did not already try and fail with these special one.
    //
    if (this.specialCredentials != null && !this.triedSpecialCredentials) {
      if (this.credentials == null) {
        // Never tried the special one before
        //
        if (this.specialCredentials.setup(requester, headers.get("WWW-Authenticate"))) {
          this.credentials = this.specialCredentials;
          return true;
        }
      }
      else if (this.credentials.equals(this.specialCredentials)) {
        // Tried the special one before, give it a chance to retry, if it wants.
        //
        if (this.specialCredentials.setup(requester, headers.get("WWW-Authenticate"), true)) {
          return true;
        }
      }

      // Nope, special handler gave up. Never try again in this context and give
      // other (default) handlers a chance
      //
      this.triedSpecialCredentials = true;
    }

    if (UserInfo.ANONYMOUS.equals(this.userInfo)) {
      this.credentials = null;
      if (log.beDebug()) {
        log.debugT("setupCredentials(220)", "setupCredentials for anonymous user not possible, returning null");
      }
    }
    else {
      this.credentials = setup(requester, headers.get("WWW-Authenticate"), this.userInfo, this.credentials);
      if (log.beDebug()) {
        log.debugT("setupCredentials(226)", "setupCredentials run, returning " + this.credentials);
      }
    }
    return (this.credentials != null);
  }

  /**
   * Set the credentials for this context depending on the header information
   * (received from a 407 response). The header is assumed to containt one ore
   * more Proxy-Authenticate headers, which indicate the authorization schemes
   * the server understands.
   *
   * @param requester TBD: Description of the incoming method parameter
   * @param headers TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception WcmException Exception raised in failure situation
   */
  public synchronized boolean setupProxyCredentials(IRequester requester, Headers headers)
    throws WcmException {
    if (UserInfo.ANONYMOUS.equals(this.proxyUserInfo)) {
      this.proxyCredentials = null;
    }
    else {
      this.proxyCredentials = setup(requester, headers.get("Proxy-Authenticate"), this.proxyUserInfo, this.proxyCredentials);
    }
    return (this.proxyCredentials != null);
  }

  /**
   * Give the context the possiblity to process authenticate information in the
   * response message, like Authenticate-Info
   *
   * @param requester TBD: Description of the incoming method parameter
   * @param response TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   */
  public synchronized void responseCredentials(IRequester requester, IResponse response)
    throws WcmException {
    if (this.credentials != null) {
      this.credentials.got(requester, response);
    }
  }

  public synchronized void setSpecialCredentials(ICredentials handler) {
    this.specialCredentials = handler;
    this.triedSpecialCredentials = false;
  }

  public synchronized ICredentials getSpecialCredentials() {
    return this.specialCredentials;
  }

  public synchronized void openedConnection(IRequester requester) {
    startUse(requester);
  }

  public synchronized void startUse(IRequester requester) {
    if (this.credentials != null) {
      this.credentials.startUse(requester);
    }
    if (this.proxyCredentials != null) {
      this.proxyCredentials.startUse(requester);
    }
  }

  public synchronized void endUse(IRequester requester) {
    if (this.credentials != null) {
      this.credentials.endUse(requester);
    }
    if (this.proxyCredentials != null) {
      this.proxyCredentials.endUse(requester);
    }
  }

  public synchronized boolean canTriggerAuthentication(IRequester requester) {
    if (UserInfo.ANONYMOUS.equals(this.userInfo)) {
      // We have no user information
      return false;
    }
    if (this.specialCredentials != null && !this.triedSpecialCredentials) {
      // we apply special credentials (login ticket) and do not expect authentication responses
      return false;
    }
    if (this.proxyCredentials != null && this.proxyCredentials.canTriggerAuthentication(requester)) {
      // we do use proxy credentials and those may require 401 responses (NTLM in proxy)
      return true;
    }
    if (this.credentials == null) {
      // we have no credentials yet, so it is possible that 401 will fly into our face
      return true;
    }
    // we have user info and credentials, ask credentials if further 401s are possible
    return this.credentials.canTriggerAuthentication(requester);
  }

  //------------------ protected / private ------------------------------

  private static ICredentials setup(IRequester requester, String value, UserInfo ui, ICredentials credentials)
    throws WcmException {
    if (value == null) {
      return credentials;
    }

    boolean debug = log.beDebug();
    if (debug) {
      log.debugT("setup(331)", "setting up credentials for header value: " + value);
    }
    Map schemes = new HashMap();
    String scheme = null;
    StringBuffer sb = new StringBuffer(value.length());
    for (StringTokenizer tok = new StringTokenizer(value, ","); tok.hasMoreTokens(); ) {
      String schemeval = tok.nextToken().trim();
      int index = schemeval.indexOf(' ');
      if (index > 0 && schemeval.lastIndexOf('"', index) < 0
         || KNOWN_AUTH_SCHEMES.contains(schemeval.toLowerCase())) {
        // This is the start of an authentication scheme
        //
        if (scheme != null) {
          schemes.put(scheme, sb.toString());
          sb.setLength(0);
        }
        scheme = (index >= 0) ? schemeval.substring(0, index).toLowerCase() : schemeval.toLowerCase();
        sb.append(schemeval);
        if (debug) {
          log.debugT("setup(350)", "accepted scheme: " + scheme);
        }
      }
      else {
        // This is a parameter to the current authentication scheme
        //
        if (scheme != null) {
          sb.append(", ").append(schemeval);
        }
      }
    }

    if (scheme != null) {
      schemes.put(scheme, sb.toString());
      sb.setLength(0);
    }

    String existing = "";
    if (credentials != null) {
      existing = credentials.getName().toLowerCase();
      if (debug) {
        log.debugT("setup(371)", "trying existing scheme: " + existing);
      }
      if (!schemes.containsKey(existing)
         || !credentials.setup(requester, (String)schemes.get(existing), true)) {
        credentials = null;
      }
    }

    if (credentials == null && schemes.containsKey("digest")) {
      if (debug) {
        log.debugT("setup(381)", "trying scheme: digest");
      }
      credentials = new WDDigestAuthentication(ui);
      if (!credentials.setup(requester, (String)schemes.get("digest"))) {
        credentials = null;
      }
    }

    if (credentials == null && schemes.containsKey("ntlm") && !"ntlm".equals(existing)) {
      // NTLM offered and not tried before
      if (debug) {
        log.debugT("setup(392)", "trying scheme: ntlm");
      }
      credentials = new WDNTLMAuthentication(ui);
      if (!credentials.setup(requester, (String)schemes.get("ntlm"))) {
        credentials = null;
      }
    }

    if (credentials == null && schemes.containsKey("basic")) {
      if (debug) {
        log.debugT("setup(402)", "trying scheme: basic");
      }
      credentials = new WDBasicAuthentication(ui);
      if (!credentials.setup(requester, (String)schemes.get("basic"))) {
        credentials = null;
      }
    }

    return credentials;
  }

}

