/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.http.slim;

import java.util.Arrays;

/**
 * Implements MD4 message digest algorithm according to RFC 1320. <p>
 *
 * Derived from the RSA Data Security, Inc. MD4 Message-Digest Algorithm. <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Id: MD4.java,v 1.1 2003/01/06 16:06:56 sei Exp $
 */
public final class MD4 {

  // 64 bytes of maximum needed padding bytes
  private final static byte[] PADDING = {
    (byte)0x80, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    };

  private final static int S11 = 3;
  private final static int S12 = 7;
  private final static int S13 = 11;
  private final static int S14 = 19;
  private final static int S21 = 3;
  private final static int S22 = 5;
  private final static int S23 = 9;
  private final static int S24 = 13;
  private final static int S31 = 3;
  private final static int S32 = 9;
  private final static int S33 = 11;
  private final static int S34 = 15;

  public static MD4 getInstance() {
    return new MD4();
  }

  // number of seen bytes of message without padding
  private int count;

  private byte[] buffer = new byte[64];
  private int state0;
  private int state1;
  private int state2;
  private int state3;


  private MD4() {
    reset();
  }

  public void update(byte[] input) {
    update(input, 0, input.length);
  }

  public void update(byte[] input, int offset, int length) {
    doUpdate(input, offset, length);
  }

  public byte[] digest() {
    return doFinal();
  }

  public byte[] digest(byte[] input) {
    update(input);
    return digest();
  }

  public byte[] digest(byte[] input, int offset, int length) {
    update(input, offset, length);
    return digest();
  }

  public void reset() {
    this.count = 0;
    this.state0 = 0x67452301;
    this.state1 = 0xefcdab89;
    this.state2 = 0x98badcfe;
    this.state3 = 0x10325476;
  }

  //-------------------------------- private -----------------------------------------------------

  private void doUpdate(byte[] input, int offset, int length) {
    int index = this.count % 64;
    this.count += length;
    int partLength = 64 - index;
    int i = 0;

    if (length >= partLength) {
      System.arraycopy(input, offset, this.buffer, index, partLength);
      transform(this.buffer, 0);
      for (i = partLength; i + 63 < length; i += 64) {
        transform(input, offset + i);
      }
      index = 0;
    }

    // copy remaining input
    System.arraycopy(input, offset + i, this.buffer, index, length - i);
  }

  private byte[] doFinal() {
    // message needs to be multiple of 64 bytes with last 8 bytes
    // indicating the message length in bits
    //
    int msgBytes = this.count;
    int index = this.count % 64;
    int padLength = (index < 56) ? (56 - index) : (120 - index);
    doUpdate(PADDING, 0, padLength);
    byte[] bits = encode(((long)msgBytes) << 3);
    doUpdate(bits, 0, bits.length);
    byte[] result = encodeState();
    reset();
    return result;
  }

  private int doF(int x, int y, int z) {
    return ((x & y) | (~x & z));
  }

  private int doG(int x, int y, int z) {
    return ((x & y) | (x & z) | (y & z));
  }

  private int doH(int x, int y, int z) {
    return (x ^ y ^ z);
  }

  private int rotateLeft(int x, int n) {
    return ((x << n) | (x >>> (32 - n)));
  }

  private int doFF(int a, int b, int c, int d, int x, int s) {
    a += doF(b, c, d) + x;
    return rotateLeft(a, s);
  }

  private int doGG(int a, int b, int c, int d, int x, int s) {
    a += doG(b, c, d) + x + 0x5a827999;
    return rotateLeft(a, s);
  }

  private int doHH(int a, int b, int c, int d, int x, int s) {
    a += doH(b, c, d) + x + 0x6ed9eba1;
    return rotateLeft(a, s);
  }

  private int[] decode(byte[] input, int offset, int length) {
    int[] output = new int[length / 4];
    int max = offset + length;
    for (int i = 0, j = offset; j < max; ++i, j += 4) {
      output[i] = ((input[j] & 0xff)
         | ((input[j + 1] & 0xff) << 8)
         | ((input[j + 2] & 0xff) << 16)
         | ((input[j + 3] & 0xff) << 24));
    }
    return output;
  }

  private byte[] encode(long l) {
    int[] input = new int[2];
    input[0] = (int)(l & 0x0000FFFFL);
    input[1] = (int)((l & 0xFFFF0000L) >>> 32);
    return encode(input);
  }

  private byte[] encodeState() {
    int[] input = new int[4];
    input[0] = this.state0;
    input[1] = this.state1;
    input[2] = this.state2;
    input[3] = this.state3;
    return encode(input);
  }

  private byte[] encode(int[] input) {
    byte[] output = new byte[input.length * 4];
    for (int i = 0, j = 0; j < output.length; ++i, j += 4) {
      int n = input[i];
      output[j] = (byte)(n & 0xff);
      output[j + 1] = (byte)((n >>> 8) & 0xff);
      output[j + 2] = (byte)((n >>> 16) & 0xff);
      output[j + 3] = (byte)((n >>> 24) & 0xff);
    }
    return output;
  }

  /**
   * @param block must hold 64 bytes
   * @param offset TBD: Description of the incoming method parameter
   */
  private void transform(byte[] block, int offset) {
    int a = this.state0;
    int b = this.state1;
    int c = this.state2;
    int d = this.state3;

    if (block.length < (64 + offset)) {
      throw new IllegalArgumentException("block must hold at least 64 bytes");
    }
    int[] x = decode(block, offset, 64);

    /*
     *  Round 1
     */
    a = doFF(a, b, c, d, x[0], S11);
    /*
     *  1
     */
    d = doFF(d, a, b, c, x[1], S12);
    /*
     *  2
     */
    c = doFF(c, d, a, b, x[2], S13);
    /*
     *  3
     */
    b = doFF(b, c, d, a, x[3], S14);
    /*
     *  4
     */
    a = doFF(a, b, c, d, x[4], S11);
    /*
     *  5
     */
    d = doFF(d, a, b, c, x[5], S12);
    /*
     *  6
     */
    c = doFF(c, d, a, b, x[6], S13);
    /*
     *  7
     */
    b = doFF(b, c, d, a, x[7], S14);
    /*
     *  8
     */
    a = doFF(a, b, c, d, x[8], S11);
    /*
     *  9
     */
    d = doFF(d, a, b, c, x[9], S12);
    /*
     *  10
     */
    c = doFF(c, d, a, b, x[10], S13);
    /*
     *  11
     */
    b = doFF(b, c, d, a, x[11], S14);
    /*
     *  12
     */
    a = doFF(a, b, c, d, x[12], S11);
    /*
     *  13
     */
    d = doFF(d, a, b, c, x[13], S12);
    /*
     *  14
     */
    c = doFF(c, d, a, b, x[14], S13);
    /*
     *  15
     */
    b = doFF(b, c, d, a, x[15], S14);
    /*
     *  16
     */
    /*
     *  Round 2
     */
    a = doGG(a, b, c, d, x[0], S21);
    /*
     *  17
     */
    d = doGG(d, a, b, c, x[4], S22);
    /*
     *  18
     */
    c = doGG(c, d, a, b, x[8], S23);
    /*
     *  19
     */
    b = doGG(b, c, d, a, x[12], S24);
    /*
     *  20
     */
    a = doGG(a, b, c, d, x[1], S21);
    /*
     *  21
     */
    d = doGG(d, a, b, c, x[5], S22);
    /*
     *  22
     */
    c = doGG(c, d, a, b, x[9], S23);
    /*
     *  23
     */
    b = doGG(b, c, d, a, x[13], S24);
    /*
     *  24
     */
    a = doGG(a, b, c, d, x[2], S21);
    /*
     *  25
     */
    d = doGG(d, a, b, c, x[6], S22);
    /*
     *  26
     */
    c = doGG(c, d, a, b, x[10], S23);
    /*
     *  27
     */
    b = doGG(b, c, d, a, x[14], S24);
    /*
     *  28
     */
    a = doGG(a, b, c, d, x[3], S21);
    /*
     *  29
     */
    d = doGG(d, a, b, c, x[7], S22);
    /*
     *  30
     */
    c = doGG(c, d, a, b, x[11], S23);
    /*
     *  31
     */
    b = doGG(b, c, d, a, x[15], S24);
    /*
     *  32
     */
    /*
     *  Round 3
     */
    a = doHH(a, b, c, d, x[0], S31);
    /*
     *  33
     */
    d = doHH(d, a, b, c, x[8], S32);
    /*
     *  34
     */
    c = doHH(c, d, a, b, x[4], S33);
    /*
     *  35
     */
    b = doHH(b, c, d, a, x[12], S34);
    /*
     *  36
     */
    a = doHH(a, b, c, d, x[2], S31);
    /*
     *  37
     */
    d = doHH(d, a, b, c, x[10], S32);
    /*
     *  38
     */
    c = doHH(c, d, a, b, x[6], S33);
    /*
     *  39
     */
    b = doHH(b, c, d, a, x[14], S34);
    /*
     *  40
     */
    a = doHH(a, b, c, d, x[1], S31);
    /*
     *  41
     */
    d = doHH(d, a, b, c, x[9], S32);
    /*
     *  42
     */
    c = doHH(c, d, a, b, x[5], S33);
    /*
     *  43
     */
    b = doHH(b, c, d, a, x[13], S34);
    /*
     *  44
     */
    a = doHH(a, b, c, d, x[3], S31);
    /*
     *  45
     */
    d = doHH(d, a, b, c, x[11], S32);
    /*
     *  46
     */
    c = doHH(c, d, a, b, x[7], S33);
    /*
     *  47
     */
    b = doHH(b, c, d, a, x[15], S34);
    /*
     *  48
     */
    this.state0 += a;
    this.state1 += b;
    this.state2 += c;
    this.state3 += d;

    // Zeroize sensitive information.
    Arrays.fill(x, 0);
  }
}
