/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.http.slim;

import java.io.IOException;
import java.io.InputStream;

/**
 * ClientInputStream is a proxy class for another InputStream. Its purpose is to
 * notify the underlying requester object that this stream has been closed by
 * the client. That gives the requester a chance to clean up its connections.
 * Otherwise it would eventually have to depend on the GC, which is not what we
 * want. <p>
 *
 * Copyright (c) SAP AG 2001-2003
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Id: ClientInputStream.java,v 1.3 2003/02/17 14:24:04 jre Exp $
 */
final class ClientInputStream extends InputStream {

  private final static com.sap.tc.logging.Location log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.http.slim.ClientInputStream.class);

  private InputStream is;
  private SlimRequester requester;
  private final boolean disableMark;
  private long len;
  private boolean closed = false;

  /**
   * Create a new InputStream with 0 length.
   */
  public ClientInputStream() {
    this.is = null;
    this.len = 0;
    this.requester = null;
    this.disableMark = false;
  }

  /**
   * @param input the InputStream to read from
   * @param requester the SlimRequester to notify about close()
   * @param len TBD: Description of the incoming method parameter
   * @param disableMark TBD: Description of the incoming method parameter
   */
  public ClientInputStream(InputStream input, long len, SlimRequester requester, boolean disableMark) {
    this.is = input;
    this.len = (len >= 0) ? len : Long.MAX_VALUE;
    this.requester = requester;
    this.disableMark = disableMark;
  }

  public int available()
    throws IOException {
    if (this.len == 0) {
      return 0;
    }
    return this.is.available();
  }

  public void close()
    throws IOException {
    if (!this.closed) {
      // On first close, tell the requester that he might
      // close the connection now, if necessary.
      //
      this.closed = true;
      if (this.len != 0) {
        if (this.requester != null) {
          if (log.beInfo()) {
            log.infoT("close(76)", "not read empty, closing requester");
          }
          this.requester.close();
        }
        else if (this.is != null) {
          if (log.beInfo()) {
            log.infoT("close(82)", "not read empty, requester null, closing input");
          }
          this.is.close();
        }
      }

      if (this.requester != null) {
        if (log.beInfo()) {
          log.infoT("close(90)", "releasing requester");
        }
        this.requester.responseClosed();
      }
      else {
        // requester is intentionally null, we should not touch
        // the underlying stream
      }

      this.requester = null;
      this.is = null;
    }
  }

  public void mark(int readlimit) {
    if (this.is != null) {
      this.is.mark(readlimit);
    }
  }

  public boolean markSupported() {
    if (this.is != null && !this.disableMark) {
      return this.is.markSupported();
    }
    else {
      return false;
    }

  }

  public int read()
    throws IOException {
    if (this.len <= 0) {
      return -1;
    }
    int result = this.is.read();
    if (result != -1) {
      --this.len;
    }
    else {
      this.len = 0;
    }
    return result;
  }

  public int read(byte[] buffer)
    throws IOException {
    return read(buffer, 0, buffer.length);
  }

  public int read(byte[] buffer, int offset, int length)
    throws IOException {
    if (this.len <= 0) {
      return -1;
    }
    if (length > this.len) {
      length = (int)this.len;
    }
    int bytes = this.is.read(buffer, offset, length);
    if (bytes >= 0) {
      this.len -= bytes;
    }
    else {
      this.len = 0;
    }
    return bytes;
  }

  public void reset()
    throws IOException {
    if (this.is != null) {
      this.is.reset();
    }
  }

  public long skip(long bytes)
    throws IOException {
    if (this.len <= 0) {
      return -1;
    }
    long skipped = this.is.skip(bytes);
    this.len -= skipped;
    return skipped;
  }

//  private void readEmpty() throws IOException {
//    if (this.is != null && this.len > 0) {
//      byte[] buffer = new byte[64*1024];
//      while (read(buffer, 0, buffer.length) != -1) {
//        //
//      }
//    }
//  }
}
