/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.http.slim;

import java.io.IOException;
import java.io.InputStream;

/**
 * ChunkedInputStream reads form a InputStream in HTTP transfer encoding
 * chunked, as from a HTTP/1.1 connection. <p>
 *
 * On close() it will therefore <b>not</b> close the underlaying HTTP stream.
 * <p>
 *
 * Copyright (c) SAP AG 2001-2003
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Id: ChunkedInputStream.java,v 1.3 2003/02/17 14:24:04 jre Exp $
 */

final class ChunkedInputStream extends InputStream {

  private final static com.sap.tc.logging.Location log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.http.slim.ChunkedInputStream.class);

  /**
   * inputstream from remote system with resource content
   */
  private final InputStream m_is;
  /**
   * indicates that stream has been closed
   */
  private boolean m_closed = false;
  /**
   * end of internal input stream has been reached
   */
  private boolean m_eof = false;

  /**
   * buffer for reading chunks
   */
  private final byte[] m_buffer = new byte[32 * 1024];
  /**
   * current offset in chunk buffer
   */
  private int m_offset = 0;
  /**
   * actual length in chunk buffer
   */
  private int m_length = 0;
  /**
   * number of unread bytes in chunk
   */
  private int m_remainingInChunk = 0;

  /**
   * Create a new ChunkInputStream for reading chunked encoded bytes from
   * another InputStream.
   *
   * @param is the other InputStream
   */
  public ChunkedInputStream(InputStream is) {
    m_is = is;
  }

  public int available()
    throws IOException {
    return m_length;
  }

  /**
   * Closes this stream.
   *
   * @exception IOException Exception raised in failure situation
   */
  public void close()
    throws IOException {
    log.debugT("close(82)", "ChunkedInputStream::close()");
    if (!m_closed) {
      m_closed = true;
      readEmpty();
    }
  }

  public void mark(int readlimit) { }

  public boolean markSupported() {
    return false;
  }

  public int read()
    throws IOException {
    if (m_offset >= m_length) {
      fillBuffer();
    }

    if (m_offset < m_length) {
      return m_buffer[m_offset++] & 0xff;
    }

    return -1;
  }

  public int read(byte[] buffer, int offset, int length)
    throws IOException {
    if (m_offset >= m_length) {
      fillBuffer();
    }
    if (m_eof) {
      return -1;
    }

    int len = Math.min(m_length - m_offset, length);
    System.arraycopy(m_buffer, m_offset, buffer, offset, len);
    m_offset += len;
    return len;
  }

  public void reset() { }

  /**
   * Fills the chunk buffer with more bytes from the input stream
   *
   * @exception IOException Exception raised in failure situation
   */
  private void fillBuffer()
    throws IOException {
    if (m_eof) {
      return;
    }

    m_offset = 0;
    m_length = 0;

    // Do we know that the current chunk has more bytes?
    // We read the current chunk empty. Only if the chunk
    // is exhausted, we care about the next chunk header
    //
    if (m_remainingInChunk <= 0) {
      // Last chunk is exhausted, we expect a new length line
      //
      String line = readLine();
      if (line == null && !m_eof) {
        // This is normal on the second and onward chunks since
        // a chunk ends with cr+lf. Length information is in
        // next line.
        //
        line = readLine();
      }

      if (log.beDebug()) {
        log.debugT("fillBuffer(156)", "reading chunk size from: (" + line + "), eof=" + m_eof);
      }

      try {
        m_remainingInChunk = Integer.parseInt(line.trim(), 16);
      }
      catch (NumberFormatException ex) {
            //$JL-EXC$        
        throw new IOException("cannot read chunk size: " + line);
      }

      if (m_remainingInChunk == 0) {
        log.debugT("fillBuffer(167)", "ChunkedInputStream::fillBuffer() was last chunk");
        // last empty chunk seen
        // TODO: parsing of possibly trailing headers
        readLine();
        m_eof = true;

        return;
      }

      if (log.beDebug()) {
        log.debugT("fillBuffer(177)", "ChunkedInputStream::fillBuffer() chunk size: " + m_remainingInChunk);
      }
    }

    // Now, m_remainingInChunk is one of
    // a) bytes left from the actual chunk
    // b) the length of the next chunk with the inputstream at the start of the chunk
    //
    int min = Math.min(m_buffer.length, m_remainingInChunk);
    m_length = m_is.read(m_buffer, 0, min);
    if (m_length == -1) {
      throw new IOException("Unexpected end if chunked stream");
    }
    // We have read m_length bytes from the current chunk
    //
    m_remainingInChunk -= m_length;
    if (log.beDebug()) {
      log.debugT("fillBuffer(194)", "ChunkedInputStream::fillBuffer() read " + m_length
         + " remaining " + m_remainingInChunk);
      StringBuffer sb = new StringBuffer();
      sb.append("read:");
      for (int i = 0; i < m_length && i < 16; ++i) {
        sb.append(' ').append(Byte.toString(m_buffer[i]));
      }
      log.debugT("fillBuffer(201)", sb.toString());
    }
  }

  /**
   * Read a line from the inpustream, up to, but not including, the cr+lf. This
   * method can safely assume that bytes == char, since the chunk meta
   * information is transmitted in US-ASCII.
   *
   * @return TBD: Description of the outgoing return value
   * @exception IOException Exception raised in failure situation
   */
  private String readLine()
    throws IOException {
    StringBuffer sb = new StringBuffer();
    while (true) {
      int n = m_is.read();
      if (n == -1) {
        m_eof = true;
        break;
      }
      if (n == '\r') {
        continue;
      }
      if (n == '\n') {
        break;
      }

      sb.append((char)n);
    }

    return (sb.length() > 0) ? sb.toString() : null;
  }

  private void readEmpty()
    throws IOException {
    if (!m_eof) {
      byte[] buffer = new byte[64 * 1024];
      while (read(buffer, 0, buffer.length) != -1) {
        //
      }
    }
  }
}
