/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.http;

import com.sapportals.wcm.util.uri.HttpUrl;
import com.sapportals.wcm.util.uri.IHierarchicalUri;

import java.util.*;

/**
 * An IWDProxyInfo objects keeps information about proxy servers for http
 * requests and determines if for a given hostname the proxy shall be used. <p>
 *
 * ProxyInfo uses the standard property names, e.g. http.proxyHost,
 * http.proxyPort and http.nonProxyHosts (or https.proxyHost). See the different
 * constructors for where the properties are looked up from. <p>
 *
 * Copyright (c) SAP AG 2001-2003
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Id: ProxyInfo.java,v 1.4 2003/02/17 14:24:04 jre Exp $
 */
public final class ProxyInfo implements IProxyInfo {

  // private static final com.sap.tc.logging.Location log = com.sap.tc.logging.Location.getLocation(ProxyInfo.class.getName());

  public final static String PROP_PROXY_HOST = "proxyHost";
  public final static String PROP_PROXY_PORT = "proxyPort";
  public final static String PROP_NO_PROXY_HOSTS = "nonProxyHosts";
  public final static String PROP_PROXY_SYSTEM = "proxySystem";

  /**
   * TBD: Description of the class.
   */
  private final static class PI {
    public final HttpUrl url;
    public final String systemID;

    public final List noProxyHosts;
    public final List noProxyDomains;

    public PI(String protocol, Properties props) {
      StringBuffer sb = new StringBuffer(128);
      sb.append(protocol).append('.').append(PROP_PROXY_SYSTEM);
      String systemid = props.getProperty(sb.toString());
      if (systemid != null && systemid.length() > 0) {
        this.systemID = systemid;
      }
      else {
        this.systemID = null;
      }

      int defaultPort = protocol.equals("https") ? 443 : 80;

      sb.setLength(0);
      sb.append(protocol).append('.').append(PROP_PROXY_HOST);
      String host = props.getProperty(sb.toString());
      if (host != null && host.length() > 0) {
        int port = defaultPort;
        sb.setLength(0);
        sb.append(protocol).append('.').append(PROP_PROXY_PORT);
        String tmp = props.getProperty(sb.toString());
        if (tmp != null) {
          try {
            port = Integer.parseInt(tmp);
            if (port < 0) {
              port = defaultPort;
            }
          }
          catch (NumberFormatException ex) {
            port = defaultPort;
          }
        }

        this.url = new HttpUrl(protocol, host, port, "/", null);

        sb.setLength(0);
        sb.append(protocol).append('.').append(PROP_NO_PROXY_HOSTS);
        tmp = props.getProperty(sb.toString());
        if (tmp != null && tmp.length() > 0) {
          this.noProxyHosts = new ArrayList(10);
          this.noProxyDomains = new ArrayList(10);

          StringTokenizer strtok = new StringTokenizer(tmp, ",;| \t\n\r");
          while (strtok.hasMoreElements()) {
            String token = strtok.nextToken().toLowerCase();
            if (token.length() > 1) {
              if (token.charAt(0) == '*') {
                token = token.substring(1);
              }
              if (token.length() > 0) {
                if (token.charAt(0) == '.') {
                  this.noProxyDomains.add(token);
                }
                else {
                  this.noProxyHosts.add(token);
                }
              }
            }
          }
        }
        else {
          this.noProxyHosts = null;
          this.noProxyDomains = null;
        }
      }
      else {
        this.url = null;
        this.noProxyHosts = null;
        this.noProxyDomains = null;
      }
    }

    public PI(PI other, HttpUrl url) {
      this.url = url;
      this.systemID = other.systemID;
      this.noProxyDomains = other.noProxyDomains;
      this.noProxyHosts = other.noProxyHosts;
    }

    public String toString() {
      return "PI[uri=" + this.url + ", id=" + this.systemID
        + ", nonDomains=" + this.noProxyDomains + ", noHosts=" + this.noProxyHosts + "]"; 
    }
  }

  private final static PI EMPTY = new PI("http", new Properties());

  private final Map pis;
  private PI http;

  /**
   * Creates a new object which initializes from System properties
   */
  public ProxyInfo() {
    this(System.getProperties());
  }

  /**
   * Creates a new object which initializes from given properties
   *
   * @param props TBD: Description of the incoming method parameter
   */
  public ProxyInfo(Properties props) {
    this.http = new PI("http", props);
    PI https = new PI("https", props);
    if (https.url == null && https.systemID == null) {
      https = this.http;
    }

    this.pis = new HashMap(5);
    this.pis.put("http", http);
    this.pis.put("https", https);
  }

  public void setProxyUri(IHierarchicalUri base, IHierarchicalUri proxy) {

    if (proxy instanceof HttpUrl) {
      String scheme = base.getScheme();
      HttpUrl url = (HttpUrl)proxy;
      PI pi = getPI(scheme);
      PI npi = new PI(pi, url);
      setPI(scheme, npi);
    }
  }

  public String toString() {
    return "ProxyInfo[" + this.pis + "]"; 
  }
  
  //---------------- interface IWDProxyInfo ------------------------------------

  public String getProxySystemID(String protocol) {
    return getPI(protocol).systemID;
  }

  public HttpUrl getProxyFor(HttpUrl server) {
    PI pi = getPI(server.getScheme());
    if (pi.url == null) {
      return null;
    }

    if (pi.noProxyHosts != null || pi.noProxyDomains != null) {
      String host = server.getHost().toLowerCase();
      synchronized (pi) {
        for (int i = 0, n = pi.noProxyHosts.size(); i < n; ++i) {
          String tmp = (String)pi.noProxyHosts.get(i);
          if (tmp.equals(host)) {
            return null;
          }
        }
        for (int i = 0, n = pi.noProxyDomains.size(); i < n; ++i) {
          String tmp = (String)pi.noProxyDomains.get(i);
          // TODO: needs better check
          if (host.indexOf(tmp) >= 0) {
            return null;
          }
        }
      }
    }

    return pi.url;
  }

  //------------------------------- private ------------------------------------

  private PI getPI(String protocol) {
    protocol = protocol.toLowerCase();
    if ("http".equals(protocol)) {
      return this.http;
    }

    synchronized (this.pis) {
      PI pi = (PI)this.pis.get(protocol);
      if (pi == null) {
        return EMPTY;
      }

      return pi;
    }
  }

  private void setPI(String scheme, PI pi) {
    if ("http".equals(scheme)) {
      this.http = pi;
    }

    synchronized (this.pis) {
      this.pis.put(scheme, pi);
    }
  }
}

