/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.http;

import com.sapportals.wcm.WcmException;

/**
 * Interface for a module implementing a HTTP authorization scheme. <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Id: ICredentials.java,v 1.1 2002/08/26 11:49:38 sei Exp $
 */

public interface ICredentials {

  /**
   * Return the name of this authorization scheme
   *
   * @return name
   */
  public String getName();

  /**
   * Calculate the credentials for this HTTP Request
   *
   * @param uri of request
   * @param request to make
   * @param headerName of authorization header
   * @param requester TBD: Description of the incoming method parameter
   * @return if credential was applied to request
   * @exception WcmException Exception raised in failure situation
   */
  public boolean apply(IRequester requester, String uri, IRequest request, String headerName)
    throws WcmException;

  /**
   * Set the credentials for this context depending on the header information
   * (received from a 401 response). The header will start with the name of this
   * credential.
   *
   * @param header value from WWW-Authenticate response header
   * @param requester TBD: Description of the incoming method parameter
   * @return if setup was successfull and credentials can be calculated
   * @exception WcmException Exception raised in failure situation
   */
  public boolean setup(IRequester requester, String header)
    throws WcmException;

  /**
   * Set the credentials for this context depending on the header information
   * (received from a 401 response). The header will start with the name of this
   * credential.
   *
   * @param header value from WWW-Authenticate response header
   * @param retry true, if the scheme was already tried before
   * @param requester TBD: Description of the incoming method parameter
   * @return if setup was successfull and credentials can be calculated
   * @exception WcmException Exception raised in failure situation
   */
  public boolean setup(IRequester requester, String header, boolean retry)
    throws WcmException;

  /**
   * Process authenticate information in the response message, like
   * Authenticate-Info.
   *
   * @param requester TBD: Description of the incoming method parameter
   * @param response TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   */
  public void got(IRequester requester, IResponse response)
    throws WcmException;

  /**
   * Announce to credentials that they are used on a new connection for the
   * specified requester.
   *
   * @param requester to apply credentials to
   */
  public void startUse(IRequester requester);

  /**
   * Announce to credentials that they are no longer used for the specified
   * requester.
   *
   * @param requester no longer used
   */
  public void endUse(IRequester requester);

  /**
   * Return if credentials can trigger HTTP 401 reponses for this requester
   *
   * @param requester to work with
   * @return TBD: Description of the outgoing return value
   */
  public boolean canTriggerAuthentication(IRequester requester);
}

