/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.http;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.util.uri.HttpUrl;

import java.util.Map;

/**
 * Interface for user and authentication information objects in http requests. A
 * IWDContext represents a user on which behalf connections are made. When
 * authorization is necessary, the context is asked to calculate the credentials
 * which shall be sent to the http server. <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author stefan.eissing@greenbytes.de
 */
public interface IContext {

  /**
   * Return user info for this context.
   *
   * @return userinfo for this context.
   */
  public UserInfo getUserInfo();

  /**
   * Return user info for proxy server
   *
   * @return userinfo for proxy server
   */
  public UserInfo getProxyUserInfo();

  /**
   * @return the map of valid cookies
   */
  public Map getCookies();

  /**
   * Return string in HTTP header value format for cookies in this context. When
   * no cookies are present, the method returns <code>null</code> .
   *
   * @param requestUri TBD: Description of the incoming method parameter
   * @return cookies in HTTP header value format or <code>null</code>
   */
  public String getCookieHeaderValue(HttpUrl requestUri);


  public boolean isNew();

  /**
   * Sets a new cookie in this context
   *
   * @param cookie
   */
  public void setCookie(ICookie cookie);

  /**
   * Sets username/password to be used in this context.
   *
   * @param ui userInfo to be set
   */
  public void setUserInfo(UserInfo ui);

  /**
   * Sets username/password to be used in this context for proxy.
   *
   * @param ui proxyUserInfo to be set
   */
  public void setProxyUserInfo(UserInfo ui);

  /**
   * Answer if authorization credentials are used in this context.
   *
   * @return if authorization credentials are used in this context
   */
  public boolean appliesCredentials();

  /**
   * Apply the credentials for a HTTP request. The type of credentials depends
   * on the last call to <code>setupCredentials</code> . The credentials are
   * applied to the request header fields.
   *
   * @param uri the requested uri
   * @param request the request to calculate the credentials for
   * @param requester TBD: Description of the incoming method parameter
   * @return if credentials have been added to the header fields
   * @exception WcmException Exception raised in failure situation
   */
  public boolean applyCredentials(IRequester requester, String uri, IRequest request)
    throws WcmException;

  /**
   * Set the credentials for this context depending on the header information
   * (received from a 401 response). The header is assumed to containt one ore
   * more WWW-Authenticate headers, which indicate the authorization schemes the
   * server understands.
   *
   * @param requester TBD: Description of the incoming method parameter
   * @param header TBD: Description of the incoming method parameter
   * @return if successful
   * @exception WcmException Exception raised in failure situation
   */
  public boolean setupCredentials(IRequester requester, Headers header)
    throws WcmException;

  /**
   * Set the credentials for this context depending on the header information
   * (received from a 407 response). The header is assumed to containt one ore
   * more Proxy-Authenticate headers, which indicate the authorization schemes
   * the server understands.
   *
   * @param requester TBD: Description of the incoming method parameter
   * @param header TBD: Description of the incoming method parameter
   * @return if successful
   * @exception WcmException Exception raised in failure situation
   */
  public boolean setupProxyCredentials(IRequester requester, Headers header)
    throws WcmException;

  /**
   * Give the context the possiblity to process authenticate information in the
   * response message, like Authenticate-Info.
   *
   * @param requester TBD: Description of the incoming method parameter
   * @param response TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   */
  public void responseCredentials(IRequester requester, IResponse response)
    throws WcmException;

  /**
   * Set a special credentials handler which is run before any of the known
   * authentication schemes (basic, digest) are run.
   *
   * @param handler to installed (or <code>null</code> to remove)
   */
  public void setSpecialCredentials(ICredentials handler);

  /**
   * Return any installed special credentials or <code>null</code> if none is
   * installed
   *
   * @return installed special credentials handler
   */
  public ICredentials getSpecialCredentials();

  /**
   * New connection in requester was opened for this context.
   *
   * @param requester which opened new connection
   */
  public void openedConnection(IRequester requester);

  /**
   * This context will be used for the given requester
   *
   * @param requester
   */
  public void startUse(IRequester requester);

  /**
   * This context will no longer be used for the given requester
   *
   * @param requester
   */
  public void endUse(IRequester requester);

  /**
   * Return if authentication responses (HTTP 401) can be triggered in this
   * context.
   *
   * @param requester to work in
   * @return TBD: Description of the outgoing return value
   */
  public boolean canTriggerAuthentication(IRequester requester);
}

