/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.http;

import com.sapportals.wcm.WcmException;

/**
 * WDStatus represents a HTTP conform status line with protocol, status code and
 * optional description. <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Id: HttpStatus.java,v 1.1 2002/08/21 17:00:02 sei Exp $
 */
public class HttpStatus {

  private String m_protocol;
  private int m_status;
  private String m_description;

  /**
   * Return code from a HTTP status line
   *
   * @param line TBD: Description of the incoming method parameter
   * @return code
   * @exception WcmException Exception raised in failure situation
   */
  public static int getCode(String line)
    throws WcmException {
    return parseInternal(line, null);
  }

  /**
   * Construct a new empty status
   */
  public HttpStatus() { }

  /**
   * Construct a new status from the given line
   *
   * @param statusLine TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   */
  public HttpStatus(String statusLine)
    throws WcmException {
    parse(statusLine);
  }

  /**
   * @return the protocol string
   */
  public String getProtocol() {
    return m_protocol;
  }

  /**
   * @return the status code
   */
  public int getCode() {
    return m_status;
  }

  /**
   * @return the status description or <code>null</code> if missing
   */
  public String getDescription() {
    return m_description;
  }


  /**
   * Reinitialize the WDStatus from the given line
   *
   * @param line TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   */
  public void parse(String line)
    throws WcmException {
    String[] result = new String[2];
    m_status = parseInternal(line, result);
    m_protocol = result[0];
    m_description = result[1];
  }

  /**
   * Reinitialize the WDStatus from the given line
   *
   * @param line TBD: Description of the incoming method parameter
   * @param result TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception WcmException Exception raised in failure situation
   */
  private static int parseInternal(String line, String[] result)
    throws WcmException {
    String protocol = null;
    int status = 0;
    String description = null;
    boolean skipping = true;
    int len = line.length();
    StringBuffer sb = new StringBuffer(len);

    for (int i = 0; i < len; ++i) {
      char c = line.charAt(i);
      if (" \t\n".indexOf(c) >= 0) {
        if (skipping) {
          continue;
        }

        if (protocol == null) {
          protocol = sb.toString();
          sb.setLength(0);
          skipping = true;
        }
        else if (status == 0) {
          status = getCodeInternal(sb.toString());
          if (result == null) {
            return status;
          }
          sb.setLength(0);
          skipping = true;
        }
        else {
          sb.append(c);
        }
      }
      else {
        sb.append(c);
        skipping = false;
      }
    }

    if (sb.length() > 0) {
      if (status == 0) {
        status = getCodeInternal(sb.toString());
        if (result == null) {
          return status;
        }
      }
      else {
        description = sb.toString();
      }
    }

    if (protocol == null || status == 0) {
      throw new WcmException("cannot parse status (" + line + ")", false);
    }

    if (result != null) {
      result[0] = protocol;
      result[1] = description;
    }
    return status;
  }

  private static int getCodeInternal(String line)
    throws WcmException {
    try {
      return Integer.parseInt(line);
    }
    catch (NumberFormatException ex) {
      throw new WcmException("cannot parse status (" + line + ")", ex, false);
    }
  }
}

