/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.http;

import com.sapportals.wcm.util.uri.*;

/**
 * Implements IRequest interface for HTTP requests. <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Id: HttpRequest.java,v 1.1 2002/08/21 17:00:02 sei Exp $
 */
public class HttpRequest implements IRequest {

  private final static String HEADER_CONTENT_TYPE = "Content-Type";
  private final static String HEADER_CONTENT_LENGTH = "Content-Length";

  private IUriReference ref;
  private String uri;
  private String method;
  private String body;
  private java.io.InputStream stream;
  private final Headers headers = new Headers();
  private boolean responseAsStream;
  private boolean responseAsDocument;

  public HttpRequest() { }

  public HttpRequest(String uri) {
    this.uri = uri;
  }

  public HttpRequest(IUriReference ref) {
    this.ref = ref;
    this.uri = ref.toString();
  }

  public void clear() {
    this.uri = null;
    this.method = null;
    this.body = null;
    this.stream = null;
    this.headers.clear();
    this.responseAsStream = false;
    this.responseAsDocument = false;
  }

  public boolean expectsResponseDocument() {
    return this.responseAsDocument;
  }

  public void expectsResponseDocument(boolean itDoes) {
    this.responseAsDocument = itDoes;
  }

  public boolean expectsResponseStream() {
    return this.responseAsStream;
  }

  public void expectsResponseStream(boolean itDoes) {
    this.responseAsStream = itDoes;
  }

  public String getBody() {
    return this.body;
  }

  public java.io.InputStream getBodyStream() {
    return this.stream;
  }

  public String getContentType() {
    return (String)this.headers.get(HEADER_CONTENT_TYPE);
  }

  public String getHeader(String name) {
    return (String)this.headers.get(name);
  }

  public java.util.Iterator getHeaderNames() {
    return this.headers.keys();
  }

  public String getMethod() {
    return this.method;
  }

  public String getURI() {
    return this.uri;
  }

  public IUriReference getReference() {
    return this.ref;
  }

  public boolean hasBody() {
    return (this.body != null) || (this.stream != null);
  }

  public void setBody(String body) {
    this.body = body;
    this.stream = null;
  }

  public void setBody(java.io.InputStream stream) {
    this.body = null;
    this.stream = stream;
  }

  public void removeBody() {
    this.body = null;
    this.stream = null;
  }

  public void setContentType(String mimeType) {
    this.headers.set(HEADER_CONTENT_TYPE, mimeType);
  }

  public void setContentLength(long length) {
    this.headers.set(HEADER_CONTENT_LENGTH, String.valueOf(length));
  }

  public void addHeader(String name, String value) {
    this.headers.add(name, value);
  }

  public void setHeader(String name, String value) {
    this.headers.set(name, value);
  }

  public void removeHeader(String name) {
    this.headers.remove(name);
  }

  public void setMethod(String name) {
    this.method = name;
  }

  public void setURI(String uri) {
    this.uri = uri;
  }

  public void setURI(IUriReference ref) {
    this.ref = ref;
    this.uri = ref.toString();
  }

  public boolean wasAddedHeader(String name) {
    return this.headers.wasAdded(name);
  }
}

